import logging
import os
import pathlib as pl
import pickledb

from cliff.command import Command
from ..runner import Runner
from ..editors import Settings_Editor
from ..managers import Model_Statistics_Manager, Recommendations_Manager

from ..controllers.data_processing import *
from ..controllers.mongo_connection import *


class RunEngine(Command):
    """
    Run the CARSKit engine

    In the present time, this class is also responsible for
    getting the engine's latest data and saving the formatted statistics
    and recommendations.
    """

    # CRIAR UMA FLAG PARA MEDIR O TEMPO DE EXECUCAO DA ENGINE
    # CRIAR UMA FLAG PARA ESPECIFICAR O TIPO DE SAIDA DA ENGINE
    # CRIAR UMA FLAG PARA ESCOLHER SE UM NOVO setting.conf SERA CRIADO OU NÃO

    def get_parser(self, prog_name):
        """RunEngine argument parsing."""
        parser = super(RunEngine, self).get_parser(prog_name)
        # group = parser.add_mutually_exclusive_group()

        #     group.add_argument(
        #         "--lowercase",
        #         help="print result in lower case",
        #         action="store_true",
        #     )

        parser.add_argument(
            "--save-mongo",
            help="Especify if the data generated by the engine will be stored in a remote MongoDB server",
            action="store_true",
        )

        return parser

    def take_action(self, parsed_args):
        """RunEngine action"""

        # Getting  where the app is on computer
        app_path = get_app_path()
        engine_folder_path = os.path.join(app_path, "carskit/")
        conf_file_path = os.path.join(app_path, "carskit/setting.conf")

        # Instantiating  the Runner and the Settings_Editor classes
        settings_editor = Settings_Editor(conf_file_path)
        runner = Runner(engine_folder_path)

        # Getting the current date for file creation
        stats_filename, recs_filename = create_file_names()

        # Instantiating the Model_Statistics_Manager and Recommendations_Manager classes
        stats_mgn = Model_Statistics_Manager(
            settings_editor.get_results_foldername(),
            stats_filename,
        )
        recs_mgn = Recommendations_Manager(
            settings_editor.get_results_foldername(), recs_filename
        )

        # Generating the setting.conf file with the latest configurations
        status = settings_editor.generate_file()

        if status == "The settings file was generated!":

            # Running engine
            runner.run_engine()

            # Getting path of the latest files the engine has created
            execution_datafiles = latest_execution_data(
                app_path,
                settings_editor.get_dataset_path(),
                settings_editor.get_results_foldername(),
            )

            # Getting the path where the files will be saved
            target_path = pl.Path(execution_datafiles[0]).parent.parent
            # Extrating the data of the statistics file
            execution_stats = extract_stat_data(stats_mgn, execution_datafiles[0])
            # Extrating the data of the recommendtations file
            recommendations = extract_recommendations(recs_mgn, execution_datafiles[1])

            if parsed_args.save_mongo:
                status = upload_output(
                    (
                        {stats_filename: execution_stats},
                        {recs_filename: recommendations},
                    )
                )
                if not status:
                    return "Error. Connection with server failed"

                return f"Documents {stats_filename} and {recs_filename} saved on mongo."

            # Saving the statistics and recommendations files in the target path
            stats_mgn.save_statistics(
                execution_stats, os.path.join(target_path, stats_filename + ".json")
            )
            recs_mgn.save_recommendations(
                recommendations, os.path.join(target_path, recs_filename + ".json")
            )

            return
        else:
            output = "An error has occured, execution canceled."

        return
