/*
 * Decompiled with CFR 0.152.
 */
package librec.data;

import java.io.Serializable;
import java.util.Arrays;
import librec.data.DenseVector;
import librec.data.MatrixEntry;
import librec.data.SVD;
import librec.data.SparseMatrix;
import librec.data.SparseVector;
import librec.data.VectorEntry;
import librec.util.Randoms;
import librec.util.Strings;

public class DenseMatrix
implements Serializable {
    private static final long serialVersionUID = -2069621030647530185L;
    protected int numRows;
    protected int numColumns;
    protected double[][] data;

    public DenseMatrix(int numRows, int numColumns) {
        this.numRows = numRows;
        this.numColumns = numColumns;
        this.data = new double[numRows][numColumns];
    }

    public DenseMatrix(double[][] array) {
        this(array.length, array[0].length);
        int i = 0;
        while (i < this.numRows) {
            int j = 0;
            while (j < this.numColumns) {
                this.data[i][j] = array[i][j];
                ++j;
            }
            ++i;
        }
    }

    public DenseMatrix(double[][] array, int numRows, int numColumns) {
        this.numRows = numRows;
        this.numColumns = numColumns;
        this.data = array;
    }

    public DenseMatrix(DenseMatrix mat) {
        this(mat.data);
    }

    public DenseMatrix clone() {
        return new DenseMatrix(this);
    }

    public static DenseMatrix eye(int dim) {
        DenseMatrix mat = new DenseMatrix(dim, dim);
        int i = 0;
        while (i < mat.numRows) {
            mat.set(i, i, 1.0);
            ++i;
        }
        return mat;
    }

    public void init(double mean, double sigma) {
        int i = 0;
        while (i < this.numRows) {
            int j = 0;
            while (j < this.numColumns) {
                this.data[i][j] = Randoms.gaussian(mean, sigma);
                ++j;
            }
            ++i;
        }
    }

    public void init(double range) {
        int i = 0;
        while (i < this.numRows) {
            int j = 0;
            while (j < this.numColumns) {
                this.data[i][j] = Randoms.uniform(0.0, range);
                ++j;
            }
            ++i;
        }
    }

    public void init() {
        this.init(1.0);
    }

    public int numRows() {
        return this.numRows;
    }

    public int numColumns() {
        return this.numColumns;
    }

    public DenseVector row(int rowId) {
        return this.row(rowId, true);
    }

    public DenseVector row(int rowId, boolean deep) {
        return new DenseVector(this.data[rowId], deep);
    }

    public DenseVector column(int column) {
        DenseVector vec = new DenseVector(this.numRows);
        int i = 0;
        while (i < this.numRows) {
            vec.set(i, this.data[i][column]);
            ++i;
        }
        return vec;
    }

    public double columnMean(int column) {
        double sum = 0.0;
        int i = 0;
        while (i < this.numRows) {
            sum += this.data[i][column];
            ++i;
        }
        return sum / (double)this.numRows;
    }

    public double norm() {
        double res = 0.0;
        int i = 0;
        while (i < this.numRows) {
            int j = 0;
            while (j < this.numColumns) {
                res += this.data[i][j] * this.data[i][j];
                ++j;
            }
            ++i;
        }
        return Math.sqrt(res);
    }

    public static double rowMult(DenseMatrix m, int mrow, DenseMatrix n, int nrow) {
        assert (m.numColumns == n.numColumns);
        double res = 0.0;
        int j = 0;
        int k = m.numColumns;
        while (j < k) {
            res += m.get(mrow, j) * n.get(nrow, j);
            ++j;
        }
        return res;
    }

    public static double colMult(DenseMatrix m, int mcol, DenseMatrix n, int ncol) {
        assert (m.numRows == n.numRows);
        double res = 0.0;
        int j = 0;
        int k = m.numRows;
        while (j < k) {
            res += m.get(j, mcol) * n.get(j, ncol);
            ++j;
        }
        return res;
    }

    public static double product(DenseMatrix m, int mrow, DenseMatrix n, int ncol) {
        assert (m.numColumns == n.numRows);
        double res = 0.0;
        int j = 0;
        while (j < m.numColumns) {
            res += m.get(mrow, j) * n.get(j, ncol);
            ++j;
        }
        return res;
    }

    public static DenseMatrix kroneckerProduct(DenseMatrix M, DenseMatrix N) {
        DenseMatrix res = new DenseMatrix(M.numRows * N.numRows, M.numColumns * N.numColumns);
        int i = 0;
        while (i < M.numRows) {
            int j = 0;
            while (j < M.numColumns) {
                double Mij = M.get(i, j);
                int ni = 0;
                while (ni < N.numRows) {
                    int nj = 0;
                    while (nj < N.numColumns) {
                        int row = i * N.numRows + ni;
                        int col = j * N.numColumns + nj;
                        res.set(row, col, Mij * N.get(ni, nj));
                        ++nj;
                    }
                    ++ni;
                }
                ++j;
            }
            ++i;
        }
        return res;
    }

    public static DenseMatrix khatriRaoProduct(DenseMatrix M, DenseMatrix N) throws Exception {
        if (M.numColumns != N.numColumns) {
            throw new Exception("The number of columns of two matrices is not equal!");
        }
        DenseMatrix res = new DenseMatrix(M.numRows * N.numRows, M.numColumns);
        int j = 0;
        while (j < M.numColumns) {
            int i = 0;
            while (i < M.numRows) {
                double Mij = M.get(i, j);
                int ni = 0;
                while (ni < N.numRows) {
                    int row = ni + i * N.numRows;
                    res.set(row, j, Mij * N.get(ni, j));
                    ++ni;
                }
                ++i;
            }
            ++j;
        }
        return res;
    }

    public static DenseMatrix hadamardProduct(DenseMatrix M, DenseMatrix N) throws Exception {
        if (M.numRows != N.numRows || M.numColumns != N.numColumns) {
            throw new Exception("The dimensions of two matrices are not consistent!");
        }
        DenseMatrix res = new DenseMatrix(M.numRows, M.numColumns);
        int i = 0;
        while (i < M.numRows) {
            int j = 0;
            while (j < M.numColumns) {
                res.set(i, j, M.get(i, j) * N.get(i, j));
                ++j;
            }
            ++i;
        }
        return res;
    }

    public DenseMatrix transMult() {
        DenseMatrix res = new DenseMatrix(this.numColumns, this.numColumns);
        int i = 0;
        while (i < this.numColumns) {
            int k = 0;
            while (k < this.numColumns) {
                double val = 0.0;
                int j = 0;
                while (j < this.numRows) {
                    val += this.get(j, i) * this.get(j, k);
                    ++j;
                }
                res.set(i, k, val);
                ++k;
            }
            ++i;
        }
        return res;
    }

    public DenseMatrix mult(DenseMatrix mat) {
        assert (this.numColumns == mat.numRows);
        DenseMatrix res = new DenseMatrix(this.numRows, mat.numColumns);
        int i = 0;
        while (i < res.numRows) {
            int j = 0;
            while (j < res.numColumns) {
                double product = 0.0;
                int k = 0;
                while (k < this.numColumns) {
                    product += this.data[i][k] * mat.data[k][j];
                    ++k;
                }
                res.set(i, j, product);
                ++j;
            }
            ++i;
        }
        return res;
    }

    public DenseMatrix mult(SparseMatrix mat) {
        assert (this.numColumns == mat.numRows);
        DenseMatrix res = new DenseMatrix(this.numRows, mat.numColumns);
        int j = 0;
        while (j < res.numColumns) {
            SparseVector col = mat.column(j);
            int i = 0;
            while (i < res.numRows) {
                double product = 0.0;
                for (VectorEntry ve : col) {
                    product += this.data[i][ve.index()] * ve.get();
                }
                res.set(i, j, product);
                ++i;
            }
            ++j;
        }
        return res;
    }

    public DenseVector mult(DenseVector vec) {
        assert (this.numColumns == vec.size);
        DenseVector res = new DenseVector(this.numRows);
        int i = 0;
        while (i < this.numRows) {
            res.set(i, this.row(i, false).inner(vec));
            ++i;
        }
        return res;
    }

    public DenseVector mult(SparseVector vec) {
        DenseVector res = new DenseVector(this.numRows);
        int i = 0;
        while (i < this.numRows) {
            double product = 0.0;
            for (VectorEntry ve : vec) {
                product += this.data[i][ve.index()] * ve.get();
            }
            res.set(i, product);
            ++i;
        }
        return res;
    }

    public static DenseMatrix mult(SparseMatrix sm, DenseMatrix dm) {
        assert (sm.numColumns == dm.numRows);
        DenseMatrix res = new DenseMatrix(sm.numRows, dm.numColumns);
        int i = 0;
        while (i < res.numRows) {
            SparseVector row = sm.row(i);
            int j = 0;
            while (j < res.numColumns) {
                double product = 0.0;
                int[] nArray = row.getIndex();
                int n = nArray.length;
                int n2 = 0;
                while (n2 < n) {
                    int k = nArray[n2];
                    product += row.get(k) * dm.data[k][j];
                    ++n2;
                }
                res.set(i, j, product);
                ++j;
            }
            ++i;
        }
        return res;
    }

    public double get(int row, int column) {
        return this.data[row][column];
    }

    public void set(int row, int column, double val) {
        this.data[row][column] = val;
    }

    public void setAll(double val) {
        int row = 0;
        while (row < this.numRows) {
            int col = 0;
            while (col < this.numColumns) {
                this.data[row][col] = val;
                ++col;
            }
            ++row;
        }
    }

    public double sumOfRow(int row) {
        double res = 0.0;
        int col = 0;
        while (col < this.numColumns) {
            res += this.data[row][col];
            ++col;
        }
        return res;
    }

    public double sumOfColumn(int col) {
        double res = 0.0;
        int row = 0;
        while (row < this.numRows) {
            res += this.data[row][col];
            ++row;
        }
        return res;
    }

    public double sum() {
        double res = 0.0;
        int row = 0;
        while (row < this.numRows) {
            int col = 0;
            while (col < this.numColumns) {
                res += this.data[row][col];
                ++col;
            }
            ++row;
        }
        return res;
    }

    public void add(int row, int column, double val) {
        double[] dArray = this.data[row];
        int n = column;
        dArray[n] = dArray[n] + val;
    }

    public DenseMatrix scale(double val) {
        DenseMatrix mat = new DenseMatrix(this.numRows, this.numColumns);
        int i = 0;
        while (i < this.numRows) {
            int j = 0;
            while (j < this.numColumns) {
                mat.data[i][j] = this.data[i][j] * val;
                ++j;
            }
            ++i;
        }
        return mat;
    }

    public DenseMatrix add(DenseMatrix mat) {
        assert (this.numRows == mat.numRows);
        assert (this.numColumns == mat.numColumns);
        DenseMatrix res = new DenseMatrix(this.numRows, this.numColumns);
        int i = 0;
        while (i < this.numRows) {
            int j = 0;
            while (j < this.numColumns) {
                res.data[i][j] = this.data[i][j] + mat.data[i][j];
                ++j;
            }
            ++i;
        }
        return res;
    }

    public DenseMatrix add(SparseMatrix mat) {
        assert (this.numRows == mat.numRows);
        assert (this.numColumns == mat.numColumns);
        DenseMatrix res = this.clone();
        for (MatrixEntry me : mat) {
            res.add(me.row(), me.column(), me.get());
        }
        return res;
    }

    public DenseMatrix add(double val) {
        DenseMatrix res = new DenseMatrix(this.numRows, this.numColumns);
        int i = 0;
        while (i < this.numRows) {
            int j = 0;
            while (j < this.numColumns) {
                res.data[i][j] = this.data[i][j] + val;
                ++j;
            }
            ++i;
        }
        return res;
    }

    public DenseMatrix minus(DenseMatrix mat) {
        assert (this.numRows == mat.numRows);
        assert (this.numColumns == mat.numColumns);
        DenseMatrix res = new DenseMatrix(this.numRows, this.numColumns);
        int i = 0;
        while (i < this.numRows) {
            int j = 0;
            while (j < this.numColumns) {
                res.data[i][j] = this.data[i][j] - mat.data[i][j];
                ++j;
            }
            ++i;
        }
        return res;
    }

    public DenseMatrix minus(SparseMatrix mat) {
        assert (this.numRows == mat.numRows);
        assert (this.numColumns == mat.numColumns);
        DenseMatrix res = this.clone();
        for (MatrixEntry me : mat) {
            res.add(me.row(), me.column(), -me.get());
        }
        return res;
    }

    public DenseMatrix minus(double val) {
        DenseMatrix res = new DenseMatrix(this.numRows, this.numColumns);
        int i = 0;
        while (i < this.numRows) {
            int j = 0;
            while (j < this.numColumns) {
                res.data[i][j] = this.data[i][j] - val;
                ++j;
            }
            ++i;
        }
        return res;
    }

    public DenseMatrix cholesky() {
        if (this.numRows != this.numColumns) {
            throw new RuntimeException("Matrix is not square");
        }
        int n = this.numRows;
        DenseMatrix L = new DenseMatrix(n, n);
        int i = 0;
        while (i < n) {
            int j = 0;
            while (j <= i) {
                double sum = 0.0;
                int k = 0;
                while (k < j) {
                    sum += L.get(i, k) * L.get(j, k);
                    ++k;
                }
                double val = i == j ? Math.sqrt(this.data[i][i] - sum) : (this.data[i][j] - sum) / L.get(j, j);
                L.set(i, j, val);
                ++j;
            }
            if (Double.isNaN(L.get(i, i))) {
                return null;
            }
            ++i;
        }
        return L.transpose();
    }

    public DenseMatrix transpose() {
        DenseMatrix mat = new DenseMatrix(this.numColumns, this.numRows);
        int i = 0;
        while (i < mat.numRows) {
            int j = 0;
            while (j < mat.numColumns) {
                mat.set(i, j, this.data[j][i]);
                ++j;
            }
            ++i;
        }
        return mat;
    }

    public DenseMatrix cov() {
        DenseMatrix mat = new DenseMatrix(this.numColumns, this.numColumns);
        int i = 0;
        while (i < this.numColumns) {
            DenseVector xi = this.column(i);
            xi = xi.minus(xi.mean());
            mat.set(i, i, xi.inner(xi) / (double)(xi.size - 1));
            int j = i + 1;
            while (j < this.numColumns) {
                DenseVector yi = this.column(j);
                double val = xi.inner(yi.minus(yi.mean())) / (double)(xi.size - 1);
                mat.set(i, j, val);
                mat.set(j, i, val);
                ++j;
            }
            ++i;
        }
        return mat;
    }

    public DenseMatrix inverse() {
        int j;
        int i;
        if (this.numRows != this.numColumns) {
            throw new RuntimeException("Only square matrix can do inversion");
        }
        int n = this.numRows;
        DenseMatrix mat = new DenseMatrix(this);
        if (n == 1) {
            mat.set(0, 0, 1.0 / mat.get(0, 0));
            return mat;
        }
        int[] row = new int[n];
        int[] col = new int[n];
        double[] temp = new double[n];
        int k = 0;
        while (k < n) {
            row[k] = k;
            col[k] = k;
            ++k;
        }
        k = 0;
        while (k < n) {
            int j2;
            double pivot = mat.get(row[k], col[k]);
            int I_pivot = k;
            int J_pivot = k;
            i = k;
            while (i < n) {
                j2 = k;
                while (j2 < n) {
                    double abs_pivot = Math.abs(pivot);
                    if (Math.abs(mat.get(row[i], col[j2])) > abs_pivot) {
                        I_pivot = i;
                        J_pivot = j2;
                        pivot = mat.get(row[i], col[j2]);
                    }
                    ++j2;
                }
                ++i;
            }
            if (Math.abs(pivot) < 1.0E-10) {
                throw new RuntimeException("Matrix is singular !");
            }
            int hold = row[k];
            row[k] = row[I_pivot];
            row[I_pivot] = hold;
            hold = col[k];
            col[k] = col[J_pivot];
            col[J_pivot] = hold;
            mat.set(row[k], col[k], 1.0 / pivot);
            j = 0;
            while (j < n) {
                if (j != k) {
                    mat.set(row[k], col[j], mat.get(row[k], col[j]) * mat.get(row[k], col[k]));
                }
                ++j;
            }
            i = 0;
            while (i < n) {
                if (k != i) {
                    j2 = 0;
                    while (j2 < n) {
                        if (k != j2) {
                            double val = mat.get(row[i], col[j2]) - mat.get(row[i], col[k]) * mat.get(row[k], col[j2]);
                            mat.set(row[i], col[j2], val);
                        }
                        ++j2;
                    }
                    mat.set(row[i], col[k], -mat.get(row[i], col[k]) * mat.get(row[k], col[k]));
                }
                ++i;
            }
            ++k;
        }
        int j3 = 0;
        while (j3 < n) {
            i = 0;
            while (i < n) {
                temp[col[i]] = mat.get(row[i], j3);
                ++i;
            }
            i = 0;
            while (i < n) {
                mat.set(i, j3, temp[i]);
                ++i;
            }
            ++j3;
        }
        int i2 = 0;
        while (i2 < n) {
            j = 0;
            while (j < n) {
                temp[row[j]] = mat.get(i2, col[j]);
                ++j;
            }
            j = 0;
            while (j < n) {
                mat.set(i2, j, temp[j]);
                ++j;
            }
            ++i2;
        }
        return mat;
    }

    public DenseMatrix inv() {
        if (this.numRows != this.numColumns) {
            throw new RuntimeException("Dimensions disagree");
        }
        int n = this.numRows;
        DenseMatrix mat = DenseMatrix.eye(n);
        if (n == 1) {
            mat.set(0, 0, 1.0 / this.get(0, 0));
            return mat;
        }
        DenseMatrix b = new DenseMatrix(this);
        int i = 0;
        while (i < n) {
            double mag2;
            double mag = 0.0;
            int pivot = -1;
            int j = i;
            while (j < n) {
                mag2 = Math.abs(b.get(j, i));
                if (mag2 > mag) {
                    mag = mag2;
                    pivot = j;
                }
                ++j;
            }
            if (pivot == -1 || mag == 0.0) {
                return mat;
            }
            if (pivot != i) {
                int j2 = i;
                while (j2 < n) {
                    double temp = b.get(i, j2);
                    b.set(i, j2, b.get(pivot, j2));
                    b.set(pivot, j2, temp);
                    ++j2;
                }
                j2 = 0;
                while (j2 < n) {
                    double temp = mat.get(i, j2);
                    mat.set(i, j2, mat.get(pivot, j2));
                    mat.set(pivot, j2, temp);
                    ++j2;
                }
            }
            mag = b.get(i, i);
            j = i;
            while (j < n) {
                b.set(i, j, b.get(i, j) / mag);
                ++j;
            }
            j = 0;
            while (j < n) {
                mat.set(i, j, mat.get(i, j) / mag);
                ++j;
            }
            int k = 0;
            while (k < n) {
                if (k != i) {
                    mag2 = b.get(k, i);
                    int j3 = i;
                    while (j3 < n) {
                        b.set(k, j3, b.get(k, j3) - mag2 * b.get(i, j3));
                        ++j3;
                    }
                    j3 = 0;
                    while (j3 < n) {
                        mat.set(k, j3, mat.get(k, j3) - mag2 * mat.get(i, j3));
                        ++j3;
                    }
                }
                ++k;
            }
            ++i;
        }
        return mat;
    }

    public DenseMatrix pinv() {
        if (this.numRows < this.numColumns) {
            DenseMatrix res = this.transpose().pinv();
            if (res != null) {
                res = res.transpose();
            }
            return res;
        }
        SVD svd = this.svd();
        DenseMatrix U = svd.getU();
        DenseMatrix S = svd.getS();
        DenseMatrix V = svd.getV();
        DenseMatrix SPlus = S.clone();
        int i = 0;
        while (i < SPlus.numRows) {
            double val = SPlus.get(i, i);
            if (val != 0.0) {
                SPlus.set(i, i, 1.0 / val);
            }
            ++i;
        }
        return V.mult(SPlus).mult(U.transpose());
    }

    public SVD svd() {
        return new SVD(this);
    }

    public void setRow(int row, double val) {
        Arrays.fill(this.data[row], val);
    }

    public void setRow(int row, DenseVector vals) {
        int j = 0;
        while (j < this.numColumns) {
            this.data[row][j] = vals.data[j];
            ++j;
        }
    }

    public void clear() {
        int i = 0;
        while (i < this.numRows) {
            this.setRow(i, 0.0);
            ++i;
        }
    }

    public String toString() {
        return Strings.toString(this.data);
    }
}

