import math
import numpy as np
import pandas as pd
import talib as ta
from sklearn.preprocessing import MinMaxScaler


# ==========================
# ==== Helper Functions ====
# ==========================


def normalize(src: np.array, range_min=0, range_max=1) -> np.array:
    """
    function Rescales a source value with an unbounded range to a bounded range
    param src: <np.array> The input series
    param range_min: <float> The minimum value of the unbounded range
    param range_max: <float> The maximum value of the unbounded range
    returns <np.array> The normalized series
    """
    scaler = MinMaxScaler(feature_range=(0, 1))
    return range_min + (range_max - range_min) * scaler.fit_transform(src.reshape(-1,1))[:,0]


def rescale(src: np.array, old_min, old_max, new_min=0, new_max=1) -> np.array:
    """
    function Rescales a source value with a bounded range to anther bounded range
    param src: <np.array> The input series
    param old_min: <float> The minimum value of the range to rescale from
    param old_max: <float> The maximum value of the range to rescale from
    param new_min: <float> The minimum value of the range to rescale to
    param new_max: <float> The maximum value of the range to rescale to 
    returns <np.array> The rescaled series
    """
    rescaled_value = new_min + (new_max - new_min) * (src - old_min) / max(old_max - old_min, 10e-10)
    return rescaled_value


def n_rsi(src: pd.Series, n1, n2) -> np.array:
    """
    function Returns the normalized RSI ideal for use in ML algorithms
    param src: <np.array> The input series
    param n1: <int> The length of the RSI
    param n2: <int> The smoothing length of the RSI
    returns <np.array> The normalized RSI
    """
    return rescale(ta.EMA(ta.RSI(src.values, n1), n2), 0, 100)


def n_cci(highSrc: pd.Series, lowSrc: pd.Series, closeSrc: pd.Series, n1, n2) -> np.array:
    """
    function Returns the normalized CCI ideal for use in ML algorithms
    param highSrc: <np.array> The input series for the high price
    param lowSrc: <np.array> The input series for the low price
    param closeSrc: <np.array> The input series for the close price
    param n1: <int> The length of the CCI
    param n2: <int> The smoothing length of the CCI
    returns <np.array> The normalized CCI
    """
    return normalize(ta.EMA(ta.CCI(highSrc.values, lowSrc.values, closeSrc.values, n1), n2))

def n_wt(src: pd.Series, n1=10, n2=11) -> np.array:
    """
    function Returns the normalized WaveTrend Classic series ideal for use in ML algorithms
    param src: <np.array> The input series
    param n1: <int> The first smoothing length for WaveTrend Classic
    param n2: <int> The second smoothing length for the WaveTrend Classic
    returns <np.array> The normalized WaveTrend Classic series
    """
    ema1 = ta.EMA(src.values, n1)
    ema2 = ta.EMA(abs(src.values - ema1), n1)
    ci = (src.values - ema1) / (0.015 * ema2)
    wt1 = ta.EMA(ci, n2)  # tci
    wt2 = ta.SMA(wt1, 4)
    return normalize(wt1 - wt2)

def n_adx(highSrc: pd.Series, lowSrc: pd.Series, closeSrc: pd.Series, n1) -> np.array:
    """
    function Returns the normalized ADX ideal for use in ML algorithms
    param highSrc: <np.array> The input series for the high price
    param lowSrc: <np.array> The input series for the low price
    param closeSrc: <np.array> The input series for the close price
    param n1: <int> The length of the ADX
    """
    return rescale(ta.ADX(highSrc.values, lowSrc.values, closeSrc.values, n1), 0, 100)
    # TODO: Replicate ADX logic from jdehorty


# =================
# ==== Filters ====
# =================
def regime_filter(src: pd.Series, high: pd.Series, low: pd.Series, useRegimeFilter, threshold) -> np.array:
    """
    regime_filter
    param src: <np.array> The source series
    param high: <np.array> The input series for the high price
    param low: <np.array> The input series for the low price
    param useRegimeFilter: <bool> Whether to use the regime filter
    param threshold: <float> The threshold
    returns <np.array> Boolean indicating whether or not to let the signal pass through the filter
    """
    if not useRegimeFilter: return np.array([True]*len(src))

    # @njit(parallel=True, cache=True)
    def klmf(src: np.array, high: np.array, low: np.array):
        value1 = np.array([0.0]*len(src))
        value2 = np.array([0.0]*len(src))
        klmf = np.array([0.0]*len(src))

        for i in range(len(src)):
            if (high[i] - low[i]) == 0: continue
            value1[i] = 0.2 * (src[i] - src[i - 1 if i >= 1 else 0]) + 0.8 * value1[i - 1 if i >= 1 else 0]
            value2[i] = 0.1 * (high[i] - low[i]) + 0.8 * value2[i - 1 if i >= 1 else 0]

        with np.errstate(divide='ignore',invalid='ignore'):
            omega = np.nan_to_num(np.abs(np.divide(value1, value2)))
        alpha = (-(omega ** 2) + np.sqrt((omega ** 4) + 16 * (omega ** 2))) / 8

        for i in range(len(src)):
            klmf[i] = alpha[i] * src[i] + (1 - alpha[i]) * klmf[i - 1 if i >= 1 else 0]

        return klmf

    filter = np.array([False]*len(src))
    absCurveSlope = np.abs(np.diff(klmf(src.values, high.values, low.values), prepend=0.0))
    exponentialAverageAbsCurveSlope = ta.EMA(absCurveSlope, 200)
    with np.errstate(divide='ignore',invalid='ignore'):
        normalized_slope_decline = (absCurveSlope - exponentialAverageAbsCurveSlope) / exponentialAverageAbsCurveSlope
    flags = (normalized_slope_decline >= threshold)
    filter[(len(filter) - len(flags)):] = flags
    return filter

def filter_adx(src: pd.Series, high: pd.Series, low: pd.Series, adxThreshold, useAdxFilter, length=14) -> np.array:
    """
    function filter_adx
    param src: <np.array> The source series
    param high: <np.array> The input series for the high price
    param low: <np.array> The input series for the low price
    param adxThreshold: <int> The ADX threshold
    param useAdxFilter: <bool> Whether to use the ADX filter
    param length: <int> The length of the ADX
    returns <np.array> Boolean indicating whether or not to let the signal pass through the filter
    """
    if not useAdxFilter: return np.array([True]*len(src))
    adx = ta.ADX(high.values, low.values, src.values, length)
    return (adx > adxThreshold)

def filter_volatility(high, low, close, useVolatilityFilter, minLength=1, maxLength=10) -> np.array:
    """
    function filter_volatility
    param high: <np.array> The input series for the high price
    param low: <np.array> The input series for the low price
    param close: <np.array> The input series for the close price
    param useVolatilityFilter: <bool> Whether to use the volatility filter
    param minLength: <int> The minimum length of the ATR
    param maxLength: <int> The maximum length of the ATR
    returns <np.array> Boolean indicating whether or not to let the signal pass through the filter
    """
    if not useVolatilityFilter: return np.array([True]*len(close))
    recentAtr = ta.ATR(high.values, low.values, close.values, minLength)
    historicalAtr = ta.ATR(high.values, low.values, close.values, maxLength)
    return (recentAtr > historicalAtr)
