# -*- coding: utf-8 -*-

# Copyright (c) 2014 - 2021 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a previewer widget for Qt style sheet files.
"""

import os

from PyQt6.QtWidgets import (
    QWidget, QMenu, QLabel, QHeaderView, QListWidgetItem
)

from .Ui_PreviewerQSS import Ui_PreviewerQSS

import Preferences
import UI.PixmapCache


class PreviewerQSS(QWidget, Ui_PreviewerQSS):
    """
    Class implementing a previewer widget for Qt style sheet files.
    """
    def __init__(self, parent=None):
        """
        Constructor
        
        @param parent reference to the parent widget (QWidget)
        """
        super().__init__(parent)
        self.setupUi(self)
        
        # menu for the tool button
        self.__toolButtonMenu = QMenu(self)
        self.__toolButtonMenu.addAction(self.tr("Action 1"))
        self.__toolButtonMenu.addSeparator()
        self.__toolButtonMenu.addAction(self.tr("Action 2"))
        self.toolButton.setMenu(self.__toolButtonMenu)
        
        # a MDI window
        self.__mdi = self.mdiArea.addSubWindow(QLabel(self.tr("MDI")))
        self.__mdi.resize(160, 80)
        
        # tree and table widgets
        self.tree.header().setSectionResizeMode(
            QHeaderView.ResizeMode.ResizeToContents)
        self.table.horizontalHeader().setSectionResizeMode(
            QHeaderView.ResizeMode.ResizeToContents)
        self.tree.topLevelItem(0).setExpanded(True)
        
        # icon list widget
        for iconName, labelText in (
            ("filePython", self.tr("Python")),
            ("fileRuby", self.tr("Ruby")),
            ("fileJavascript", self.tr("JavaScript")),
        ):
            self.iconsListWidget.addItem(QListWidgetItem(
                UI.PixmapCache.getIcon(iconName), labelText))
    
    def processEditor(self, editor=None):
        """
        Public slot to process an editor's text.
        
        @param editor editor to be processed (Editor)
        """
        if editor is not None:
            fn = editor.getFileName()
            
            if fn:
                extension = os.path.normcase(os.path.splitext(fn)[1][1:])
            else:
                extension = ""
            if (
                extension in Preferences.getEditor(
                    "PreviewQssFileNameExtensions")
            ):
                styleSheet = editor.text()
                if styleSheet:
                    self.scrollAreaWidgetContents.setStyleSheet(styleSheet)
                else:
                    self.scrollAreaWidgetContents.setStyleSheet("")
                self.toolButton.menu().setStyleSheet(
                    self.scrollAreaWidgetContents.styleSheet())
