# This file is part of the Reproducible and Reusable Data Analysis Workflow
# Server (flowServ).
#
# Copyright (C) 2019-2021 NYU.
#
# flowServ is free software; you can redistribute it and/or modify it under the
# terms of the MIT License; see LICENSE file for more details.

"""Interface for the API service component provides methods that execute,
access, and manipulate workflow runs and their results.
"""

from abc import ABCMeta, abstractmethod
from typing import Dict, IO, List, Optional


class RunService(metaclass=ABCMeta):  # pragma: no cover
    """API component that provides methods to start, access, and manipulate
    workflow runs and their resources.
    """
    @abstractmethod
    def cancel_run(self, run_id: str, reason: Optional[str] = None) -> Dict:
        """Cancel the run with the given identifier. Returns a serialization of
        the handle for the canceled run.

        Raises an unauthorized access error if the user does not have the
        necessary access rights to cancel the run.

        Parameters
        ----------
        run_id: string
            Unique run identifier
        reason: string, optional
            Optional text describing the reason for cancelling the run

        Returns
        -------
        dict
        """
        raise NotImplementedError()

    @abstractmethod
    def delete_run(self, run_id: str) -> Dict:
        """Delete the run with the given identifier.

        Raises an unauthorized access error if the user does not have the
        necessary access rights to delete the run.

        Parameters
        ----------
        run_id: string
            Unique run identifier
        flowserv.error.InvalidRunStateError
        """
        raise NotImplementedError()

    @abstractmethod
    def get_result_archive(self, run_id: str) -> IO:
        """Get compressed tar-archive containing all result files that were
        generated by a given workflow run. If the run is not in sucess state
        a unknown resource error is raised.

        Raises an unauthorized access error if the user does not have read
        access to the run.

        Parameters
        ----------
        run_id: string
            Unique run identifier

        Returns
        -------
        io.BytesIO
        """
        raise NotImplementedError()

    @abstractmethod
    def get_result_file(self, run_id: str, file_id: str) -> IO:
        """Get file handle for a resource file that was generated as the result
        of a successful workflow run.

        Raises an unauthorized access error if the user does not have read
        access to the run.

        Parameters
        ----------
        run_id: string
            Unique run identifier.
        file_id: string
            Unique result file identifier.

        Returns
        -------
        flowserv.model.files.base.FileHandle
        """
        raise NotImplementedError()

    @abstractmethod
    def get_run(self, run_id: str) -> Dict:
        """Get handle for the given run.

        Raises an unauthorized access error if the user does not have read
        access to the run.

        Parameters
        ----------
        run_id: string
            Unique run identifier

        Returns
        -------
        dict
        """
        raise NotImplementedError()

    @abstractmethod
    def list_runs(self, group_id: str, state: Optional[str] = None):
        """Get a listing of all run handles for the given workflow group.

        Raises an unauthorized access error if the user does not have read
        access to the workflow group.

        Parameters
        ----------
        group_id: string
            Unique workflow group identifier
        state: string, default=None
            State identifier query

        Returns
        -------
        dict
        """
        raise NotImplementedError()

    @abstractmethod
    def start_run(self, group_id: str, arguments: List[Dict]) -> Dict:
        """Start a new workflow run for the given group. The user provided
        arguments are expected to be a list of (key,value)-pairs. The key value
        identifies the template parameter. The data type of the value depends
        on the type of the parameter.

        Returns a serialization of the handle for the started run.

        Raises an unauthorized access error if the user does not have the
        necessary access to modify the workflow group.

        Parameters
        ----------
        group_id: string
            Unique workflow group identifier
        arguments: list(dict)
            List of user provided arguments for template parameters.

        Returns
        -------
        dict
        """
        raise NotImplementedError()
