# This file is part of the Reproducible and Reusable Data Analysis Workflow
# Server (flowServ).
#
# Copyright (C) 2019-2021 NYU.
#
# flowServ is free software; you can redistribute it and/or modify it under the
# terms of the MIT License; see LICENSE file for more details.

"""Monitor for remote workflow executions. The monitor is a separate thread
that continously polls the remote workflow engine to update the workflow state
in the local database.
"""

import logging
import os
import tempfile
import time

from threading import Thread

from flowserv.model.workflow.state import StateSuccess

import flowserv.util as util


class WorkflowMonitor(Thread):
    """Thread that monitors execution of an external workflow. Polls the state
    of the workflow in regular intervals. Updates the local workflow state as
    the remote state changes.
    """
    def __init__(
        self, run_id, workflow_id, state, output_files, client, poll_interval,
        service, tasks
    ):
        """Initialize the workflow information and the connection to the local
        service API.

        Parameters
        ----------
        run_id: string
            Unique run identifier.
        workflow_id: string
            Unique identifier for the workflow on the remote engine.
        state: flowserv.model.workflow.state.WorkflowState
            Current workflow state (to access the timestamps).
        output_files: list(string)
            Relative path of output files that are generated by the workflow
            run.
        client: flowserv.controller.remote.client.RemoteClient
            Implementation of the remote client that is used to interact with
            the workflow engine.
        poll_interval: float
            Frequency (in sec.) at which the remote workflow engine is polled.
        service: contextlib,contextmanager, default=None
            Context manager to create an instance of the service API.
        """
        Thread.__init__(self)
        self.run_id = run_id
        self.workflow_id = workflow_id
        self.state = state
        self.output_files = output_files
        self.client = client
        self.poll_interval = poll_interval
        self.service = service
        self.tasks = tasks

    def run(self):
        """Poll the remote server continuously until execution is finished."""
        monitor_workflow(
            run_id=self.run_id,
            state=self.state,
            workflow_id=self.workflow_id,
            output_files=self.output_files,
            client=self.client,
            poll_interval=self.poll_interval,
            service=self.service
        )
        # Remove the workflow information form the task list.
        try:
            del self.tasks[self.run_id]
        except Exception as ex:
            logging.error(ex)
            logging.debug('\n'.join(util.stacktrace(ex)))


# -- Helper functions ---------------------------------------------------------

def monitor_workflow(
    run_id, state, workflow_id, output_files, client, poll_interval,
    service=None
):
    """Monitor a remote workflow run by continuous polling at a given interval.
    Updates the local workflow state as the remote state changes.

    Returns the state of the inactive workflow and the temporary directory that
    contains the downloaded run result files. The run directory may be None for
    unsuccessful runs.

    Parameters
    ----------
    run_id: string
        Unique run identifier.
    workflow_id: string
        Unique identifier for the workflow on the remote engine.
    state: flowserv.model.workflow.state.WorkflowState
        Current workflow state (to access the timestamps).
    output_files: list(string)
        Relative path of output files that are generated by the workflow
        run.
    client: flowserv.controller.remote.client.RemoteClient
        Implementation of the remote client that is used to interact with
        the workflow engine.
    poll_interval: float
        Frequency (in sec.) at which the remote workflow engine is polled.
    service: contextlib,contextmanager, default=None
        Context manager to create an instance of the service API. If the value
        is None the monitor is running in synchronous mode. In this case the
        run state cannot be update by the monitor.

    Returns
    -------
    flowserv.model.workflow.state.WorkflowState, string
    """
    logging.info('start monitoring workflow {}'.format(workflow_id))
    rundir = None
    try:
        # Monitor the workflow state until the workflow is not in an active
        # state anymore.
        while state.is_active():
            time.sleep(poll_interval)
            # Get the current workflow status
            curr_state = client.get_workflow_state(
                workflow_id=workflow_id,
                current_state=state
            )
            logging.info('current state is {}'.format(curr_state))
            if state == curr_state:
                # Do nothing if the workflow status hasn't changed
                continue
            state = curr_state
            if state.is_success():
                # Create a temporary directory to download the run result
                # files.
                rundir = tempfile.mkdtemp()
                # Download the result files. The curr_state object is not
                # expected to contain the resource file information.
                files = list()
                for relative_path in output_files:
                    target = os.path.join(rundir, relative_path)
                    client.download_file(
                        workflow_id=workflow_id,
                        source=relative_path,
                        target=target
                    )
                    files.append(relative_path)
                # Create a modified workflow state handle that contains the
                # workflow result resources.
                state = StateSuccess(
                    created_at=state.created_at,
                    started_at=state.started_at,
                    finished_at=state.finished_at,
                    files=files
                )
            # Update the local state and the workflow state in the service
            # API. If the service object is None the run state will be updated
            # by the calling code.
            if service is None:
                continue
            try:
                with service() as api:
                    api.runs().update_run(run_id, state, rundir=rundir)
            except Exception as ex:
                # If the workflow is canceled for example, the state in the
                # API will have been changed and this may cause an error
                # here. If the remote workflow, however, remains active we
                # notify the remote engine to stop the workflow.
                logging.error('attempt to update run {}'.format(run_id))
                logging.error(ex)
                if state.is_active():
                    try:
                        client.stop_workflow(workflow_id)
                    except Exception as ex:
                        logging.error(ex)
                # Stop the thread by exiting the run method.
                return state, rundir
    except Exception as ex:
        logging.error(ex)
        strace = util.stacktrace(ex)
        logging.debug('\n'.join(strace))
        state = state.error(messages=strace)
        if service is not None:
            with service() as api:
                api.runs().update_run(run_id, state)
    msg = 'finished run {} = {}'.format(run_id, state.type_id)
    logging.info(msg)
    return state, rundir
