"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComputeEnvironment = exports.AllocationStrategy = exports.ComputeResourceType = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const batch_generated_1 = require("./batch.generated");
/**
 * (experimental) Property to specify if the compute environment uses On-Demand or SpotFleet compute resources.
 *
 * @experimental
 */
var ComputeResourceType;
(function (ComputeResourceType) {
    ComputeResourceType["ON_DEMAND"] = "EC2";
    ComputeResourceType["SPOT"] = "SPOT";
})(ComputeResourceType = exports.ComputeResourceType || (exports.ComputeResourceType = {}));
/**
 * (experimental) Properties for how to prepare compute resources that are provisioned for a compute environment.
 *
 * @experimental
 */
var AllocationStrategy;
(function (AllocationStrategy) {
    AllocationStrategy["BEST_FIT"] = "BEST_FIT";
    AllocationStrategy["BEST_FIT_PROGRESSIVE"] = "BEST_FIT_PROGRESSIVE";
    AllocationStrategy["SPOT_CAPACITY_OPTIMIZED"] = "SPOT_CAPACITY_OPTIMIZED";
})(AllocationStrategy = exports.AllocationStrategy || (exports.AllocationStrategy = {}));
/**
 * (experimental) Batch Compute Environment.
 *
 * Defines a batch compute environment to run batch jobs on.
 *
 * @experimental
 */
class ComputeEnvironment extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props = { enabled: true, managed: true }) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.computeEnvironmentName,
        });
        this.validateProps(props);
        const spotFleetRole = this.getSpotFleetRole(props);
        let computeResources;
        // Only allow compute resources to be set when using MANAGED type
        if (props.computeResources && this.isManaged(props)) {
            computeResources = {
                allocationStrategy: props.computeResources.allocationStrategy
                    || (props.computeResources.type === ComputeResourceType.SPOT
                        ? AllocationStrategy.SPOT_CAPACITY_OPTIMIZED
                        : AllocationStrategy.BEST_FIT),
                bidPercentage: props.computeResources.bidPercentage,
                desiredvCpus: props.computeResources.desiredvCpus,
                ec2KeyPair: props.computeResources.ec2KeyPair,
                imageId: props.computeResources.image && props.computeResources.image.getImage(this).imageId,
                instanceRole: props.computeResources.instanceRole
                    ? props.computeResources.instanceRole
                    : new iam.CfnInstanceProfile(this, 'Instance-Profile', {
                        roles: [new iam.Role(this, 'Ecs-Instance-Role', {
                                assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                                managedPolicies: [
                                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'),
                                ],
                            }).roleName],
                    }).attrArn,
                instanceTypes: this.buildInstanceTypes(props.computeResources.instanceTypes),
                launchTemplate: props.computeResources.launchTemplate,
                maxvCpus: props.computeResources.maxvCpus || 256,
                minvCpus: props.computeResources.minvCpus || 0,
                placementGroup: props.computeResources.placementGroup,
                securityGroupIds: this.buildSecurityGroupIds(props.computeResources.vpc, props.computeResources.securityGroups),
                spotIamFleetRole: spotFleetRole === null || spotFleetRole === void 0 ? void 0 : spotFleetRole.roleArn,
                subnets: props.computeResources.vpc.selectSubnets(props.computeResources.vpcSubnets).subnetIds,
                tags: props.computeResources.computeResourcesTags,
                type: props.computeResources.type || ComputeResourceType.ON_DEMAND,
            };
        }
        const computeEnvironment = new batch_generated_1.CfnComputeEnvironment(this, 'Resource', {
            computeEnvironmentName: this.physicalName,
            computeResources,
            serviceRole: (_b = (_a = props.serviceRole) === null || _a === void 0 ? void 0 : _a.roleArn) !== null && _b !== void 0 ? _b : new iam.Role(this, 'Resource-Service-Instance-Role', {
                managedPolicies: [
                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBatchServiceRole'),
                ],
                assumedBy: new iam.ServicePrincipal('batch.amazonaws.com'),
            }).roleArn,
            state: props.enabled === undefined ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            type: this.isManaged(props) ? 'MANAGED' : 'UNMANAGED',
        });
        if (props.computeResources && props.computeResources.vpc) {
            this.node.addDependency(props.computeResources.vpc);
        }
        this.computeEnvironmentArn = this.getResourceArnAttribute(computeEnvironment.ref, {
            service: 'batch',
            resource: 'compute-environment',
            resourceName: this.physicalName,
        });
        this.computeEnvironmentName = this.getResourceNameAttribute(computeEnvironment.ref);
    }
    /**
     * (experimental) Fetches an existing batch compute environment by its amazon resource name.
     *
     * @experimental
     */
    static fromComputeEnvironmentArn(scope, id, computeEnvironmentArn) {
        const stack = core_1.Stack.of(scope);
        const computeEnvironmentName = stack.parseArn(computeEnvironmentArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.computeEnvironmentArn = computeEnvironmentArn;
                this.computeEnvironmentName = computeEnvironmentName;
            }
        }
        return new Import(scope, id);
    }
    isManaged(props) {
        var _a;
        return (_a = props.managed) !== null && _a !== void 0 ? _a : true;
    }
    /**
     * Validates the properties provided for a new batch compute environment.
     */
    validateProps(props) {
        if (props === undefined) {
            return;
        }
        if (!this.isManaged(props) && props.computeResources !== undefined) {
            throw new Error('It is not allowed to set computeResources on an AWS unmanaged compute environment');
        }
        if (this.isManaged(props) && props.computeResources === undefined) {
            throw new Error('computeResources is missing but required on a managed compute environment');
        }
        // Setting a bid percentage is only allowed on SPOT resources +
        // Cannot use SPOT_CAPACITY_OPTIMIZED when using ON_DEMAND
        if (props.computeResources) {
            if (props.computeResources.type === ComputeResourceType.ON_DEMAND) {
                // VALIDATE FOR ON_DEMAND
                // Bid percentage is not allowed
                if (props.computeResources.bidPercentage !== undefined) {
                    throw new Error('Setting the bid percentage is only allowed for SPOT type resources on a batch compute environment');
                }
                // SPOT_CAPACITY_OPTIMIZED allocation is not allowed
                if (props.computeResources.allocationStrategy && props.computeResources.allocationStrategy === AllocationStrategy.SPOT_CAPACITY_OPTIMIZED) {
                    throw new Error('The SPOT_CAPACITY_OPTIMIZED allocation strategy is only allowed if the environment is a SPOT type compute environment');
                }
            }
            else {
                // VALIDATE FOR SPOT
                // Bid percentage must be from 0 - 100
                if (props.computeResources.bidPercentage !== undefined &&
                    (props.computeResources.bidPercentage < 0 || props.computeResources.bidPercentage > 100)) {
                    throw new Error('Bid percentage can only be a value between 0 and 100');
                }
            }
            if (props.computeResources.minvCpus) {
                // minvCpus cannot be less than 0
                if (props.computeResources.minvCpus < 0) {
                    throw new Error('Minimum vCpus for a batch compute environment cannot be less than 0');
                }
                // minvCpus cannot exceed max vCpus
                if (props.computeResources.maxvCpus &&
                    props.computeResources.minvCpus > props.computeResources.maxvCpus) {
                    throw new Error('Minimum vCpus cannot be greater than the maximum vCpus');
                }
            }
        }
    }
    buildInstanceTypes(instanceTypes) {
        if (instanceTypes === undefined) {
            return [
                'optimal',
            ];
        }
        return instanceTypes.map((type) => type.toString());
    }
    buildSecurityGroupIds(vpc, securityGroups) {
        if (securityGroups === undefined) {
            return [
                new ec2.SecurityGroup(this, 'Resource-Security-Group', { vpc }).securityGroupId,
            ];
        }
        return securityGroups.map((group) => group.securityGroupId);
    }
    /**
     * Generates an AWS IAM role for provisioning spotfleet resources
     * if the allocation strategy is set to BEST_FIT or not defined.
     *
     * @param props - the compute environment construct properties
     */
    getSpotFleetRole(props) {
        var _a;
        if (((_a = props.computeResources) === null || _a === void 0 ? void 0 : _a.allocationStrategy) && props.computeResources.allocationStrategy !== AllocationStrategy.BEST_FIT) {
            return undefined;
        }
        if (props.computeResources) {
            if (props.computeResources.spotFleetRole) {
                return props.computeResources.spotFleetRole;
            }
            else if (props.computeResources.type === ComputeResourceType.SPOT) {
                return iam.Role.fromRoleArn(this, 'Resource-SpotFleet-Role', `arn:${this.stack.partition}:iam::${this.stack.account}:role/aws-service-role/spotfleet.amazonaws.com/AWSServiceRoleForEC2SpotFleet`);
            }
        }
        return undefined;
    }
}
exports.ComputeEnvironment = ComputeEnvironment;
//# sourceMappingURL=data:application/json;base64,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