import subprocess
from typing import Optional

from .config import *


def get_python_relative_path(path: Path):
    """
    Converting path to python format with relative to ROOT DIR
    :param path: pathlib.Path object
    :return: str with python relative path
    """
    return str(path.relative_to(ROOT_DIR)).replace("/", '.').replace('\\', '.')


def converter(ui_path: Path) -> None:
    """
    Converts the .ui files to .py with resource imports
    Args:
        ui_path: path to ui file generated by PyQt-tools Designer
    """
    output_file = str(VIEWS_DIR / ui_path.stem) + '.py'
    try:
        subprocess.run([
            'pyuic5',
            str(ui_path),
            '--import-from',
            get_python_relative_path(VIEWS_DIR),
            '-o',
            output_file
        ], check=True)
        print(f"Converted successful: {ui_path.name} -> {output_file}")
    except subprocess.CalledProcessError as e:
        print(f"Error of converting {ui_path.name}: {e}")


def convert_ui_to_py(ui_file: Optional[str] = None, auto_convert_qrc: bool = False):
    """
    Converts .ui files to .py files using pyuic5.

    Args:
        :param ui_file: (str, optional): Specific .ui file to convert.
                                If None, converts all .ui files.
        :param auto_convert_qrc: All qrc file will be converted if True
    """

    if ui_file:
        # Convert specific file
        ui_path = UI_DIR / ui_file
        if not ui_path.exists():
            print(f"Error: File {ui_file} not found in {UI_DIR}")
            return

        if not ui_file.endswith('.ui'):
            print("Error: File must have .ui extension")
            return

        converter(ui_path)

    else:
        # Convert all .ui files
        ui_files = list(UI_DIR.glob('*.ui'))
        if not ui_files:
            print(f"Error: No .ui files found in {UI_DIR}")
            return

        for ui_path in ui_files:
            converter(ui_path)

    if auto_convert_qrc:
        convert_qrc_to_py()


def convert_qrc(qrc_path: Path) -> None:
    """
    Converts a single .qrc file to .py using pyrcc5

    Args:
        qrc_path: Path to the .qrc file to convert
    """
    output_file = str(VIEWS_DIR / qrc_path.stem) + '_rc.py'
    try:
        subprocess.run(['pyrcc5', str(qrc_path), '-o', output_file], check=True)
        print(f"Converted successfully: {qrc_path.name} -> {output_file}")
    except subprocess.CalledProcessError as e:
        print(f"Error converting {qrc_path.name}: {e}")


def convert_qrc_to_py(qrc_file: Optional[str] = None):
    """
    Converts .qrc files to .py files using pyrcc5.

    Args:
        qrc_file (str, optional): Specific .qrc file to convert.
                                 If None, converts all .qrc files.
    """
    if qrc_file:
        # Convert specific file
        qrc_path = RESOURCES_DIR / qrc_file
        if not qrc_path.exists():
            print(f"Error: File {qrc_file} not found in {RESOURCES_DIR}")
            return

        if not qrc_file.endswith('.qrc'):
            print("Error: File must have .qrc extension")
            return

        convert_qrc(qrc_path)

    else:
        # Convert all .qrc files in the directory
        qrc_files = list(RESOURCES_DIR.glob('*.qrc'))
        if not qrc_files:
            print(f"Error: No .qrc files found in {RESOURCES_DIR}")
            return

        for qrc_path in qrc_files:
            convert_qrc(qrc_path)
