#!/usr/bin/env python
''' Definition of constants

@author: Camille Maumet <c.m.j.maumet@warwick.ac.uk>
@copyright: University of Warwick 2014
@author: David Keator <dbkeator@uci.edu>
    Added Python provtoolbox  support
    10/3/17 Modified Namespace to be QualifiedName for provtoolbox support...left most of the NIDM-Results Namespaces the same
@author: Sanu Ann Abraham <sanuann@mit.edu>
	05/04/2018 Added python ProvONE support
'''
import six
from rdflib import Namespace, Graph
from prov.model import ProvDocument, QualifiedName
from prov.model import Namespace as provNamespace
from prov.constants import PROV_ATTRIBUTE_QNAMES, PROV_ATTRIBUTE_LITERALS, \
	PROV_N_MAP

from collections import namedtuple
DD = namedtuple("DD", ["source", "variable"])

PROV = Namespace('http://www.w3.org/ns/prov#')
PROVONE = provNamespace('provone', 'http://purl.dataone.org/provone/2015/01/15/ontology#')

NIDM_URL = 'http://purl.org/nidash/nidm#'
NIDM = Namespace(NIDM_URL)

NIIRI = Namespace('http://iri.nidash.org/')
AFNI = Namespace('http://purl.org/nidash/afni#')
SPM = Namespace('http://purl.org/nidash/spm#')
FSL = Namespace('http://purl.org/nidash/fsl#')
FREESURFER = Namespace('https://surfer.nmr.mgh.harvard.edu/')
ANTS = Namespace('http://stnava.github.io/ANTs/')
RDFS = Namespace('http://www.w3.org/2000/01/rdf-schema#')
CRYPTO = Namespace('http://id.loc.gov/vocabulary/preservation/\
cryptographicHashFunctions#')
DC = Namespace('http://purl.org/dc/elements/1.1/')
DCT = Namespace('http://purl.org/dc/terms/')
OWL = Namespace('http://www.w3.org/2002/07/owl#')
XSD = Namespace('http://www.w3.org/2001/XMLSchema#')

OBO_URL = "http://purl.obolibrary.org/obo/"
OBO = Namespace(OBO_URL)
#Added by DBK for NIDM-Experiment 1/13/17
NFO = Namespace('http://www.semanticdesktop.org/ontologies/2007/03/22/nfo#')
SCR = Namespace("http://scicrunch.org/resolver/")
NLX = Namespace("http://uri.neuinfo.org/nif/nifstd/")
SKOS = Namespace("http://www.w3.org/2004/02/skos/core#")
FOAF = Namespace("http://xmlns.com/foaf/0.1/")
VC = Namespace("http://www.w3.org/2006/vcard/ns#")
DICOM = Namespace("http://neurolex.org/wiki/Category:DICOM_term/")
DCTYPES = Namespace("http://purl.org/dc/dcmitype/")
NCIT = Namespace("http://ncitt.ncit.nih.gov/")
DCAT = Namespace("http://www.w3.org/ns/dcat#")
BIRNLEX = Namespace("http://bioontology.org/projects/ontologies/birnlex/")
NDAR = Namespace("https://ndar.nih.gov/api/datadictionary/v2/dataelement/")
NCICB = Namespace("http://ncicb.nci.nih.gov/xml/owl/EVS/Thesaurus.owl#")
SIO = Namespace("http://semanticscience.org/ontology/sio.owl#")
BIDS = Namespace("http://bids.neuroimaging.io/")
OWL = Namespace("http://www.w3.org/2002/07/owl#")
ONLI = Namespace("http://neurolog.unice.fr/ontoneurolog/v3.0/instrument.owl#")
PATO = Namespace("http://purl.obolibrary.org/obo/pato#")
DATALAD = Namespace("http://datasets.datalad.org/")

namespaces = {
   # "prov": PROV,
    "nidm": NIDM,
    "niiri": NIIRI,
    "afni": AFNI,
    "spm": SPM,
    "fsl": FSL,
	"freesurfer": FREESURFER,
	"ants": ANTS,
    "rdfs": RDFS,
    "crypto": CRYPTO,
    "dct": DCT,
    "obo": OBO,
    "nfo": NFO,
    "dc": DC,
    "nlx": NLX,
    "scr": SCR,
    "foaf": FOAF,
    "vc": VC,
    "dicom": DICOM,
    "dctypes": DCTYPES,
    "ncit": NCIT,
    "dcat": DCAT,
    "birnlex" : BIRNLEX,
    "ndar" : NDAR,
    "ncicb" : NCICB,
    "sio" : SIO,
    "bids" : BIDS,
    "owl" : OWL,
    "onli" : ONLI,
    "pato" : PATO,
	"datalad" : DATALAD
    }

# Empty graph used to compute qnames
q_graph = Graph()
for name, namespace in namespaces.items():
	q_graph.bind(name, namespace)

# DBK Added - Empty graph using provtoolbox used to compute qnames
# dj: chnaged to a new class
class NIDMDocument(ProvDocument):
	def __init__(self, namespaces=None):
		if namespaces is not None:
			super(NIDMDocument, self).__init__(namespaces=namespaces)
		else:
			super(NIDMDocument, self).__init__()


# NIDM constants
FSL_GAMMAHRF = FSL['FSL_0000007']
FSL_FSLS_GAMMA_HRF = FSL['FSL_0000006']
NIDM_HAS_MRI_PROTOCOL = NIDM['NIDM_0000172']
NIDM_NUMBER_OF_SUBJECTS = NIDM['NIDM_0000171']
NIDM_GROUP_NAME = NIDM['NIDM_0000170']
NIDM_DATA = NIDM['NIDM_0000169']
NIDM_SPM_RESULTS_NIDM = NIDM['NIDM_0000168']
NIDM_NIDMFSL = NIDM['NIDM_0000167']
NIDM_NIDM_RESULTS_EXPORT = NIDM['NIDM_0000166']
NIDM_NIDM_RESULTS_EXPORTER = NIDM['NIDM_0000165']
NIDM_NEUROIMAGING_ANALYSIS_SOFTWARE = NIDM['NIDM_0000164']
NIDM_CONTRAST_EXPLAINED_MEAN_SQUARE_MAP = NIDM['NIDM_0000163']
NIDM_THRESHOLD = NIDM['NIDM_0000162']
NIDM_EQUIVALENT_THRESHOLD = NIDM['NIDM_0000161']
NIDM_P_VALUE_UNCORRECTED = NIDM['NIDM_0000160']
NIDM_P_VALUE_UNCORRECTED_QNAME = q_graph.qname(NIDM_P_VALUE_UNCORRECTED)
NIDM_NOISE_FWHM_IN_VOXELS = NIDM['NIDM_0000159']
NIDM_NOISE_FWHM_IN_VERTICES = NIDM['NIDM_0000158']
NIDM_NOISE_FWHM_IN_UNITS = NIDM['NIDM_0000157']
FSL_FEAT_VERSION = FSL['FSL_0000005']
FSL_DRIFT_CUTOFF_PERIOD = FSL['FSL_0000004']
FSL_TEMPORAL_DERIVATIVE = FSL['FSL_0000003']
FSL_GAUSSIAN_RUNNING_LINE_DRIFT_MODEL = FSL['FSL_0000002']
FSL_FSLS_GAMMA_DIFFERENCE_HRF = FSL['FSL_0000001']
SPM_PARTIAL_CONJUNCTION_DEGREE = SPM['SPM_0000015']
SPM_SMALLEST_SUPRA_THRESHOLD_CLUSTER_SIZE_IN_VOXELS_FWE05 = SPM['SPM_0000014']
SPM_SMALLEST_SUPRA_THRESHOLD_CLUSTER_SIZE_IN_VOXELS_FDR05 = SPM['SPM_0000013']
SPM_SMALLEST_SUPRA_THRESHOLD_CLUSTER_SIZE_IN_VERTICES_FWE05 = SPM['SPM_0000012']
SPM_SMALLEST_SUPRA_THRESHOLD_CLUSTER_SIZE_IN_VERTICES_FDR05 = SPM['SPM_0000011']
SPM_SEARCH_VOLUME_RESELS_GEOMETRY = SPM['SPM_0000010']
SPM_TEMPORAL_DERIVATIVE = SPM['SPM_0000006']
SPM_KCONJUNCTION_INFERENCE = SPM['SPM_0000005']
SPM_CANONICAL_HRF = SPM['SPM_0000004']
SPM_DISPERSION_DERIVATIVE = SPM['SPM_0000003']
SPM_DCT_DRIFT_MODEL = SPM['SPM_0000002']
SPM_SPMS_DRIFT_CUT_OFF_PERIOD = SPM['SPM_0000001']
NIDM_CLUSTERSIZEINRESELS = NIDM['NIDM_0000156']
NIDM_F_MRI_DESIGN = NIDM['NIDM_0000155']
NIDM_MIXED_DESIGN = NIDM['NIDM_0000154']
NIDM_EVENT_RELATED_DESIGN = NIDM['NIDM_0000153']
NIDM_BLOCK_BASED_DESIGN = NIDM['NIDM_0000152']
NIDM_SINE_BASIS_SET = NIDM['NIDM_0000151']
NIDM_LINEAR_SPLINE_BASIS_SET = NIDM['NIDM_0000150']
NIDM_SEARCH_VOLUME_IN_RESELS = NIDM['NIDM_0000149']
NIDM_RESEL_SIZE_IN_VOXELS = NIDM['NIDM_0000148']
NIDM_HEIGHT_CRITICAL_THRESHOLD_FWE_05 = NIDM['NIDM_0000147']
NIDM_HEIGHT_CRITICAL_THRESHOLD_FDR_05 = NIDM['NIDM_0000146']
NIDM_NOISE_ROUGHNESS_IN_VOXELS = NIDM['NIDM_0000145']
NIDM_RESELS_PER_VOXEL_MAP = NIDM['NIDM_0000144']
NIDM_EXPECTED_NUMBER_OF_VOXELS_PER_CLUSTER = NIDM['NIDM_0000143']
NIDM_EXPECTED_NUMBER_OF_VERTICES_PER_CLUSTER = NIDM['NIDM_0000142']
NIDM_EXPECTED_NUMBER_OF_CLUSTERS = NIDM['NIDM_0000141']
NIDM_CLUSTER_CENTER_OF_GRAVITY = NIDM['NIDM_0000140']
NIDM_COORDINATE_VECTOR_IN_VOXELS = NIDM['NIDM_0000139']
NIDM_HAS_MAXIMUM_INTENSITY_PROJECTION = NIDM['NIDM_0000138']
NIDM_SEARCH_VOLUME_IN_VERTICES = NIDM['NIDM_0000137']
NIDM_SEARCH_VOLUME_IN_UNITS = NIDM['NIDM_0000136']
NIDM_CONTRAST_VARIANCE_MAP = NIDM['NIDM_0000135']
NIDM_WITH_ESTIMATION_METHOD = NIDM['NIDM_0000134']
NIDM_VOXEL_UNITS = NIDM['NIDM_0000133']
NIDM_VOXEL_TO_WORLD_MAPPING = NIDM['NIDM_0000132']
NIDM_VOXEL_SIZE = NIDM['NIDM_0000131']
NIDM_VOXEL6CONNECTED = NIDM['NIDM_0000130']
NIDM_VOXEL26CONNECTED = NIDM['NIDM_0000129']
NIDM_VOXEL18CONNECTED = NIDM['NIDM_0000128']
NIDM_VERSION = NIDM['NIDM_0000127']
NIDM_VARIANCE_SPATIAL_MODEL = NIDM['NIDM_0000126']
NIDM_USER_SPECIFIED_THRESHOLD_TYPE = NIDM['NIDM_0000125']
NIDM_TARGET_INTENSITY = NIDM['NIDM_0000124']
NIDM_STATISTIC_TYPE = NIDM['NIDM_0000123']
NIDM_SOFTWARE_VERSION = NIDM['NIDM_0000122']
NIDM_SEARCH_VOLUME_IN_VOXELS = NIDM['NIDM_0000121']
NIDM_RANDOM_FIELD_STATIONARITY = NIDM['NIDM_0000120']
NIDM_Q_VALUE_FDR = NIDM['NIDM_0000119']
NIDM_PIXEL8CONNECTED = NIDM['NIDM_0000118']
NIDM_PIXEL4CONNECTED = NIDM['NIDM_0000117']
NIDM_P_VALUE_UNCORRECTED = NIDM['NIDM_0000116']
NIDM_P_VALUE_FWER = NIDM['NIDM_0000115']
NIDM_P_VALUE = NIDM['NIDM_0000114']
NIDM_OBJECT_MODEL = NIDM['NIDM_0000113']
NIDM_NUMBER_OF_DIMENSIONS = NIDM['NIDM_0000112']
NIDM_NUMBER_OF_CLUSTERS = NIDM['NIDM_0000111']
NIDM_GAUSSIAN_HRF = NIDM['NIDM_0000110']
NIDM_MIN_DISTANCE_BETWEEN_PEAKS = NIDM['NIDM_0000109']
NIDM_MAX_NUMBER_OF_PEAKS_PER_CLUSTER = NIDM['NIDM_0000108']
NIDM_MASKED_MEDIAN = NIDM['NIDM_0000107']
NIDM_IS_USER_DEFINED = NIDM['NIDM_0000106']
NIDM_IN_WORLD_COORDINATE_SYSTEM = NIDM['NIDM_0000105']
NIDM_IN_COORDINATE_SPACE = NIDM['NIDM_0000104']
NIDM_HAS_MAP_HEADER = NIDM['NIDM_0000103']
NIDM_HAS_HRF_BASIS = NIDM['NIDM_0000102']
NIDM_HAS_ERROR_DISTRIBUTION = NIDM['NIDM_0000101']
NIDM_HAS_ERROR_DEPENDENCE = NIDM['NIDM_0000100']
NIDM_HAS_CONNECTIVITY_CRITERION = NIDM['NIDM_0000099']
NIDM_HAS_CLUSTER_LABELS_MAP = NIDM['NIDM_0000098']
NIDM_HAS_ALTERNATIVE_HYPOTHESIS = NIDM['NIDM_0000097']
NIDM_GRAND_MEAN_SCALING = NIDM['NIDM_0000096']
NIDM_ERROR_VARIANCE_HOMOGENEOUS = NIDM['NIDM_0000094']
NIDM_ERROR_DEGREES_OF_FREEDOM = NIDM['NIDM_0000093']
NIDM_EQUIVALENT_ZSTATISTIC = NIDM['NIDM_0000092']
NIDM_EFFECT_DEGREES_OF_FREEDOM = NIDM['NIDM_0000091']
NIDM_DIMENSIONS_IN_VOXELS = NIDM['NIDM_0000090']
NIDM_DEPENDENCE_SPATIAL_MODEL = NIDM['NIDM_0000089']
NIDM_HAS_DRIFT_MODEL = NIDM['NIDM_0000088']
NIDM_DRIFT_MODEL = NIDM['NIDM_0000087']
NIDM_COORDINATE_VECTOR = NIDM['NIDM_0000086']
NIDM_CONTRAST_NAME = NIDM['NIDM_0000085']
NIDM_CLUSTER_SIZE_IN_VOXELS = NIDM['NIDM_0000084']
NIDM_CLUSTER_SIZE_IN_VERTICES = NIDM['NIDM_0000083']
NIDM_CLUSTER_LABEL_ID = NIDM['NIDM_0000082']
NIDM_WORLD_COORDINATE_SYSTEM = NIDM['NIDM_0000081']
NIDM_VOXEL_CONNECTIVITY_CRITERION = NIDM['NIDM_0000080']
NIDM_TWO_TAILED_TEST = NIDM['NIDM_0000079']
NIDM_TALAIRACH_COORDINATE_SYSTEM = NIDM['NIDM_0000078']
NIDM_SUBJECT_COORDINATE_SYSTEM = NIDM['NIDM_0000077']
NIDM_STATISTIC_MAP = NIDM['NIDM_0000076']
NIDM_STANDARDIZED_COORDINATE_SYSTEM = NIDM['NIDM_0000075']
NIDM_SPATIALLY_REGULARIZED_MODEL = NIDM['NIDM_0000074']
NIDM_SPATIALLY_LOCAL_MODEL = NIDM['NIDM_0000073']
NIDM_SPATIALLY_GLOBAL_MODEL = NIDM['NIDM_0000072']
NIDM_SPATIAL_MODEL = NIDM['NIDM_0000071']
NIDM_SUPRA_THRESHOLD_CLUSTER = NIDM['NIDM_0000070']
NIDM_FOURIER_BASIS_SET = NIDM['NIDM_0000069']
NIDM_SEARCH_SPACE_MASK_MAP = NIDM['NIDM_0000068']
NIDM_CUSTOM_BASIS_SET = NIDM['NIDM_0000067']
NIDM_RESIDUAL_MEAN_SQUARES_MAP = NIDM['NIDM_0000066']
NIDM_POISSON_DISTRIBUTION = NIDM['NIDM_0000065']
NIDM_PIXEL_CONNECTIVITY_CRITERION = NIDM['NIDM_0000064']
NIDM_PEAK_DEFINITION_CRITERIA = NIDM['NIDM_0000063']
NIDM_PEAK = NIDM['NIDM_0000062']
NIDM_PARAMETER_ESTIMATE_MAP = NIDM['NIDM_0000061']
NIDM_ONE_TAILED_TEST = NIDM['NIDM_0000060']
NIDM_NON_PARAMETRIC_SYMMETRIC_DISTRIBUTION = NIDM['NIDM_0000059']
NIDM_NON_PARAMETRIC_DISTRIBUTION = NIDM['NIDM_0000058']
NIDM_NIDM_OBJECT_MODEL = NIDM['NIDM_0000057']
NIDM_MODEL_PARAMETERS_ESTIMATION = NIDM['NIDM_0000056']
NIDM_MNI305_COORDINATE_SYSTEM = NIDM['NIDM_0000055']
NIDM_MASK_MAP = NIDM['NIDM_0000054']
NIDM_MAP_HEADER = NIDM['NIDM_0000053']
NIDM_MAP = NIDM['NIDM_0000052']
NIDM_MNI_COORDINATE_SYSTEM = NIDM['NIDM_0000051']
NIDM_IXI549_COORDINATE_SYSTEM = NIDM['NIDM_0000050']
NIDM_INFERENCE = NIDM['NIDM_0000049']
NIDM_INDEPENDENT_ERROR = NIDM['NIDM_0000048']
NIDM_ICBM_MNI152_NON_LINEAR6TH_GENERATION_COORDINATE_SYSTEM = NIDM['NIDM_0000047']
NIDM_ICBM_MNI152_NON_LINEAR2009C_SYMMETRIC_COORDINATE_SYSTEM = NIDM['NIDM_0000046']
NIDM_ICBM_MNI152_NON_LINEAR2009C_ASYMMETRIC_COORDINATE_SYSTEM = NIDM['NIDM_0000045']
NIDM_ICBM_MNI152_NON_LINEAR2009B_SYMMETRIC_COORDINATE_SYSTEM = NIDM['NIDM_0000044']
NIDM_ICBM_MNI152_NON_LINEAR2009B_ASYMMETRIC_COORDINATE_SYSTEM = NIDM['NIDM_0000043']
NIDM_ICBM_MNI152_NON_LINEAR2009A_SYMMETRIC_COORDINATE_SYSTEM = NIDM['NIDM_0000042']
NIDM_ICBM_MNI152_NON_LINEAR2009A_ASYMMETRIC_COORDINATE_SYSTEM = NIDM['NIDM_0000041']
NIDM_ICBM_MNI152_LINEAR_COORDINATE_SYSTEM = NIDM['NIDM_0000040']
NIDM_ICBM452_WARP5_COORDINATE_SYSTEM = NIDM['NIDM_0000039']
NIDM_ICBM452_AIR_COORDINATE_SYSTEM = NIDM['NIDM_0000038']
NIDM_HEMODYNAMIC_RESPONSE_FUNCTION_DERIVATIVE = NIDM['NIDM_0000037']
NIDM_HEMODYNAMIC_RESPONSE_FUNCTION_BASIS = NIDM['NIDM_0000036']
NIDM_HEMODYNAMIC_RESPONSE_FUNCTION = NIDM['NIDM_0000035']
NIDM_HEIGHT_THRESHOLD = NIDM['NIDM_0000034']
NIDM_GRAND_MEAN_MAP = NIDM['NIDM_0000033']
NIDM_GAMMA_HRF = NIDM['NIDM_0000031']
NIDM_GAMMA_HRB = NIDM['NIDM_0000030']
NIDM_GAMMA_DIFFERENCE_HRF = NIDM['NIDM_0000029']
NIDM_FINITE_IMPULSE_RESPONSE_HRB = NIDM['NIDM_0000028']
NIDM_RESULTS = NIDM['NIDM_0000027']
NIDM_EXTENT_THRESHOLD = NIDM['NIDM_0000026']
NIDM_EXCURSION_SET_MAP = NIDM['NIDM_0000025']
NIDM_EXCHANGEABLE_ERROR = NIDM['NIDM_0000024']
NIDM_ERROR_MODEL = NIDM['NIDM_0000023']
NIDM_ERROR_DISTRIBUTION = NIDM['NIDM_0000022']
NIDM_REGRESSOR_NAMES = NIDM['NIDM_0000021']
NIDM_DISPLAY_MASK_MAP = NIDM['NIDM_0000020']
NIDM_DESIGN_MATRIX = NIDM['NIDM_0000019']
NIDM_CUSTOM_COORDINATE_SYSTEM = NIDM['NIDM_0000017']
NIDM_COORDINATE_SPACE = NIDM['NIDM_0000016']
NIDM_COORDINATE = NIDM['NIDM_0000015']
NIDM_LEGENDRE_POLYNOMIAL_ORDER = NIDM['NIDM_0000014']
NIDM_CONTRAST_STANDARD_ERROR_MAP = NIDM['NIDM_0000013']
NIDM_CONNECTIVITY_CRITERION = NIDM['NIDM_0000012']
NIDM_CONJUNCTION_INFERENCE = NIDM['NIDM_0000011']
NIDM_HAS_FMRI_DESIGN = NIDM['NIDM_0000010']
NIDM_COLIN27_COORDINATE_SYSTEM = NIDM['NIDM_0000009']
NIDM_CLUSTER_LABELS_MAP = NIDM['NIDM_0000008']
NIDM_CLUSTER_DEFINITION_CRITERIA = NIDM['NIDM_0000007']
NIDM_CLUSTER = NIDM['NIDM_0000006']
NIDM_BINOMIAL_DISTRIBUTION = NIDM['NIDM_0000005']
NIDM_BINARY_MAP = NIDM['NIDM_0000004']
NIDM_CONTRAST_ESTIMATION = NIDM['NIDM_0000001']
NIDM_CONTRAST_MAP = NIDM['NIDM_0000002']
# NIDM-Experiment##############################################################
NIDM_DATAELEMENT = QualifiedName(provNamespace("nidm", NIDM), 'DataElement')
NIDM_PROJECT = QualifiedName(provNamespace("nidm", NIDM), 'Project')
#NIDM_PROJECT_TYPE = QualifiedName(provNamespace("dctypes", DCTYPES),"Dataset")
NIDM_PROJECT_IDENTIFIER = QualifiedName(provNamespace("sio", SIO),"Identifier")
NIDM_PROJECT_NAME = QualifiedName(provNamespace("dctypes", DCTYPES),"title")
NIDM_PROJECT_DESCRIPTION = QualifiedName(provNamespace("dct", DCT),"description")
NIDM_DESCRIPTION = QualifiedName(provNamespace("dct", DCT),"description")
NIDM_DEFINITION = QualifiedName(provNamespace("dct", DCT),"description")
NIDM_PROJECT_LICENSE = QualifiedName(provNamespace("dct", DCT),"license")
NIDM_PROJECT_URL = QualifiedName(provNamespace("sio", SIO),"URL")
NIDM_PROJECT_REFERENCES = QualifiedName(provNamespace("dcat", DCAT),"creator")
NIDM_AUTHOR = QualifiedName(provNamespace("ncit", DCAT),"author")
NIDM_SESSION = QualifiedName(provNamespace("nidm", NIDM), 'Session')
NIDM_ACQUISITION_ACTIVITY = QualifiedName(provNamespace("nidm", NIDM), "Acquisition")
NIDM_ACQUISITION_MODALITY = QualifiedName(provNamespace("nidm",NIDM),"hadAcquisitionModality")
NIDM_ASSESSMENT_ACQUISITION = QualifiedName(provNamespace("onli", ONLI), "instrument-based-assessment")
NIDM_ACQUISITION_ENTITY = QualifiedName(provNamespace("nidm", NIDM), "AcquisitionObject")

NIDM_PROJECT_SOURCE = QualifiedName(provNamespace("dctypes", DCTYPES), "source")
NIDM_HAD_NUMERICAL_VALUE = QualifiedName(provNamespace("nidm", NIDM), "hadNumericalValue")
NIDM_BATH_SOLUTION = QualifiedName(provNamespace("nidm", NIDM), "BathSolution")
NIDM_CELL_TYPE = QualifiedName(provNamespace("nidm", NIDM), "CellType")
NIDM_CHANNEL_NUMBER = QualifiedName(provNamespace("nidm", NIDM), "ChannelNumber")
NIDM_ELECTRODE_IMPEDANCE = QualifiedName(provNamespace("nidm", NIDM), "ElectrodeImpedance")
NIDM_GROUP_LABEL = QualifiedName(provNamespace("nidm", NIDM), "GroupLabel")
NIDM_HOLLOW_ELECTRODE_SOLUTION = QualifiedName(provNamespace("nidm", NIDM), "HollowElectrodeSolution")
NIDM_HAD_IMAGE_CONTRACT_TYPE = QualifiedName(provNamespace("nidm", NIDM), "hadImageContractType")
NIDM_HAD_IMAGE_USAGE_TYPE = QualifiedName(provNamespace("nidm", NIDM), "hadImageUsageType")
NIDM_NUBMER_OF_CHANNELS = QualifiedName(provNamespace("nidm", NIDM), "NubmerOfChannels")
NIDM_APPLIED_FILTER = QualifiedName(provNamespace("nidm", NIDM), "AppliedFilter")
NIDM_SOLUTION_FLOW_SPEED = QualifiedName(provNamespace("nidm", NIDM), "SolutionFlowSpeed")
NIDM_RECORDING_LOCATION = QualifiedName(provNamespace("nidm", NIDM), "RecordingLocation")

NIDM_DEMOGRAPHICS_ENTITY = QualifiedName(provNamespace("nidm", NIDM), "DemographicsInstrument")
NIDM_ASSESSMENT_USAGE_TYPE = QualifiedName(provNamespace("nidm", NIDM),"AssessmentUsageType")


NIDM_ASSESSMENT_ENTITY = QualifiedName(provNamespace("onli", ONLI), "assessment-instrument")
#files
NIDM_FILENAME = QualifiedName(provNamespace("nfo", NFO), "filename")
NIDM_FILE = QualifiedName(provNamespace("sio", SIO), "file")
#Roles
NIDM_PI = QualifiedName(provNamespace("birnlex", BIRNLEX), "birnlex_2152")
NIDM_COI = QualifiedName(provNamespace("birnlex", BIRNLEX),"birnlex_2199")
NIDM_PARTICIPANT = QualifiedName(provNamespace("sio", SIO),"Subject")
#Demographics
NIDM_AGE = QualifiedName(provNamespace("ncidb",NCICB),"Age")
NIDM_GENDER = QualifiedName(provNamespace("ndar",NDAR),"gender")
NIDM_SEX = QualifiedName(provNamespace("pato",PATO),"PhenotypicSex")
NIDM_HANDEDNESS = QualifiedName(provNamespace("obo",OBO),"handedness")
#NIDM_HANDEDNESS = OBO["PATO_0002201"] is correct term ID for handedness above
NIDM_ETHNICITY = QualifiedName(provNamespace("sio",SIO),"ethnicity")
NIDM_RACE = QualifiedName(provNamespace("sio",SIO),"race")

#NCICB_ETHNICITY = NCICB["C16564"] is correct term ID for ethnic group
NIDM_DIAGNOSIS = QualifiedName(provNamespace("ncit",NCIT),"Diagnosis")
NIDM_FAMILY_NAME = QualifiedName(provNamespace("foaf",FOAF),"familyName")
NIDM_GIVEN_NAME = QualifiedName(provNamespace("foaf",FOAF),"givenName")
NIDM_SUBJECTID = QualifiedName(provNamespace("ndar",NDAR),"src_subject_id")
#MRI scan types
NIDM_IMAGE_CONTRAST_TYPE = QualifiedName(provNamespace("nidm", NIDM),"hadImageContrastType")
NIDM_IMAGE_USAGE_TYPE = QualifiedName(provNamespace("nidm", NIDM),"hadImageUsageType")
NIDM_PET = QualifiedName(provNamespace("nidm", NIDM),"PositronEmissionTomography")
NIDM_MRI = QualifiedName(provNamespace("nidm", NIDM),"MagneticResonanceImaging")
NIDM_MRI_ANATOMIC_SCAN = QualifiedName(provNamespace("nidm", NIDM),"Anatomical")
NIDM_MRI_STRUCTURE_SCAN = QualifiedName(provNamespace("nidm", NIDM),"Structural")
NIDM_MRI_FUNCTION_SCAN = QualifiedName(provNamespace("nidm", NIDM),"Functional")
NIDM_MRI_DWI_SCAN = QualifiedName(provNamespace("nidm", NIDM),"DiffusionWeighted")
NIDM_MRI_DWI_BVAL = QualifiedName(provNamespace("nidm", NIDM),"b-value")
NIDM_MRI_DWI_BVEC = QualifiedName(provNamespace("nidm", NIDM),"b-vector")
NIDM_MRI_FUNCTION_TASK = QualifiedName(provNamespace("nidm", NIDM),"Task")
NIDM_MRI_T1 = QualifiedName(provNamespace("nidm", NIDM),"T1Weighted")
NIDM_MRI_T2 = QualifiedName(provNamespace("nidm", NIDM),"T2Weighted")
NIDM_MRI_T2_STAR = QualifiedName(provNamespace("nidm", NIDM),"T2StarWeighted")
NIDM_MRI_DIFFUSION_TENSOR = QualifiedName(provNamespace("nidm", NIDM),"DiffusionTensor")
NIDM_MRI_FLOW = QualifiedName(provNamespace("nidm", NIDM),"FlowWeighted")
NIDM_MRI_BOLD_EVENTS = QualifiedName(provNamespace("nidm", NIDM),"StimulusResponseFile")
CRYPTO_SHA512  =QualifiedName(provNamespace("crypto", CRYPTO),"sha512")
DATALAD_LOCATION = QualifiedName(provNamespace("datalad", DATALAD),"Location")
##############################################################################
# OBO constants
OBO_EXAMPLE = OBO['IAO_0000112']
OBO_TERM_EDITOR = OBO['IAO_0000117']
OBO_EDITOR_NOTE = OBO['IAO_0000116']

OBO_PENDING_FINAL = OBO['IAO_0000125']
OBO_METADATA_COMPLETE = OBO['IAO_0000120']
OBO_METADATA_INCOMPLETE = OBO['IAO_0000123']
OBO_REQUIRES_DISCUSSION = OBO['IAO_0000428']
OBO_UNCURATED = OBO['IAO_0000124']
OBO_TO_BE_REPLACED = OBO['IAO_0000423']
OBO_READY = OBO['IAO_0000122']
OBO_DEFINITION = OBO['IAO_0000115']

OBO_STATISTIC = OBO['STATO_0000039']
OBO_STATISTIC_QNAME = q_graph.qname(OBO_STATISTIC)
OBO_P_VALUE_FWER = OBO['OBI_0001265']
OBO_P_VALUE_FWER_QNAME = q_graph.qname(OBO_P_VALUE_FWER)
OBO_Q_VALUE_FDR = OBO['OBI_0001442']
OBO_Q_VALUE_FDR_QNAME = q_graph.qname(OBO_Q_VALUE_FDR)

HAS_CURATION_STATUS = OBO['IAO_0000114']

STATO_OLS = OBO['STATO_0000370']
STATO_OLS_STR = q_graph.qname(STATO_OLS)
# TODO: labels should be grabbed automatically from the corresponding owl file
STATO_OLS_LABEL = "obo:'ordinary least squares estimation'"
STATO_GLS = OBO['STATO_0000372']
STATO_GLS_STR = q_graph.qname(STATO_GLS)
STATO_GLS_LABEL = "obo:'generalized least squares estimation'"
STATO_TSTATISTIC = OBO['STATO_0000176']
STATO_TSTATISTIC_STR = q_graph.qname(STATO_TSTATISTIC)
STATO_TSTATISTIC_LABEL = "obo:'t-statistic'"
STATO_ZSTATISTIC = OBO['STATO_0000376']
STATO_ZSTATISTIC_STR = q_graph.qname(STATO_ZSTATISTIC)
STATO_ZSTATISTIC_LABEL = "obo:'Z-statistic'"
STATO_CONTRAST_WEIGHT_MATRIX = OBO['STATO_0000323']
STATO_GAUSSIAN_DISTRIBUTION = OBO['STATO_0000227']
STATO_UNSTRUCTURED_COVARIANCE = OBO['STATO_0000405']
STATO_GROUP = OBO['STATO_0000193']

SPM_SOFTWARE = SCR['SCR_007037']
FSL_SOFTWARE = SCR['SCR_002823']

NLX_MRI_SCANNER = NLX['birnlex_2100']
NLX_FMRI_PROTOCOL = NLX['birnlex_2250']
NLX_IMAGING_INSTRUMENT = NLX['birnlex_2094']

SKOS_DEFINITION = SKOS['definition']


# ProvONE Constants for classes
PROVONE_PROCESS = PROVONE['Process']
PROVONE_USER = PROVONE['User']
PROVONE_PROCESSEXEC = PROVONE['ProcessExec']
PROVONE_DATA = PROVONE['Data']
PROVONE_INPUTPORT = PROVONE['InputPort']
PROVONE_OUTPUTPORT = PROVONE['OutputPort']
PROVONE_DATALINK = PROVONE['DataLink']
PROVONE_SEQCTRLLINK = PROVONE['seqCtrlLink']

# ProvONE Constants for Associations
PROVONE_HASOUTPORT = PROVONE['hasOutPort']
PROVONE_HASINPORT = PROVONE['hasInPort']
PROVONE_HASSUBPROCESS = PROVONE['hasSubProcess']
PROVONE_INPORTTODL = PROVONE['inPortToDL']
PROVONE_DLTOINPORT = PROVONE['DLToInPort']
PROVONE_OUTPORTTODL = PROVONE['outPortToDL']
PROVONE_DLTOOUTPORT = PROVONE['DLToOutPort']
PROVONE_CLTODESTP = PROVONE['CLtoDestP']
PROVONE_SOURCEPTOCL = PROVONE['sourcePToCL']
PROVONE_DATAONLINK = PROVONE['dataOnLink']
PROVONE_HASDEFAULTPARAM = PROVONE['hasDefaultParameter']
PROVONE_ISPARTOF = PROVONE['isPartOf']
PROVONE_MEMBERSHIP = PROVONE['hadMember']

# ProvONE notation mapping
PROVONE_N_MAP = {
	PROVONE_PROCESS: 			u'process',
	PROVONE_PROCESSEXEC: 		u'processExec',
	PROVONE_USER: 				u'user',
	PROVONE_DATA:				u'data',
	PROVONE_HASINPORT:			u'hasInPort',
	PROVONE_INPUTPORT:			u'inputPort',
	PROVONE_OUTPUTPORT:			u'outputPort',
	PROVONE_HASOUTPORT:			u'hasOutPort',
	PROVONE_HASSUBPROCESS:		u'hasSubProcess',
	PROVONE_INPORTTODL:			u'inPortToDL',
	PROVONE_DATALINK:			u'dataLink',
	PROVONE_SEQCTRLLINK:		u'seqCtrlLink',
	PROVONE_CLTODESTP:			u'CLtoDestP',
	PROVONE_SOURCEPTOCL:		u'sourcePtoCL',
	PROVONE_OUTPORTTODL:		u'outPortToDL',
	PROVONE_DLTOOUTPORT:		u'DLToOutPort',
	PROVONE_DLTOINPORT:			u'DLToInPort',
	PROVONE_DATAONLINK:			u'dataOnLink',
	PROVONE_HASDEFAULTPARAM: 	u'hasDefaultParamter',
	PROVONE_ISPARTOF:			u'isPartOf',
	PROVONE_MEMBERSHIP:			u'hadMember',

}

# Identifiers for PROVONE's attributes
PROVONE_ATTR_PROCESS = PROVONE['process']
PROVONE_ATTR_USER = PROVONE['user']
PROVONE_ATTR_PROCESSEXEC = PROVONE['processExec']
PROVONE_ATTR_PLAN = PROVONE['plan']
PROVONE_ATTR_GENERATED_DATA = PROVONE['generatedData']
PROVONE_ATTR_USED_DATA = PROVONE['usedData']
PROVONE_ATTR_GENERATION = PROVONE['generation']
#PROVONE_ATTR_USAGE = PROVONE['usage']
PROVONE_ATTR_DATA = PROVONE['data']
PROVONE_ATTR_INFORMED = PROVONE['informed']
PROVONE_ATTR_INFORMANT = PROVONE['informant']
PROVONE_ATTR_HASINPORT = PROVONE['hasInPort']
PROVONE_ATTR_HASOUTPORT = PROVONE['HasOutPort']
PROVONE_ATTR_INPUTPORT = PROVONE['InputPort']
PROVONE_ATTR_OUTPUTPORT = PROVONE['OutputPort']
PROVONE_ATTR_GENERATED_PROCESS = PROVONE['generatedProcess']
PROVONE_ATTR_USED_PROCESS = PROVONE['usedProcess']
PROVONE_ATTR_HASSUBPROCESS = PROVONE['hasSubProcess']
PROVONE_ATTR_DATALINK = PROVONE['dataLink']
PROVONE_ATTR_SEQCTRLLINK = PROVONE['seqCtrlLink']
PROVONE_ATTR_CLTODESTP = PROVONE['clToDestP']
PROVONE_ATTR_SOURCEPTOCL = PROVONE['sourcePtoCL']
PROVONE_ATTR_RELATED_PREXEC = PROVONE['relatedProcessExec'],
PROVONE_ATTR_USED_PREXEC = PROVONE['usedProcessExec']
PROVONE_ATTR_CHILD_PREXEC = PROVONE['childProcessExec']


PROVONE_ATTRIBUTE_QNAMES = {
	PROVONE_ATTR_PROCESS,
	PROVONE_ATTR_USER,
	PROVONE_ATTR_PROCESSEXEC,
	PROVONE_ATTR_PLAN,
	PROVONE_ATTR_GENERATED_DATA,
	PROVONE_ATTR_USED_DATA,
	PROVONE_ATTR_DATA,
	PROVONE_ATTR_INFORMED,
	PROVONE_ATTR_INFORMANT,
	PROVONE_ATTR_HASINPORT,
	PROVONE_ATTR_HASOUTPORT,
	PROVONE_ATTR_INPUTPORT,
	PROVONE_ATTR_OUTPUTPORT,
	PROVONE_ATTR_GENERATED_PROCESS,
	PROVONE_ATTR_USED_PROCESS,
	PROVONE_ATTR_HASSUBPROCESS,
	PROVONE_ATTR_DATALINK,
	PROVONE_ATTR_SEQCTRLLINK,
	PROVONE_ATTR_CLTODESTP,
	PROVONE_ATTR_SOURCEPTOCL,
	PROVONE_ATTR_RELATED_PREXEC,
	PROVONE_ATTR_USED_PREXEC,
	PROVONE_ATTR_CHILD_PREXEC,
    #PROV_ATTR_COLLECTION
}


# Set of formal attributes of PROV records
PROVONE_ATTRIBUTES = PROVONE_ATTRIBUTE_QNAMES | PROV_ATTRIBUTE_QNAMES | \
											  PROV_ATTRIBUTE_LITERALS
PROVONE_RECORD_ATTRIBUTES = list((attr, six.text_type(attr)) for attr in
							  PROVONE_ATTRIBUTES)

PROV_RECORD_IDS_MAP = dict(
    (PROV_N_MAP[rec_type_id], rec_type_id) for rec_type_id in PROV_N_MAP
)
PROVONE_ID_ATTRIBUTES_MAP = dict(
    (prov_id, attribute) for (prov_id, attribute) in PROVONE_RECORD_ATTRIBUTES
)
PROVONE_ATTRIBUTES_ID_MAP = dict(
    (attribute, prov_id) for (prov_id, attribute) in PROVONE_RECORD_ATTRIBUTES
)



####ADDED BY DBK to make searching NIDM-Experiment Terms easier...temporary, should be done in the OWL file#####
nidm_experiment_terms = [NIDM_PROJECT,
NIDM_PROJECT_IDENTIFIER,
NIDM_PROJECT_NAME,
NIDM_PROJECT_DESCRIPTION,
NIDM_PROJECT_LICENSE,
NIDM_PROJECT_URL,
NIDM_PROJECT_REFERENCES,
NIDM_AUTHOR,
NIDM_SESSION,
NIDM_ACQUISITION_ACTIVITY,
NIDM_ACQUISITION_MODALITY,
NIDM_ASSESSMENT_ACQUISITION,
NIDM_ACQUISITION_ENTITY,
NIDM_DEMOGRAPHICS_ENTITY,
NIDM_ASSESSMENT_ENTITY,
NIDM_FILENAME,
NIDM_FILE,
NIDM_PI,
NIDM_COI,
NIDM_PARTICIPANT,
NIDM_AGE,
NIDM_GENDER,
NIDM_SEX,
NIDM_HANDEDNESS,
NIDM_RACE,
NIDM_ETHNICITY,
NIDM_DIAGNOSIS,
NIDM_FAMILY_NAME,
NIDM_GIVEN_NAME,
NIDM_SUBJECTID,
NIDM_IMAGE_CONTRAST_TYPE,
NIDM_IMAGE_USAGE_TYPE,
NIDM_MRI,
NIDM_MRI_ANATOMIC_SCAN,
NIDM_MRI_STRUCTURE_SCAN,
NIDM_MRI_FUNCTION_SCAN,
NIDM_MRI_DWI_SCAN,
NIDM_MRI_DWI_BVAL,
NIDM_MRI_DWI_BVEC,
NIDM_MRI_FUNCTION_TASK,
NIDM_MRI_T1,
NIDM_MRI_T2,
NIDM_MRI_T2_STAR,
NIDM_MRI_DIFFUSION_TENSOR,
NIDM_MRI_FLOW,
NIDM_MRI_BOLD_EVENTS]


# cannonical CDE file locations
CDE_FILE_LOCATIONS = [
	"https://raw.githubusercontent.com/ReproNim/fsl_seg_to_nidm/master/fsl_seg_to_nidm/mapping_data/fsl_cde.ttl",
	"https://raw.githubusercontent.com/ReproNim/ants_seg_to_nidm/master/ants_seg_to_nidm/mapping_data/ants_cde.ttl",
	"https://raw.githubusercontent.com/ReproNim/segstats_jsonld/master/segstats_jsonld/mapping_data/fs_cde.ttl"
]