[![Build Status](https://travis-ci.org/childsish/dynamic-yaml.svg?branch=master)](https://travis-ci.org/childsish/dynamic-yaml)

dynamic-yaml
============

Dynamic YAML is a couple of classes and functions that add extra functionality to YAML that turns it into a great configuration language for Python. If you prefer JSON, then see [dynamic-json][dynamic-json].

YAML already provides:

* A very readable and clean syntax
* Infinitely nestable key:value pairs
* Sequence types
* A regulated portable syntax that conforms to strict standards

In addition, the PyYAML parser provides:

* Automatic type identification (a result of implementing the YAML standard)

Finally, the classes introduced by Dynamic YAML enable:

* Dynamic string resolution

Dynamic PyYAML requires PyYAML (https://bitbucket.org/xi/pyyaml).

Usage
-----
The key feature that was introduced is the ability for a string scalar to reference other parts of the configuration tree. This is done using the Python string formatting syntax. The characters '{' and '}' enclose a reference to another entry in the configuration structure. The reference takes the form key1.key2 where key1 maps to another mapping object and can be found in the root mapping, and key2 can be found in key1's mapping object. Multiple levels of nesting can be used (eg. key1.key2.key3 etc...).

An example yaml configuration:
```yaml
project_name: hello-world
dirs:
    home: /home/user
    venv: "{dirs.home}/venvs/{project_name}"
    bin: "{dirs.venv}/bin"
    data: "{dirs.venv}/data"
    errors: "{dirs.data}/errors"
    sessions: "{dirs.data}/sessions"
    databases: "{dirs.data}/databases"
    output: "{dirs.data}/output-{parameters.parameter1}-{parameters.parameter2}"
exes:
    main: "{dirs.bin}/main"
    test: tests
parameters:
    parameter1: a
    parameter2: b
```

Reading in a yaml file:

```python
import dynamic_yaml

with open('/path/to/file.yaml') as fileobj:
    cfg = dynamic_yaml.load(fileobj)
    assert cfg.dirs.venv == '/home/user/venvs/hello-world'
    assert cfg.dirs.output == '/home/user/venvs/hello-world/data/output-a-b'
```

As the variables are dynamically resolved, it is also possible to combine this with `argparse`:

```python
import dynamic_yaml

from argparse import ArgumentParser

with open('/path/to/file.yaml') as fileobj:
    cfg = dynamic_yaml.load(fileobj)
    parser = ArgumentParser()
    parser.add_argument('--parameter1')
    parser.add_argument('--parameter2')
    parser.parse_args('--parameter1 c --parameter2 d'.split(), namespace=cfg.parameters)
    assert cfg.dirs.output == '/home/user/venvs/hello-world/data/output-c-d'
```

Installation
------------

To install, simply run:

```bash
pip install dynamic-yaml
```

Restrictions
------------

Due to the short amount of time I was willing to spend on working upon this, there are a few restrictions that I could not overcome.

* **Wild card strings must sometimes be surrounded by quotes.**
Braces ('{' and '}') in a YAML file usually enclose a mapping object.
However, braces are also used by the Python string formatting syntax to enclose a reference.
As there is no way to change either of these easily, strings that look like a yaml mapping must be explicitly declared using single or double quotes to enclose them.
For example:
  ```yaml
  quotes_needed: '{variable}'
  no_quotes_needed: {variable1}-{variable2}
  ```
* **Certain keys can only be used via `__getitem__` and not `__getattr__`.**
Because `dict` comes with it's own set of attributes that are always resolved first, the values for the following keys must be gotten using the item getter rather than the attribute getter (eg. config['items'] vs. config.items):
  * append
  * extend
  * insert
  * remove
  * pop
  * clear
  * index
  * count
  * sort
  * reverse
  * copy 

[dynamic-json]: https://github.com/childsish/dynamic-json