"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VpcProvider = exports.AmiProvider = exports.Cluster = void 0;
const cdk = require("@aws-cdk/core");
const ec2 = require("@aws-cdk/aws-ec2");
const s3 = require("@aws-cdk/aws-s3");
const iam = require("@aws-cdk/aws-iam");
const autoscaling = require("@aws-cdk/aws-autoscaling");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const cr = require("@aws-cdk/custom-resources");
const path = require("path");
const DEFAULT_INSTANCE_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M6G, ec2.InstanceSize.MEDIUM);
let PriceMap = new Map([
    ['m6g.medium', '0.0385'],
    ['m6g.large', '0.077'],
]);
/**
 * Represents the k3sCluster construct
 */
class Cluster extends cdk.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        // VPC configuration
        const vpc = (_a = props.vpc) !== null && _a !== void 0 ? _a : new ec2.Vpc(this, 'Vpc', { maxAzs: 3, natGateways: 1 });
        // S3 bucket to host K3s token + kubeconfig file 
        const k3sBucket = new s3.Bucket(this, 'k3sBucket', {
            removalPolicy: (_b = props.bucketRemovalPolicy) !== null && _b !== void 0 ? _b : cdk.RemovalPolicy.RETAIN,
        });
        // Delete S3 Object CustomResource
        if (props.bucketRemovalPolicy === cdk.RemovalPolicy.DESTROY) {
            const onEvent = new lambda.Function(this, 'onEventHandler', {
                runtime: lambda.Runtime.PYTHON_3_8,
                code: lambda.Code.fromAsset(path.join(__dirname, '../custom-resource-handler')),
                handler: 'index.on_event',
            });
            const deleteS3ObjectProvider = new cr.Provider(this, 'deleteS3ObjectProvider', {
                onEventHandler: onEvent,
                logRetention: logs.RetentionDays.ONE_DAY,
            });
            const CRdeleteS3ObjectProvider = new cdk.CustomResource(this, 'CRdeleteS3ObjectProvider', {
                serviceToken: deleteS3ObjectProvider.serviceToken,
                properties: {
                    Bucket: k3sBucket.bucketName,
                },
            });
            CRdeleteS3ObjectProvider.node.addDependency(k3sBucket);
            k3sBucket.grantDelete(onEvent);
            k3sBucket.grantReadWrite(onEvent);
        }
        // control plane node Security Group      
        const k3scontrolplanesg = new ec2.SecurityGroup(this, 'k3s-controlplane-SG', {
            vpc,
            securityGroupName: 'k3s-controlplane-SG',
            allowAllOutbound: true,
        });
        k3scontrolplanesg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'SSH');
        k3scontrolplanesg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(6443), 'K3s port');
        // worker nodes Security Group      
        const k3sworkersg = new ec2.SecurityGroup(this, 'k3s-worker-SG', {
            vpc,
            securityGroupName: 'k3-worker-SG',
            allowAllOutbound: true,
        });
        // for this prototype the workers are being placed in a public subnet 
        // ideally they should land on a private subnet 
        /// also ingress traffic - ssh (bastion style) or 6443 - should come from the control plane node only 
        k3sworkersg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'SSH');
        k3sworkersg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(6443), 'K3s port');
        // check if the user requires a particular instance type for workers and control plane
        // if not, the default instance type is used 
        this.controlPlaneInstanceType = (_c = props.controlPlaneInstanceType) !== null && _c !== void 0 ? _c : DEFAULT_INSTANCE_TYPE;
        this.workerInstanceType = (_d = props.workerInstanceType) !== null && _d !== void 0 ? _d : DEFAULT_INSTANCE_TYPE;
        // create control plane node
        const k3scontrolplane = new ec2.Instance(this, 'k3s-controlplane', {
            instanceType: this.controlPlaneInstanceType,
            machineImage: new AmiProvider().amiId,
            vpc,
            vpcSubnets: {
                subnets: vpc.publicSubnets,
            },
            instanceName: 'k3s-controlplane',
            securityGroup: k3scontrolplanesg,
        });
        k3scontrolplane.addUserData(`
       #!/bin/bash
       curl -L -o k3s https://github.com/rancher/k3s/releases/download/v1.16.9%2Bk3s1/k3s-arm64
       chmod +x k3s
       ./k3s server &
       sleep 30
       ENDPOINT=$(curl http://169.254.169.254/latest/meta-data/public-hostname) 
       cp /etc/rancher/k3s/k3s.yaml /etc/rancher/k3s/kubeconfig.yaml
       sed -i s/127.0.0.1/$ENDPOINT/ /etc/rancher/k3s/kubeconfig.yaml
       aws s3 cp /var/lib/rancher/k3s/server/node-token s3://${k3sBucket.bucketName}/node-token
       aws s3 cp /etc/rancher/k3s/kubeconfig.yaml s3://${k3sBucket.bucketName}/kubeconfig.yaml
     `);
        this.endpointUri = k3scontrolplane.instancePublicIp;
        // create worker ASG
        const workerAsg = new autoscaling.AutoScalingGroup(this, 'WorkerAsg', {
            instanceType: this.workerInstanceType,
            machineImage: new AmiProvider().amiId,
            vpc,
            vpcSubnets: {
                subnetType: ec2.SubnetType.PUBLIC,
            },
            minCapacity: (_e = props.workerMinCapacity) !== null && _e !== void 0 ? _e : 3,
            spotPrice: props.spotWorkerNodes ? PriceMap.get(this.workerInstanceType.toString()) : undefined,
        });
        workerAsg.addUserData(`
       #!/bin/bash
       LOGFILE='/var/log/k3s.log'
       curl -L -o k3s https://github.com/rancher/k3s/releases/download/v1.16.13%2Bk3s1/k3s-arm64
       chmod +x k3s
       echo the bucket name is ${k3sBucket.bucketName} 
       aws s3 cp s3://${k3sBucket.bucketName}/node-token /node-token 
       (./k3s agent --server https://${k3scontrolplane.instancePrivateIp}:6443 \
       --token $(cat /node-token) 2>&1 | tee -a $LOGFILE || echo "failed" > $LOGFILE &)
    `);
        workerAsg.addSecurityGroup(k3sworkersg);
        // enable the SSM session manager
        workerAsg.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        // grant the S3 write permission to the control plane node and read permissions to the worker nodes
        k3sBucket.grantWrite(k3scontrolplane.role);
        k3sBucket.grantRead(workerAsg.role);
        // endpoint info
        new cdk.CfnOutput(this, 'Endpoint', { value: `https://${k3scontrolplane.instancePublicIp}:6443` });
        // kubeconfig.yaml path
        new cdk.CfnOutput(this, 'Kubernetes configuration file', { value: `s3://${k3sBucket.bucketName}/kubeconfig.yaml` });
        workerAsg.node.addDependency(k3scontrolplane);
    }
}
exports.Cluster = Cluster;
/**
 * The AMI provider to get the latest Amazon Linux 2 AMI for ARM64
 */
class AmiProvider {
    get amiId() {
        return ec2.MachineImage.latestAmazonLinux({
            cpuType: ec2.AmazonLinuxCpuType.ARM_64,
            generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
        });
    }
}
exports.AmiProvider = AmiProvider;
/**
 * The VPC provider to create or import the VPC
 */
class VpcProvider {
    static getOrCreate(scope) {
        const vpc = scope.node.tryGetContext('use_default_vpc') === '1' ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
            scope.node.tryGetContext('use_vpc_id') ?
                ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
                new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
        return vpc;
    }
}
exports.VpcProvider = VpcProvider;
//# sourceMappingURL=data:application/json;base64,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