# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['qudi_hira_analysis', 'qudi_hira_analysis.fitmethods']

package_data = \
{'': ['*']}

install_requires = \
['lmfit>=1.1.0,<2.0.0',
 'matplotlib>=3.6.2,<4.0.0',
 'notebook>=6.5.3,<7.0.0',
 'numpy>=1.24.0,<2.0.0',
 'openpyxl>=3.0.10,<4.0.0',
 'pandas>=1.5.2,<2.0.0',
 'tqdm>=4.64.1,<5.0.0',
 'xlrd>=2.0.1,<3.0.0']

setup_kwargs = {
    'name': 'qudi-hira-analysis',
    'version': '1.3.3',
    'description': 'A Python toolkit to analzye photon timetrace data from qubit sensors',
    'long_description': '[![DOI](https://zenodo.org/badge/288670453.svg)](https://zenodo.org/badge/latestdoi/288670453)\n[![PyPi version](https://img.shields.io/pypi/v/qudi-hira-analysis)](https://pypi.python.org/pypi/qudi-hira-analysis/)\n[![Python 3.10](https://img.shields.io/badge/python-3.10-orange.svg)](https://www.python.org/downloads/release/python-3100//)\n[![Python 3.11](https://img.shields.io/badge/python-3.11-orange.svg)](https://www.python.org/downloads/release/python-3110//)\n\n# Qudi Hira Analysis\n\nThis toolkit automates a large portion of the work surrounding data analysis on quantum sensing experiments where the\nprimary raw data extracted is photon counts.\n\nThe high level interface is abstracted, and provides a set of functions to automate data import, handling and analysis.\nIt is designed to be exposed through Jupyter Notebooks, although the abstract interface allows it to be integrated into\nlarger, more general frameworks as well (with only some pain). Using the toolkit itself should only require a\nbeginner-level understanding of Python.\n\nIt also aims to improve transparency and reproducibility in experimental data analysis. In an ideal scenario,\ntwo lines of code are sufficient to recreate all output data.\n\nPython offers some very handy features like dataclasses, which are heavily used by this toolkit. Dataclasses offer a\nfull OOP (object-oriented programming) experience while analyzing complex data sets. They provide a solid and\ntransparent structure to the data to\nreduce errors arising from data fragmentation. This generally comes at a large performance cost, but this is (largely)\nsidestepped by lazy loading data and storing metadata instead wherever possible.\n\n## Installation\n\n```bash\npip install qudi-hira-analysis\n```\n\n## Citation\n\nIf you are publishing scientific results, you can cite this work as:  https://doi.org/10.5281/zenodo.7604670\n\n## Schema\n\nThe visual structure of the toolkit is shown in the schema below. It largely consists of three portions:\n\n- `IOHandler` assumes a central store of raw data, which is never modified (read-only)\n- `DataHandler` automates the extraction of large amounts of data from the `IOHandler` interface\n- `AnalysisLogic` contains a set of automated fitting routines using `lmfit` internally (built on top of fitting\n  routines from the [qudi](https://github.com/Ulm-IQO/qudi) project)\n\nThis license of this project is located in the top level folder under `LICENSE`. Some specific files contain their\nindividual licenses in the file header docstring.\n\n```mermaid\nflowchart TD;\n    IOHandler<-- Handle all IO operations -->DataLoader;\n    DataLoader<-- Map IO callables to data -->DataHandler;\n    DataHandler-- Structure extracted data -->MeasurementDataclass;\n    MeasurementDataclass-- Plot fitted data --> Plot[Visualize data and add context in JupyterLab];\n    Plot-- Save plotted data --> DataHandler;\n    style MeasurementDataclass fill:#bbf,stroke:#f66,stroke-width:2px,color:#fff,stroke-dasharray: 5 5\n```\n\n### Measurement Dataclass\n\n```mermaid\nflowchart LR;\n    subgraph Standard Data\n        MeasurementDataclass-->filepath1[filepath: Path];\n        MeasurementDataclass-->data1[data: DataFrame];\n        MeasurementDataclass-->params1[params: dict];\n        MeasurementDataclass-->timestamp1[timestamp: datetime];\n        MeasurementDataclass-- analysis --oAnalysisLogic;\n    end\n    subgraph Pulsed Data\n        MeasurementDataclass-- pulsed --oPulsedMeasurementDataclass;\n        PulsedMeasurementDataclass-- measurement --oPulsedMeasurement;\n        PulsedMeasurement--> filepath2[filepath: Path];\n        PulsedMeasurement--> data2[data: DataFrame];\n        PulsedMeasurement--> params2[params: dict];\n        PulsedMeasurementDataclass-- laser_pulses --oLaserPulses; \n        LaserPulses--> filepath3[filepath: Path];\n        LaserPulses--> data3[data: DataFrame];\n        LaserPulses--> params3[params: dict];\n        PulsedMeasurementDataclass-- timetrace --oRawTimetrace;\n        RawTimetrace--> filepath4[filepath: Path];\n        RawTimetrace--> data4[data: DataFrame];\n        RawTimetrace--> params4[params: dict];\n    end\n```\n\n### Supports common fitting routines\n\nFit routines included in `AnalysisLogic`\n\n| Dimension | Fit                           |\n|-----------|-------------------------------|\n| 1d        | decayexponential              |\n|           | biexponential                 |\n|           | decayexponentialstretched     |\n|           | gaussian                      |\n|           | gaussiandouble                |\n|           | gaussianlinearoffset          |\n|           | hyperbolicsaturation          |\n|           | linear                        |\n|           | lorentzian                    |\n|           | lorentziandouble              |\n|           | lorentziantriple              |\n|           | sine                          |\n|           | sinedouble                    |\n|           | sinedoublewithexpdecay        |\n|           | sinedoublewithtwoexpdecay     |\n|           | sineexponentialdecay          |\n|           | sinestretchedexponentialdecay |\n|           | sinetriple                    |\n|           | sinetriplewithexpdecay        |\n|           | sinetriplewiththreeexpdecay   |\n| 2d        | twoDgaussian                  |\n\n### Inbuilt measurement tree visualizer\n\n```ipython\n>>> tip_2S6 = DataHandler(data_folder="C:\\\\Data", figure_folder="C:\\\\QudiHiraAnalysis",\n                      measurement_folder="20220621_FR0612-F2-2S6_uhv")\n>>> tip_2S6.data_folder_tree()\n\n# Output\n├── 20211116_NetworkAnalysis_SampleIn_UpperPin.csv\n├── 20211116_NetworkAnalysis_SampleOut_UpperPin.csv\n├── 20211116_NetworkAnalysis_TipIn_LowerPin.csv\n├── 20211116_NetworkAnalysis_TipIn_UpperPin.csv\n├── 20211116_NetworkAnalysis_TipOut_LowerPin.csv\n├── 20211116_NetworkAnalysis_TipOut_UpperPin.csv\n├── ContactTestingMeasurementHead\n│   ├── C2_Reference.txt\n│   ├── C2_SampleLowerPin.txt\n│   ├── C2_SampleUpperPin.txt\n│   ├── C2_TipLowerPin.txt\n│   └── C2_TipUpperPin.txt\n├── Sample_MW_Pin_comparision.png\n├── Tip_MW_Pin_comparision.png\n└── Tip_Sample_MW_Pin_comparision.png\n```\n\n### Automated data extraction\n\n#### Example 1: Extract, fit and plot all Rabi measurements\n\n```python\nfrom pathlib import Path\nimport matplotlib.pyplot as plt\nimport seaborn as sns\n\nfrom qudi_hira_analysis import DataHandler\n\nnv1 = DataHandler(\n  data_folder=Path("C:\\\\", "Data"),\n  figure_folder=Path("C:\\\\", "QudiHiraAnalysis"),\n  measurement_folder=Path("20230101_NV1")\n)\n\nrabi_measurements = nv1.load_measurements(measurement_str="rabi", qudi=True, pulsed=True)\n\nfig, ax = plt.subplots()\n\nfor rabi in rabi_measurements:\n  sns.lineplot(data=rabi.data, x="Controlled variable(s)", y="Signal", ax=ax)\n  fit_x, fit_y, result = rabi.analysis.fit(\n    x="Controlled variable(s)", y="Signal",\n    data=rabi.data,\n    fit_function=rabi_measurements.sineexponentialdecay\n  )\n  sns.lineplot(x=fit_x, y=fit_y, ax=ax)\n\nnv1.save_figures(filepath="rabi_variation", fig=fig)\n```\n\n#### Example 2: Combine all temperature data, plot and save\n\n```python\nfrom pathlib import Path\n\nimport matplotlib.pyplot as plt\nimport pandas as pd\nimport seaborn as sns\n\nfrom qudi_hira_analysis import DataHandler\n\nnv1 = DataHandler(\n  data_folder=Path("C:\\\\", "Data"),\n  figure_folder=Path("C:\\\\", "QudiHiraAnalysis"),\n  measurement_folder=Path("20230101_NV1"),\n  copy_measurement_folder_structure=False\n)\n\ntemperature_measurements = nv1.load_measurements(measurement_str="temperature-monitoring")\n\ndft = pd.concat([t.data for t in temperature_measurements.values()])\n\nfig, ax = plt.subplots()\nsns.lineplot(data=dft, x="Time", y="Temperature", ax=ax)\nnv1.save_figures(filepath="temperature-monitoring", fig=fig)\n```\n\n## Build\n\n### Prerequisites\n\nLatest version of:\n\n- [Poetry](https://python-poetry.org) (recommended) or [conda](https://docs.conda.io/en/latest/miniconda.html) package\n  manager\n- [git](https://git-scm.com/downloads) version control system\n\n### Clone the repository\n\n```shell\ngit clone https://github.com/dineshpinto/qudi-hira-analysis.git\n```\n\n### Installing dependencies with Poetry\n\n```bash\npoetry install\n```\n\n#### Add Poetry environment to Jupyter kernel\n\n```bash\npoetry run python -m ipykernel install --user --name=qudi-hira-analysis\n```\n\n### OR installing dependencies with conda\n\n#### Creating the conda environment\n\n```shell\nconda env create -f tools/conda-env-xx.yml\n```\n\nwhere `xx` is either `win10`, `osx-intel` or `osx-apple-silicon`.\n\n#### Activate conda environment\n\n```shell\nconda activate qudi-hira-analysis\n```\n\n#### Add conda environment to Jupyter kernel\n\n```shell\npython -m ipykernel install --user --name=qudi-hira-analysis\n```\n\n### Start the analysis\n\n#### If installed with Poetry\n\n```shell\npoetry run jupyter lab\n```\n\n#### OR with conda\n\n```shell\njupyter lab\n```\n\nDon\'t forget to switch to the `qudi-hira-analysis` kernel in JupyterLab.\n\n## Makefile\n\nThe Makefile located in `notebooks/` is configured to generate a variety of outputs:\n\n+ `make pdf` : Converts all notebooks to PDF (requires LaTeX backend)\n+ `make html`: Converts all notebooks to HTML\n+ `make py`  : Converts all notebooks to Python (can be useful for VCS)\n+ `make all` : Sequentially runs all the notebooks in folder\n\nTo use the `make` command on Windows you can install [Chocolatey](https://chocolatey.org/install), then\ninstall make with `choco install make`\n',
    'author': 'dineshpinto',
    'author_email': 'annual.fallout_0z@icloud.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/dineshpinto/qudi-hira-analysis',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
