"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const METRICKEY_SYMBOL = Symbol('@aws-cdk/aws-cloudwatch.MetricKey');
/**
 * Return a unique string representation for this metric.
 *
 * Can be used to determine as a hash key to determine if 2 Metric objects
 * represent the same metric. Excludes rendering properties.
 */
function metricKey(metric) {
    // Cache on the object itself. This is safe because Metric objects are immutable.
    if (metric.hasOwnProperty(METRICKEY_SYMBOL)) {
        return metric[METRICKEY_SYMBOL];
    }
    const parts = new Array();
    const conf = metric.toMetricConfig();
    if (conf.mathExpression) {
        parts.push(conf.mathExpression.expression);
        for (const id of Object.keys(conf.mathExpression.usingMetrics).sort()) {
            parts.push(id);
            parts.push(metricKey(conf.mathExpression.usingMetrics[id]));
        }
    }
    if (conf.metricStat) {
        parts.push(conf.metricStat.namespace);
        parts.push(conf.metricStat.metricName);
        for (const dim of conf.metricStat.dimensions || []) {
            parts.push(dim.name);
            parts.push(dim.value);
        }
        if (conf.metricStat.statistic) {
            parts.push(conf.metricStat.statistic);
        }
        if (conf.metricStat.period) {
            parts.push(`${conf.metricStat.period.toSeconds()}`);
        }
        if (conf.metricStat.region) {
            parts.push(conf.metricStat.region);
        }
        if (conf.metricStat.account) {
            parts.push(conf.metricStat.account);
        }
    }
    const ret = parts.join('|');
    Object.defineProperty(metric, METRICKEY_SYMBOL, { value: ret });
    return ret;
}
exports.metricKey = metricKey;
/**
 * Return the period of a metric
 *
 * For a stat metric, return the immediate period.
 *
 * For an expression metric, all metrics used in it have been made to have the
 * same period, so we return the period of the first inner metric.
 */
function metricPeriod(metric) {
    return dispatchMetric(metric, {
        withStat(stat) {
            return stat.period;
        },
        withExpression() {
            return metric.period || core_1.Duration.minutes(5);
        },
    });
}
exports.metricPeriod = metricPeriod;
/**
 * Given a metric object, inspect it and call the correct function for the type of output
 *
 * In addition to the metric object itself, takes a callback object with two
 * methods, to be invoked for the particular type of metric.
 *
 * If the metric represent a metric query (nominally generated through an
 * instantiation of `Metric` but can be generated by any class that implements
 * `IMetric`) a particular field in its `toMetricConfig()` output will be set
 * (to wit, `metricStat`) and the `withStat()` callback will be called with
 * that object.
 *
 * If the metric represents an expression (usually by instantiating `MathExpression`
 * but users can implement `IMetric` arbitrarily) the `mathExpression` field
 * will be set in the object returned from `toMetricConfig` and the callback
 * called `withExpression` will be applied to that object.
 *
 * Will return the values returned by the callbacks.
 *
 * To be used as such:
 *
 * ```ts
 * const ret = dispatchMetric(someMetric, {
 *   withStat(stat) {
 *     // do something with stat
 *     return 1;
 *   },
 *   withExpression(expr) {
 *     // do something with expr
 *     return 2;
 *   },
 * });
 * ```
 *
 * This function encapsulates some type analysis that would otherwise have to be
 * repeated in all places where code needs to make a distinction on the type
 * of metric object that is being passed.
 */
// tslint:disable-next-line:max-line-length
function dispatchMetric(metric, fns) {
    const conf = metric.toMetricConfig();
    if (conf.metricStat && conf.mathExpression) {
        throw new Error('Metric object must not produce both \'metricStat\' and \'mathExpression\'');
    }
    else if (conf.metricStat) {
        return fns.withStat(conf.metricStat, conf);
    }
    else if (conf.mathExpression) {
        return fns.withExpression(conf.mathExpression, conf);
    }
    else {
        throw new Error('Metric object must have either \'metricStat\' or \'mathExpression\'');
    }
}
exports.dispatchMetric = dispatchMetric;
//# sourceMappingURL=data:application/json;base64,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