# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['aredis_om',
 'aredis_om.model',
 'aredis_om.model.cli',
 'aredis_om.model.migrations']

package_data = \
{'': ['*']}

install_requires = \
['aioredis>=2.0.0,<3.0.0',
 'cleo==1.0.0a4',
 'click>=8.0.1,<9.0.0',
 'hiredis>=2.0.0,<3.0.0',
 'pptree>=3.1,<4.0',
 'pydantic>=1.8.2,<2.0.0',
 'python-ulid>=1.0.3,<2.0.0',
 'redis>=3.5.3,<5.0.0',
 'six>=1.16.0,<2.0.0',
 'types-redis>=3.5.9,<5.0.0',
 'types-six>=1.16.1,<2.0.0',
 'typing-extensions>=4.0.0,<5.0.0']

entry_points = \
{'console_scripts': ['migrate = redis_om.model.cli.migrate:migrate']}

setup_kwargs = {
    'name': 'redis-om',
    'version': '0.0.27',
    'description': 'Objecting mapping, and more, for Redis.',
    'long_description': '<div align="center">\n  <br/>\n  <br/>\n  <img width="360" src="https://raw.githubusercontent.com/redis-developer/redis-om-python/main/images/logo.svg?token=AAAXXHUYL6RHPESRRAMBJOLBSVQXE" alt="Redis OM" />\n  <br/>\n  <br/>\n</div>\n\n<p align="center">\n    <p align="center">\n        Object mapping, and more, for Redis and Python\n    </p>\n</p>\n\n---\n\n[![Version][version-svg]][package-url]\n[![License][license-image]][license-url]\n[![Build Status][ci-svg]][ci-url]\n\n**Redis OM Python** makes it easy to model Redis data in your Python applications.\n\n[Redis OM .NET](https://github.com/redis/redis-om-dotnet) | [Redis OM Node.js](https://github.com/redis/redis-om-node) | [Redis OM Spring](https://github.com/redis/redis-om-spring) | **Redis OM Python**\n\n<details>\n  <summary><strong>Table of contents</strong></summary>\n\nspan\n\n<!-- DON\'T EDIT THIS SECTION, INSTEAD RE-RUN doctoc TO UPDATE -->\n\n- [💡 Why Redis OM?](#-why-redis-om)\n- [📇 Modeling Your Data](#-modeling-your-data)\n- [✓ Validating Data With Your Model](#-validating-data-with-your-model)\n- [🔎 Rich Queries and Embedded Models](#-rich-queries-and-embedded-models)\n  - [Querying](#querying)\n  - [Embedded Models](#embedded-models)\n- [Calling Other Redis Commands](#calling-other-redis-commands)\n- [💻 Installation](#-installation)\n- [📚 Documentation](#-documentation)\n- [⛏️ Troubleshooting](#️-troubleshooting)\n- [✨ So How Do You Get RediSearch and RedisJSON?](#-so-how-do-you-get-redisearch-and-redisjson)\n- [❤️ Contributing](#️-contributing)\n- [📝 License](#-license)\n\n<!-- END doctoc generated TOC please keep comment here to allow auto update -->\n\n</details>\n\n## 💡 Why Redis OM?\n\nRedis OM provides high-level abstractions that make it easy to model and query data in Redis with modern Python applications.\n\nThis **preview** release contains the following features:\n\n* Declarative object mapping for Redis objects\n* Declarative secondary-index generation\n* Fluent APIs for querying Redis\n\n## 📇 Modeling Your Data\n\nRedis OM contains powerful declarative models that give you data validation, serialization, and persistence to Redis.\n\nCheck out this example of modeling customer data with Redis OM. First, we create a `Customer` model:\n\n```python\nimport datetime\nfrom typing import Optional\n\nfrom pydantic import EmailStr\n\nfrom redis_om import HashModel\n\n\nclass Customer(HashModel):\n    first_name: str\n    last_name: str\n    email: EmailStr\n    join_date: datetime.date\n    age: int\n    bio: Optional[str]\n```\n\nNow that we have a `Customer` model, let\'s use it to save customer data to Redis.\n\n```python\nimport datetime\nfrom typing import Optional\n\nfrom pydantic import EmailStr\n\nfrom redis_om import HashModel\n\n\nclass Customer(HashModel):\n    first_name: str\n    last_name: str\n    email: EmailStr\n    join_date: datetime.date\n    age: int\n    bio: Optional[str]\n\n\n# First, we create a new `Customer` object:\nandrew = Customer(\n    first_name="Andrew",\n    last_name="Brookins",\n    email="andrew.brookins@example.com",\n    join_date=datetime.date.today(),\n    age=38,\n    bio="Python developer, works at Redis, Inc."\n)\n\n# The model generates a globally unique primary key automatically\n# without needing to talk to Redis.\nprint(andrew.pk)\n# > "01FJM6PH661HCNNRC884H6K30C"\n\n# We can save the model to Redis by calling `save()`:\nandrew.save()\n\n# Expire the model after 2 mins (120 seconds)\nandrew.expire(120)\n\n# To retrieve this customer with its primary key, we use `Customer.get()`:\nassert Customer.get(andrew.pk) == andrew\n```\n\n**Ready to learn more?** Check out the [getting started](docs/getting_started.md) guide.\n\nOr, continue reading to see how Redis OM makes data validation a snap.\n\n## ✓ Validating Data With Your Model\n\nRedis OM uses [Pydantic][pydantic-url] to validate data based on the type annotations you assign to fields in a model class.\n\nThis validation ensures that fields like `first_name`, which the `Customer` model marked as a `str`, are always strings. **But every Redis OM model is also a Pydantic model**, so you can use Pydantic validators like `EmailStr`, `Pattern`, and many more for complex validations!\n\nFor example, because we used the `EmailStr` type for the `email` field, we\'ll get a validation error if we try to create a `Customer` with an invalid email address:\n\n```python\nimport datetime\nfrom typing import Optional\n\nfrom pydantic import EmailStr, ValidationError\n\nfrom redis_om import HashModel\n\n\nclass Customer(HashModel):\n    first_name: str\n    last_name: str\n    email: EmailStr\n    join_date: datetime.date\n    age: int\n    bio: Optional[str]\n\n\ntry:\n    Customer(\n        first_name="Andrew",\n        last_name="Brookins",\n        email="Not an email address!",\n        join_date=datetime.date.today(),\n        age=38,\n        bio="Python developer, works at Redis, Inc."\n    )\nexcept ValidationError as e:\n    print(e)\n    """\n    pydantic.error_wrappers.ValidationError: 1 validation error for Customer\n     email\n       value is not a valid email address (type=value_error.email)\n    """\n```\n\n**Any existing Pydantic validator should work** as a drop-in type annotation with a Redis OM model. You can also write arbitrarily complex custom validations!\n\nTo learn more, see the [documentation on data validation](docs/validation.md).\n\n## 🔎 Rich Queries and Embedded Models\n\nData modeling, validation, and saving models to Redis all work regardless of how you run Redis.\n\nNext, we\'ll show you the **rich query expressions** and **embedded models** Redis OM provides when the [RediSearch][redisearch-url] and [RedisJSON][redis-json-url] modules are installed in your Redis deployment, or you\'re using [Redis Enterprise][redis-enterprise-url].\n\n**TIP**: *Wait, what\'s a Redis module?* If you aren\'t familiar with Redis modules, review the [So, How Do You Get RediSearch and RedisJSON?](#-so-how-do-you-get-redisearch-and-redisjson) section of this README.\n\n### Querying\n\nRedis OM comes with a rich query language that allows you to query Redis with Python expressions.\n\nTo show how this works, we\'ll make a small change to the `Customer` model we defined earlier. We\'ll add `Field(index=True)` to tell Redis OM that we want to index the `last_name` and `age` fields:\n\n```python\nimport datetime\nfrom typing import Optional\n\nfrom pydantic import EmailStr\n\nfrom redis_om import (\n    Field,\n    HashModel,\n    Migrator\n)\n\n                 \nclass Customer(HashModel):\n    first_name: str\n    last_name: str = Field(index=True)\n    email: EmailStr\n    join_date: datetime.date\n    age: int = Field(index=True)\n    bio: Optional[str]\n\n\n# Now, if we use this model with a Redis deployment that has the\n# RediSearch module installed, we can run queries like the following.\n\n# Before running queries, we need to run migrations to set up the\n# indexes that Redis OM will use. You can also use the `migrate`\n# CLI tool for this!\nMigrator().run()\n\n# Find all customers with the last name "Brookins"\nCustomer.find(Customer.last_name == "Brookins").all()\n\n# Find all customers that do NOT have the last name "Brookins"\nCustomer.find(Customer.last_name != "Brookins").all()\n\n# Find all customers whose last name is "Brookins" OR whose age is \n# 100 AND whose last name is "Smith"\nCustomer.find((Customer.last_name == "Brookins") | (\n        Customer.age == 100\n) & (Customer.last_name == "Smith")).all()\n```\n\nThese queries -- and more! -- are possible because **Redis OM manages indexes for you automatically**.\n\nQuerying with this index features a rich expression syntax inspired by the Django ORM, SQLAlchemy, and Peewee. We think you\'ll enjoy it!\n\n**Note:** Indexing only works for data stored in Redis logical database 0.  If you are using a different database number when connecting to Redis, you can expect the code to raise a `MigrationError` when you run the migrator.\n\n### Embedded Models\n\nRedis OM can store and query **nested models** like any document database, with the speed and power you get from Redis. Let\'s see how this works.\n\nIn the next example, we\'ll define a new `Address` model and embed it within the `Customer` model.\n\n```python\nimport datetime\nfrom typing import Optional\n\nfrom redis_om import (\n    EmbeddedJsonModel,\n    JsonModel,\n    Field,\n    Migrator,\n)\n\n\nclass Address(EmbeddedJsonModel):\n    address_line_1: str\n    address_line_2: Optional[str]\n    city: str = Field(index=True)\n    state: str = Field(index=True)\n    country: str\n    postal_code: str = Field(index=True)\n\n\nclass Customer(JsonModel):\n    first_name: str = Field(index=True)\n    last_name: str = Field(index=True)\n    email: str = Field(index=True)\n    join_date: datetime.date\n    age: int = Field(index=True)\n    bio: Optional[str] = Field(index=True, full_text_search=True,\n                               default="")\n\n    # Creates an embedded model.\n    address: Address\n\n\n# With these two models and a Redis deployment with the RedisJSON \n# module installed, we can run queries like the following.\n\n# Before running queries, we need to run migrations to set up the\n# indexes that Redis OM will use. You can also use the `migrate`\n# CLI tool for this!\nMigrator().run()\n\n# Find all customers who live in San Antonio, TX\nCustomer.find(Customer.address.city == "San Antonio",\n              Customer.address.state == "TX")\n```\n\n## Calling Other Redis Commands\n\nSometimes you\'ll need to run a Redis command directly.  Redis OM supports this through the `db` method on your model\'s class.  This returns a connected Redis client instance which exposes a function named for each Redis command.  For example, let\'s perform some basic set operations:\n\n```python\nfrom redis_om import HashModel\n\nclass Demo(HashModel):\n    some_field: str\n\nredis_conn = Demo.db()\n\nredis_conn.sadd("myset", "a", "b", "c", "d")\n\n# Prints False\nprint(redis_conn.sismember("myset", "e"))\n\n# Prints True\nprint(redis_conn.sismember("myset", "b"))\n```\n\nThe parameters expected by each command function are those documented on the command\'s page on [redis.io](https://redis.io/commands/).\n\nIf you don\'t want to get a Redis connection from a model class, you can also use `get_redis_connection`:\n\n```python\nfrom redis_om import get_redis_connection\n\nredis_conn = get_redis_connection()\nredis_conn.set("hello", "world")\n```\n\n## 💻 Installation\n\nInstallation is simple with `pip`, Poetry, or Pipenv.\n\n```sh\n# With pip\n$ pip install redis-om\n\n# Or, using Poetry\n$ poetry add redis-om\n```\n\n## 📚 Documentation\n\nThe Redis OM documentation is available [here](docs/index.md).\n\n## ⛏️ Troubleshooting\n\nIf you run into trouble or have any questions, we\'re here to help!\n\nHit us up on the [Redis Discord Server](http://discord.gg/redis) or [open an issue on GitHub](https://github.com/redis-developer/redis-om-python/issues/new).\n\n## ✨ So How Do You Get RediSearch and RedisJSON?\n\nSome advanced features of Redis OM rely on core features from two source available Redis modules: [RediSearch][redisearch-url] and [RedisJSON][redis-json-url].\n\nYou can run these modules in your self-hosted Redis deployment, or you can use [Redis Enterprise][redis-enterprise-url], which includes both modules.\n\nTo learn more, read [our documentation](docs/redis_modules.md).\n\n## ❤️ Contributing\n\nWe\'d love your contributions!\n\n**Bug reports** are especially helpful at this stage of the project. [You can open a bug report on GitHub](https://github.com/redis/redis-om-python/issues/new).\n\nYou can also **contribute documentation** -- or just let us know if something needs more detail. [Open an issue on GitHub](https://github.com/redis/redis-om-python/issues/new) to get started.\n\n## 📝 License\n\nRedis OM uses the [MIT license][license-url].\n\n<!-- Badges -->\n\n[version-svg]: https://img.shields.io/pypi/v/redis-om?style=flat-square\n[package-url]: https://pypi.org/project/redis-om/\n[ci-svg]: https://img.shields.io/github/workflow/status/redis/redis-om-python/CI?style=flat-square\n[ci-url]: https://github.com/redis/redis-om-python/actions/workflows/CI.yml\n[license-image]: https://img.shields.io/badge/license-mit-green.svg?style=flat-square\n[license-url]: LICENSE\n<!-- Links -->\n\n[redis-om-website]: https://developer.redis.com\n[redis-om-js]: https://github.com/redis-om/redis-om-js\n[redis-om-dotnet]: https://github.com/redis-om/redis-om-dotnet\n[redis-om-spring]: https://github.com/redis-om/redis-om-spring\n[redisearch-url]: https://oss.redis.com/redisearch/\n[redis-json-url]: https://oss.redis.com/redisjson/\n[pydantic-url]: https://github.com/samuelcolvin/pydantic\n[ulid-url]: https://github.com/ulid/spec\n[redis-enterprise-url]: https://redis.com/try-free/\n',
    'author': 'Andrew Brookins',
    'author_email': 'andrew.brookins@redis.com',
    'maintainer': 'Andrew Brookins',
    'maintainer_email': 'andrew.brookins@redis.com',
    'url': 'https://github.com/redis-developer/redis-om-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
