# This file is part of tad-dftd4.
#
# SPDX-Identifier: LGPL-3.0
# Copyright (C) 2022 Marvin Friede
#
# tad-dftd4 is free software: you can redistribute it and/or modify it under
# the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# tad-dftd4 is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with tad-dftd4. If not, see <https://www.gnu.org/licenses/>.
"""
Data for testing D4 coordination number (taken from D4 testsuite).
"""
from __future__ import annotations

import torch

from tad_dftd4._typing import Molecule, Tensor, TypedDict

from ..molecules import merge_nested_dicts, mols
from ..utils import reshape_fortran


class Refs(TypedDict):
    """Format of reference values."""

    gw: Tensor
    """
    Gaussian weights. Shape must be `(nrefs, natoms)`, which we have to
    transpose for comparison to the Python-calculated tensor later.
    """

    c6: Tensor
    """
    Reference C6 coefficients. Shape is `(natoms, natoms)`.
    """


class Record(Molecule, Refs):
    """Store for molecular information and reference values."""


refs: dict[str, Refs] = {
    "LiH": {  # CN, q
        "gw": reshape_fortran(
            torch.tensor(
                [
                    +1.8699287753787968e-02,
                    +9.7889292523075033e-01,
                    +1.8718044551687104e-37,
                    +7.9608926855620182e-02,
                    +3.5225968112617356e00,
                    +0.0000000000000000e00,
                ],
                dtype=torch.double,
            ),
            (3, 2),
        ),
        "c6": torch.tensor(
            [
                +4.1059628873073926e01,
                +2.9129176877403175e01,
                +2.9129176877403175e01,
                +4.0408036338319796e01,
            ],
            dtype=torch.double,
        ).reshape(2, 2),
    },
    "SiH4": {  # CN, q
        "gw": reshape_fortran(
            torch.tensor(
                [
                    +1.9334597142903089e-35,
                    +1.4946886024790536e-20,
                    +2.0260041599668397e-09,
                    +1.3096099611387583e-02,
                    +9.8937148831611499e-01,
                    +6.3922186078456374e-03,
                    +9.3633202240072710e-01,
                    +0.0000000000000000e00,
                    +0.0000000000000000e00,
                    +0.0000000000000000e00,
                    +6.3922186078456374e-03,
                    +9.3633202240072710e-01,
                    +0.0000000000000000e00,
                    +0.0000000000000000e00,
                    +0.0000000000000000e00,
                    +6.3922186078456374e-03,
                    +9.3633202240072710e-01,
                    +0.0000000000000000e00,
                    +0.0000000000000000e00,
                    +0.0000000000000000e00,
                    +6.3922186078456374e-03,
                    +9.3633202240072710e-01,
                    +0.0000000000000000e00,
                    +0.0000000000000000e00,
                    +0.0000000000000000e00,
                ],
                dtype=torch.double,
            ),
            (5, 5),
        ),
        "c6": torch.tensor(
            [
                +1.5985885538475173e02,
                +2.0584140554801550e01,
                +2.0584140554801550e01,
                +2.0584140554801550e01,
                +2.0584140554801550e01,
                +2.0584140554801550e01,
                +2.7202385043230000e00,
                +2.7202385043230000e00,
                +2.7202385043230000e00,
                +2.7202385043230000e00,
                +2.0584140554801550e01,
                +2.7202385043230000e00,
                +2.7202385043230000e00,
                +2.7202385043230000e00,
                +2.7202385043230000e00,
                +2.0584140554801550e01,
                +2.7202385043230000e00,
                +2.7202385043230000e00,
                +2.7202385043230000e00,
                +2.7202385043230000e00,
                +2.0584140554801550e01,
                +2.7202385043230000e00,
                +2.7202385043230000e00,
                +2.7202385043230000e00,
                +2.7202385043230000e00,
            ],
            dtype=torch.double,
        ).reshape(5, 5),
    },
    "MB16_43_01": {  # CN
        "gw": reshape_fortran(
            torch.tensor(
                [
                    1.2088234202205e-11,
                    1.0375924114812e00,
                    2.4445602531076e-09,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    5.5306460372092e-03,
                    9.9446935396279e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    3.1404764970412e-06,
                    6.3956931077352e-02,
                    4.7846317290481e-01,
                    2.9468368515225e-01,
                    0.0000000000000e00,
                    7.4310920077799e-05,
                    9.9992568907992e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    4.4395780891660e-02,
                    8.5383545436310e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    1.1950286131613e-02,
                    9.8804971386839e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    5.6992908407601e-05,
                    9.9994300709159e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    5.6167960886507e-07,
                    2.9078967459651e-02,
                    6.1859560324107e-01,
                    1.5104287585421e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    4.9771287942905e-14,
                    1.3962072360858e-04,
                    6.8484995485525e-01,
                    1.6516463249478e-12,
                    1.9880162618842e-02,
                    9.8011983738116e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    1.1966389405645e-02,
                    9.8803361059436e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    1.2874045548253e-05,
                    9.7726740456727e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    4.1491138092838e-06,
                    8.6768202830170e-01,
                    0.0000000000000e00,
                    4.3169267440676e-12,
                    3.6465051654232e-04,
                    8.8903503100593e-01,
                    1.2720379452861e-05,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    1.4066048552144e-05,
                    6.8494819902320e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    1.9574616732692e-10,
                    1.0064486114128e00,
                    0.0000000000000e00,
                ],
                dtype=torch.double,
            ),
            (5, 16),
        ),
        "c6": torch.tensor(
            [
                +9.2195970681311991e01,
                +9.2915439274313609e00,
                +2.8146607387192226e01,
                +1.0775977508514995e01,
                +1.8352938120680449e01,
                +6.9256661617462303e00,
                +1.0300621112824674e01,
                +3.0484610765349100e01,
                +3.3918238356514138e01,
                +6.4766138204142480e00,
                +6.4922519019669256e00,
                +6.9282713428363266e01,
                +4.8619305479898145e01,
                +4.3364114441872893e01,
                +3.7773137469159217e01,
                +9.4819536752314136e01,
                +9.2915439274313609e00,
                +1.7162047349009455e00,
                +5.7990721126563534e00,
                +1.9925588226254198e00,
                +4.1529084941730767e00,
                +1.2775865350597853e00,
                +1.9046684821850091e00,
                +6.2754724806093298e00,
                +6.5446665006430811e00,
                +1.1928911659344870e00,
                +1.1976302445456630e00,
                +1.3053810854181670e01,
                +8.8678073889297409e00,
                +7.7888498604059606e00,
                +7.2884836870559822e00,
                +1.4723882690435001e01,
                +2.8146607387192226e01,
                +5.7990721126563534e00,
                +2.0322446513248725e01,
                +6.7343200224524979e00,
                +1.4915049509329648e01,
                +4.3158943984974778e00,
                +6.4372783937555162e00,
                +2.1984342075678786e01,
                +2.2540760424766276e01,
                +4.0285410771690495e00,
                +4.0457865525221370e00,
                +4.4440038553551688e01,
                +2.9926910685255862e01,
                +2.6194837765153459e01,
                +2.5102571723162029e01,
                +4.8059235287218819e01,
                +1.0775977508514995e01,
                +1.9925588226254198e00,
                +6.7343200224524979e00,
                +2.3134172161508406e00,
                +4.8234212783344121e00,
                +1.4833085389308831e00,
                +2.2113740543864617e00,
                +7.2875397374296860e00,
                +7.5993054885377020e00,
                +1.3849717326666449e00,
                +1.3904773652700402e00,
                +1.5156457827338222e01,
                +1.0295584693450651e01,
                +9.0426690304625250e00,
                +8.4629849467909484e00,
                +1.7089770111859078e01,
                +1.8352938120680449e01,
                +4.1529084941730767e00,
                +1.4915049509329648e01,
                +4.8234212783344121e00,
                +1.1121625148998874e01,
                +3.0901931077520812e00,
                +4.6106690588499681e00,
                +1.6131060231215859e01,
                +1.6350726163094549e01,
                +2.8838036045730786e00,
                +2.8967937758905151e00,
                +3.1989891272947684e01,
                +2.1410657159900445e01,
                +1.8693996640429365e01,
                +1.8209024833681802e01,
                +3.3532590882087042e01,
                +6.9256661617462303e00,
                +1.2775865350597853e00,
                +4.3158943984974778e00,
                +1.4833085389308831e00,
                +3.0901931077520812e00,
                +9.5107023428020221e-01,
                +1.4178808540351600e00,
                +4.6704592716581299e00,
                +4.8714353281692251e00,
                +8.8802328302553557e-01,
                +8.9154859790635033e-01,
                +9.7171097068987322e00,
                +6.6015513035883231e00,
                +5.7985108827431544e00,
                +5.4250857435598849e00,
                +1.0964613701101552e01,
                +1.0300621112824674e01,
                +1.9046684821850091e00,
                +6.4372783937555162e00,
                +2.2113740543864617e00,
                +4.6106690588499681e00,
                +1.4178808540351600e00,
                +2.1138319427900139e00,
                +6.9660963076141282e00,
                +7.2641079346924871e00,
                +1.3238816038516765e00,
                +1.3291443974122232e00,
                +1.4487919292348902e01,
                +9.8414533114041927e00,
                +8.6438021973771111e00,
                +8.0896913798057266e00,
                +1.6335937257210038e01,
                +3.0484610765349100e01,
                +6.2754724806093298e00,
                +2.1984342075678786e01,
                +7.2875397374296860e00,
                +1.6131060231215859e01,
                +4.6704592716581299e00,
                +6.9660963076141282e00,
                +2.3782238131624815e01,
                +2.4388022059359354e01,
                +4.3595095079586539e00,
                +4.3781612023270764e00,
                +4.8087243718248345e01,
                +3.2385554030311042e01,
                +2.8347659104092045e01,
                +2.7159779076468158e01,
                +5.2020556181259558e01,
                +3.3918238356514138e01,
                +6.5446665006430811e00,
                +2.2540760424766276e01,
                +7.5993054885377020e00,
                +1.6350726163094549e01,
                +4.8714353281692251e00,
                +7.2641079346924871e00,
                +2.4388022059359354e01,
                +2.5213789399198038e01,
                +4.5478271115225732e00,
                +4.5665607640851649e00,
                +4.9974720410596369e01,
                +3.3800644145323879e01,
                +2.9638102916194356e01,
                +2.8079397346641731e01,
                +5.5254816852785403e01,
                +6.4766138204142480e00,
                +1.1928911659344870e00,
                +4.0285410771690495e00,
                +1.3849717326666449e00,
                +2.8838036045730786e00,
                +8.8802328302553557e-01,
                +1.3238816038516765e00,
                +4.3595095079586539e00,
                +4.5478271115225732e00,
                +8.2915871048509038e-01,
                +8.3244737362574239e-01,
                +9.0723820483097413e00,
                +6.1640611695479173e00,
                +5.4144434609928869e00,
                +5.0646986711001167e00,
                +1.0242058685933733e01,
                +6.4922519019669256e00,
                +1.1976302445456630e00,
                +4.0457865525221370e00,
                +1.3904773652700402e00,
                +2.8967937758905151e00,
                +8.9154859790635033e-01,
                +1.3291443974122232e00,
                +4.3781612023270764e00,
                +4.5665607640851649e00,
                +8.3244737362574239e-01,
                +8.3575205467571911e-01,
                +9.1089743976350785e00,
                +6.1884010772150004e00,
                +5.4356184036894515e00,
                +5.0855614473817186e00,
                +1.0278414763476997e01,
                +6.9282713428363266e01,
                +1.3053810854181670e01,
                +4.4440038553551688e01,
                +1.5156457827338222e01,
                +3.1989891272947684e01,
                +9.7171097068987322e00,
                +1.4487919292348902e01,
                +4.8087243718248345e01,
                +4.9974720410596369e01,
                +9.0723820483097413e00,
                +9.1089743976350785e00,
                +9.9444353388399350e01,
                +6.7435693634237566e01,
                +5.9190275027468758e01,
                +5.5654467988948483e01,
                +1.1125328010581040e02,
                +4.8619305479898145e01,
                +8.8678073889297409e00,
                +2.9926910685255862e01,
                +1.0295584693450651e01,
                +2.1410657159900445e01,
                +6.6015513035883231e00,
                +9.8414533114041927e00,
                +3.2385554030311042e01,
                +3.3800644145323879e01,
                +6.1640611695479173e00,
                +6.1884010772150004e00,
                +6.7435693634237566e01,
                +4.5831414377859105e01,
                +4.0264713770651987e01,
                +3.7642169166613321e01,
                +7.6311662100126583e01,
                +4.3364114441872893e01,
                +7.7888498604059606e00,
                +2.6194837765153459e01,
                +9.0426690304625250e00,
                +1.8693996640429365e01,
                +5.7985108827431544e00,
                +8.6438021973771111e00,
                +2.8347659104092045e01,
                +2.9638102916194356e01,
                +5.4144434609928869e00,
                +5.4356184036894515e00,
                +5.9190275027468758e01,
                +4.0264713770651987e01,
                +3.5388459036200665e01,
                +3.3006545115181545e01,
                +6.7323234267199609e01,
                +3.7773137469159217e01,
                +7.2884836870559822e00,
                +2.5102571723162029e01,
                +8.4629849467909484e00,
                +1.8209024833681802e01,
                +5.4250857435598849e00,
                +8.0896913798057266e00,
                +2.7159779076468158e01,
                +2.8079397346641731e01,
                +5.0646986711001167e00,
                +5.0855614473817186e00,
                +5.5654467988948483e01,
                +3.7642169166613321e01,
                +3.3006545115181545e01,
                +3.1270688545372174e01,
                +6.1534661835125377e01,
                +9.4819536752314136e01,
                +1.4723882690435001e01,
                +4.8059235287218819e01,
                +1.7089770111859078e01,
                +3.3532590882087042e01,
                +1.0964613701101552e01,
                +1.6335937257210038e01,
                +5.2020556181259558e01,
                +5.5254816852785403e01,
                +1.0242058685933733e01,
                +1.0278414763476997e01,
                +1.1125328010581040e02,
                +7.6311662100126583e01,
                +6.7323234267199609e01,
                +6.1534661835125377e01,
                +1.3271711768686609e02,
            ],
            dtype=torch.double,
        ).reshape(16, 16),
    },
    "MB16_43_02": {  # q
        "gw": reshape_fortran(
            torch.tensor(
                [
                    8.1529926850468e-01,
                    2.2414425700956e-03,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    1.0229633690838e00,
                    2.0291440612814e-03,
                    4.4863013350169e-10,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    1.1578430925721e00,
                    2.0553787158892e-03,
                    3.1900095597967e-10,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    1.3687709975145e00,
                    8.2192921625604e-03,
                    5.9806186105627e-08,
                    1.4414431691639e-03,
                    0.0000000000000e00,
                    9.1646421111918e-01,
                    5.3618567538002e-03,
                    1.7524886555881e-08,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    5.5812105498128e-01,
                    1.5344013422165e-03,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    7.5659186904499e-01,
                    2.0800426162236e-03,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    7.7216222381926e-01,
                    2.1228490523027e-03,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    9.9399938189677e-01,
                    2.8186797641147e-03,
                    7.1776130956295e-10,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    5.5220460179834e-01,
                    1.5181356707747e-03,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    1.1425733041093e00,
                    2.0284312531719e-03,
                    3.1484305175579e-10,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    8.1572049134917e-01,
                    6.4067287425100e-03,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    1.2272138342948e00,
                    1.3612868628584e-02,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    8.3285816475829e-01,
                    2.2897159576321e-03,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    7.4607907232618e-01,
                    2.0511405541139e-03,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    1.0523038058719e00,
                    2.0872155477415e-03,
                    4.6145584504737e-10,
                    0.0000000000000e00,
                    0.0000000000000e00,
                ],
                dtype=torch.double,
            ),
            (5, 16),
        ),
        "c6": torch.tensor(
            [
                +2.0445120203823866e00,
                +1.7218633174121731e01,
                +9.9165146215717357e00,
                +7.3727562533175162e00,
                +1.1274430670476232e01,
                +1.4092868776138123e00,
                +1.8966990564690756e00,
                +1.9356055931843188e00,
                +1.7551879918280203e01,
                +1.3944366208059402e00,
                +9.4919347855846397e00,
                +3.4338846794543656e00,
                +5.1890967952531826e00,
                +2.0893065689528356e00,
                +1.8643394892776621e00,
                +1.7711860198517961e01,
                +1.7218633174121731e01,
                +1.4551701715663216e02,
                +8.3663440943855434e01,
                +6.1615521213900891e01,
                +9.7890899611424800e01,
                +1.1870822247740330e01,
                +1.5973649169482306e01,
                +1.6301286963067067e01,
                +1.4919780310396965e02,
                +1.1745752390466828e01,
                +8.0017221114768205e01,
                +2.9166607632863368e01,
                +4.3109104756999578e01,
                +1.7596043013362824e01,
                +1.5699897192125434e01,
                +1.4968681631232548e02,
                +9.9165146215717357e00,
                +8.3663440943855434e01,
                +4.8141736682149315e01,
                +3.5614701451627965e01,
                +5.5547060433990431e01,
                +6.8360637704831371e00,
                +9.1995401636562146e00,
                +9.3882404637613881e00,
                +8.5540161797400231e01,
                +6.7640346115584551e00,
                +4.6061357945537566e01,
                +1.6728500037949299e01,
                +2.4988526491419485e01,
                +1.0133828188297187e01,
                +9.0422230814540008e00,
                +8.6060423769457486e01,
                +7.3727562533175162e00,
                +6.1615521213900891e01,
                +3.5614701451627965e01,
                +2.7536917224782034e01,
                +3.7725575510338921e01,
                +5.0796706366563082e00,
                +6.8398710356417141e00,
                +6.9802069042060273e00,
                +6.1752652354394669e01,
                +5.0261221898125381e00,
                +3.4174002955129453e01,
                +1.2127754626423620e01,
                +1.9868384670852333e01,
                +7.5341030689407287e00,
                +6.7246550380598267e00,
                +6.3378071517051573e01,
                +1.1274430670476232e01,
                +9.7890899611424800e01,
                +5.5547060433990431e01,
                +3.7725575510338921e01,
                +7.9567306399544009e01,
                +7.7832784975309419e00,
                +1.0458596915984323e01,
                +1.0672977479452079e01,
                +1.0490835149185172e02,
                +7.7013703644913090e00,
                +5.2786783745701989e01,
                +2.0379660856443916e01,
                +2.4959357020808635e01,
                +1.1522376352043223e01,
                +1.0272860536405052e01,
                +1.0070373791573728e02,
                +1.4092868776138123e00,
                +1.1870822247740330e01,
                +6.8360637704831371e00,
                +5.0796706366563082e00,
                +7.7832784975309419e00,
                +9.7143307201423013e-01,
                +1.3073985471981771e00,
                +1.3342167933910600e00,
                +1.2104224858299013e01,
                +9.6119674377061370e-01,
                +6.5430973466126110e00,
                +2.3679791659612284e00,
                +3.5739076317634009e00,
                +1.4401645210667646e00,
                +1.2850878529532428e00,
                +1.2210868267102807e01,
                +1.8966990564690756e00,
                +1.5973649169482306e01,
                +9.1995401636562146e00,
                +6.8398710356417141e00,
                +1.0458596915984323e01,
                +1.3073985471981771e00,
                +1.7595726212201164e00,
                +1.7956663226673035e00,
                +1.6282576900800656e01,
                +1.2936219265079247e00,
                +8.8056748754158409e00,
                +3.1855627793589902e00,
                +4.8141189065837828e00,
                +1.9382550340934208e00,
                +1.7295528839418528e00,
                +1.6431213262280721e01,
                +1.9356055931843188e00,
                +1.6301286963067067e01,
                +9.3882404637613881e00,
                +6.9802069042060273e00,
                +1.0672977479452079e01,
                +1.3342167933910600e00,
                +1.7956663226673035e00,
                +1.8325004071960138e00,
                +1.6616503336047927e01,
                +1.3201575763107232e00,
                +8.9862998672227778e00,
                +3.2508944656814283e00,
                +4.9129083461328635e00,
                +1.9780139901008735e00,
                +1.7650308652612590e00,
                +1.6768236129037149e01,
                +1.7551879918280203e01,
                +1.4919780310396965e02,
                +8.5540161797400231e01,
                +6.1752652354394669e01,
                +1.0490835149185172e02,
                +1.2104224858299013e01,
                +1.6282576900800656e01,
                +1.6616503336047927e01,
                +1.5457378985322518e02,
                +1.1976729263976347e01,
                +8.1690303068599093e01,
                +3.0168338609030528e01,
                +4.2644349548950380e01,
                +1.7936881600003225e01,
                +1.6001265308988039e01,
                +1.5347610462145525e02,
                +1.3944366208059402e00,
                +1.1745752390466828e01,
                +6.7640346115584551e00,
                +5.0261221898125381e00,
                +7.7013703644913090e00,
                +9.6119674377061370e-01,
                +1.2936219265079247e00,
                +1.3201575763107232e00,
                +1.1976729263976347e01,
                +9.5106827997534493e-01,
                +6.4741525381489815e00,
                +2.3430358371754840e00,
                +3.5362209268025282e00,
                +1.4249888993613478e00,
                +1.2715462823762271e00,
                +1.2082215779412374e01,
                +9.4919347855846397e00,
                +8.0017221114768205e01,
                +4.6061357945537566e01,
                +3.4174002955129453e01,
                +5.2786783745701989e01,
                +6.5430973466126110e00,
                +8.8056748754158409e00,
                +8.9862998672227778e00,
                +8.1690303068599093e01,
                +6.4741525381489815e00,
                +4.4080204545172919e01,
                +1.5979568067558814e01,
                +2.4022019560488655e01,
                +9.6999221657322199e00,
                +8.6552649635203096e00,
                +8.2309501485739617e01,
                +3.4338846794543656e00,
                +2.9166607632863368e01,
                +1.6728500037949299e01,
                +1.2127754626423620e01,
                +2.0379660856443916e01,
                +2.3679791659612284e00,
                +3.1855627793589902e00,
                +3.2508944656814283e00,
                +3.0168338609030528e01,
                +2.3430358371754840e00,
                +1.5979568067558814e01,
                +5.8905372773199716e00,
                +8.4001330720023315e00,
                +3.5091980963054388e00,
                +3.1305978208902947e00,
                +3.0002861062353571e01,
                +5.1890967952531826e00,
                +4.3109104756999578e01,
                +2.4988526491419485e01,
                +1.9868384670852333e01,
                +2.4959357020808635e01,
                +3.5739076317634009e00,
                +4.8141189065837828e00,
                +4.9129083461328635e00,
                +4.2644349548950380e01,
                +3.5362209268025282e00,
                +2.4022019560488655e01,
                +8.4001330720023315e00,
                +1.4579684651546014e01,
                +5.3025562421129129e00,
                +4.7338120894817912e00,
                +4.4341020299181721e01,
                +2.0893065689528356e00,
                +1.7596043013362824e01,
                +1.0133828188297187e01,
                +7.5341030689407287e00,
                +1.1522376352043223e01,
                +1.4401645210667646e00,
                +1.9382550340934208e00,
                +1.9780139901008735e00,
                +1.7936881600003225e01,
                +1.4249888993613478e00,
                +9.6999221657322199e00,
                +3.5091980963054388e00,
                +5.3025562421129129e00,
                +2.1350826022081422e00,
                +1.9051860735457213e00,
                +1.8100081462570529e01,
                +1.8643394892776621e00,
                +1.5699897192125434e01,
                +9.0422230814540008e00,
                +6.7246550380598267e00,
                +1.0272860536405052e01,
                +1.2850878529532428e00,
                +1.7295528839418528e00,
                +1.7650308652612590e00,
                +1.6001265308988039e01,
                +1.2715462823762271e00,
                +8.6552649635203096e00,
                +3.1305978208902947e00,
                +4.7338120894817912e00,
                +1.9051860735457213e00,
                +1.7000485157308447e00,
                +1.6149615456526835e01,
                +1.7711860198517961e01,
                +1.4968681631232548e02,
                +8.6060423769457486e01,
                +6.3378071517051573e01,
                +1.0070373791573728e02,
                +1.2210868267102807e01,
                +1.6431213262280721e01,
                +1.6768236129037149e01,
                +1.5347610462145525e02,
                +1.2082215779412374e01,
                +8.2309501485739617e01,
                +3.0002861062353571e01,
                +4.4341020299181721e01,
                +1.8100081462570529e01,
                +1.6149615456526835e01,
                +1.5397610791310674e02,
            ],
            dtype=torch.double,
        ).reshape(16, 16),
    },
    "MB16_43_03": {  # CN, q
        "gw": reshape_fortran(
            torch.tensor(
                [
                    0.0000000000000e00,
                    0.0000000000000e00,
                    3.9504240794198e-09,
                    1.0097386820563e-02,
                    9.3220073517801e-01,
                    2.5639475147259e-02,
                    0.0000000000000e00,
                    1.4797717135069e-11,
                    1.9052325554662e-04,
                    1.1116141991349e00,
                    2.9351813145404e-03,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    5.3735027174640e-03,
                    8.8839284731689e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    9.7753605500113e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    9.8036041832883e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    1.2906805265616e-09,
                    9.9812405470369e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    3.9706733315821e-14,
                    3.4070161345981e-05,
                    1.0390095052930e00,
                    1.4775589359887e-04,
                    0.0000000000000e00,
                    1.4709587472355e-03,
                    8.7779708361772e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    4.9950676848882e-03,
                    8.2586385264927e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    4.7331797500540e-03,
                    7.8936056984439e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    6.1261133783377e-04,
                    1.1351788724746e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    1.4636823892691e-06,
                    1.0081631910738e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    1.5989624842058e-05,
                    1.0653935240443e00,
                    7.4688541496264e-05,
                    0.0000000000000e00,
                    9.7208858575852e-03,
                    5.7638169663657e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    5.8433000707751e-10,
                    9.6868781982789e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    2.8478831017041e-04,
                    9.0297474706410e-01,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                    0.0000000000000e00,
                ],
                dtype=torch.double,
            ),
            (7, 16),
        ),
        "c6": torch.tensor(
            [
                +2.6969645804631604e01,
                +2.8729803918609342e01,
                +8.0783557426272719e00,
                +1.2572598284646498e01,
                +3.9383726731881893e01,
                +6.0100611314023915e01,
                +2.8877446590256088e01,
                +7.9285493229103343e00,
                +7.5097624231198781e00,
                +7.1772575509757184e00,
                +1.9756649427858040e01,
                +6.3114791359145649e01,
                +2.9607901769798627e01,
                +5.3279952989769619e00,
                +3.4651507628617878e01,
                +8.1388552398630072e00,
                +2.8729803918609342e01,
                +3.0858718804005470e01,
                +8.5314526128998711e00,
                +1.3158527195522042e01,
                +4.0242276058963732e01,
                +6.2518208475630352e01,
                +3.0769885934381300e01,
                +8.3738757317567192e00,
                +7.9309682052197692e00,
                +7.5798206446137968e00,
                +2.1471517618889017e01,
                +6.6436238381645140e01,
                +3.1548229972467226e01,
                +5.6258043424412678e00,
                +3.4923486847229228e01,
                +8.5961975589898643e00,
                +8.0783557426272719e00,
                +8.5314526128998711e00,
                +2.4429152644641161e00,
                +3.8470627309562269e00,
                +1.2418251750012546e01,
                +1.8522898371215359e01,
                +8.6470455821783965e00,
                +2.3973918803546712e00,
                +2.2709711896641291e00,
                +2.1704185011554786e00,
                +5.7906292579429293e00,
                +1.9170869616128407e01,
                +8.8657656755301080e00,
                +1.6115591315647257e00,
                +1.1136757306621792e01,
                +2.4609116715906052e00,
                +1.2572598284646498e01,
                +1.3158527195522042e01,
                +3.8470627309562269e00,
                +6.1873811861491532e00,
                +2.1045559351132013e01,
                +3.0123412184968540e01,
                +1.3450271401073154e01,
                +3.7748067440421131e00,
                +3.5762880002649533e00,
                +3.4179332169981902e00,
                +8.7894143888806884e00,
                +3.0447639291352356e01,
                +1.3790466532352390e01,
                +2.5387769376831475e00,
                +1.9680032604563024e01,
                +3.8746391243000158e00,
                +3.9383726731881893e01,
                +4.0242276058963732e01,
                +1.2418251750012546e01,
                +2.1045559351132013e01,
                +8.0792669914256635e01,
                +1.0524757517715317e02,
                +4.2068874980498862e01,
                +1.2180230990667235e01,
                +1.1544195337928167e01,
                +1.1032977672473640e01,
                +2.5675572843387016e01,
                +1.0047223990865119e02,
                +4.3132752418889773e01,
                +8.2028913059177953e00,
                +8.3004740679372276e01,
                +1.2500821517587408e01,
                +6.0100611314023915e01,
                +6.2518208475630352e01,
                +1.8522898371215359e01,
                +3.0123412184968540e01,
                +1.0524757517715317e02,
                +1.4754968408298424e02,
                +6.4275900488174102e01,
                +1.8173467936826448e01,
                +1.7219167833650303e01,
                +1.6456702330152496e01,
                +4.1319144188437214e01,
                +1.4726172884743752e02,
                +6.5901567272212375e01,
                +1.2226230276259367e01,
                +1.0042992019072095e02,
                +1.8653608543109257e01,
                +2.8877446590256088e01,
                +3.0769885934381300e01,
                +8.6470455821783965e00,
                +1.3450271401073154e01,
                +4.2068874980498862e01,
                +6.4275900488174102e01,
                +3.0920640741304151e01,
                +8.4867269265488794e00,
                +8.0384251534348703e00,
                +7.6825133853358922e00,
                +2.1168508106271982e01,
                +6.7543056296917484e01,
                +3.1702779636089545e01,
                +5.7030141029870425e00,
                +3.6962950298763566e01,
                +8.7118494250595724e00,
                +7.9285493229103343e00,
                +8.3738757317567192e00,
                +2.3973918803546712e00,
                +3.7748067440421131e00,
                +1.2180230990667235e01,
                +1.8173467936826448e01,
                +8.4867269265488794e00,
                +2.3527194049705815e00,
                +2.2286519594791270e00,
                +2.1299730839454756e00,
                +5.6843960534634066e00,
                +1.8812501010281672e01,
                +8.7013919729876186e00,
                +1.5815237509996671e00,
                +1.0919772339795125e01,
                +2.4150564158071561e00,
                +7.5097624231198781e00,
                +7.9309682052197692e00,
                +2.2709711896641291e00,
                +3.5762880002649533e00,
                +1.1544195337928167e01,
                +1.7219167833650303e01,
                +8.0384251534348703e00,
                +2.2286519594791270e00,
                +2.1111293622441099e00,
                +2.0176540534750380e00,
                +5.3830571387834780e00,
                +1.7821531942660105e01,
                +8.2417506804364393e00,
                +1.4981298823013842e00,
                +1.0352898443989817e01,
                +2.2877009236807915e00,
                +7.1772575509757184e00,
                +7.5798206446137968e00,
                +2.1704185011554786e00,
                +3.4179332169981902e00,
                +1.1032977672473640e01,
                +1.6456702330152496e01,
                +7.6825133853358922e00,
                +2.1299730839454756e00,
                +2.0176540534750380e00,
                +1.9283175881762038e00,
                +5.1447273217294684e00,
                +1.7032428915733121e01,
                +7.8768364100925279e00,
                +1.4317965486738418e00,
                +9.8943978385701605e00,
                +2.1864075234915714e00,
                +1.9756649427858040e01,
                +2.1471517618889017e01,
                +5.7906292579429293e00,
                +8.7894143888806884e00,
                +2.5675572843387016e01,
                +4.1319144188437214e01,
                +2.1168508106271982e01,
                +5.6843960534634066e00,
                +5.3830571387834780e00,
                +5.1447273217294684e00,
                +1.5194417412575918e01,
                +4.4825188609896031e01,
                +2.1704001718253391e01,
                +3.8172825287704240e00,
                +2.1539554554162795e01,
                +5.8355457277838685e00,
                +6.3114791359145649e01,
                +6.6436238381645140e01,
                +1.9170869616128407e01,
                +3.0447639291352356e01,
                +1.0047223990865119e02,
                +1.4726172884743752e02,
                +6.7543056296917484e01,
                +1.8812501010281672e01,
                +1.7821531942660105e01,
                +1.7032428915733121e01,
                +4.4825188609896031e01,
                +1.5096768893658918e02,
                +6.9251466330288324e01,
                +1.2648593650527900e01,
                +9.1879170429484873e01,
                +1.9310583681270991e01,
                +2.9607901769798627e01,
                +3.1548229972467226e01,
                +8.8657656755301080e00,
                +1.3790466532352390e01,
                +4.3132752418889773e01,
                +6.5901567272212375e01,
                +3.1702779636089545e01,
                +8.7013919729876186e00,
                +8.2417506804364393e00,
                +7.8768364100925279e00,
                +2.1704001718253391e01,
                +6.9251466330288324e01,
                +3.2504702769579829e01,
                +5.8472671394680997e00,
                +3.7897574429129875e01,
                +8.9322087944903910e00,
                +5.3279952989769619e00,
                +5.6258043424412678e00,
                +1.6115591315647257e00,
                +2.5387769376831475e00,
                +8.2028913059177953e00,
                +1.2226230276259367e01,
                +5.7030141029870425e00,
                +1.5815237509996671e00,
                +1.4981298823013842e00,
                +1.4317965486738418e00,
                +3.8172825287704240e00,
                +1.2648593650527900e01,
                +5.8472671394680997e00,
                +1.0631312285249483e00,
                +7.3621267055024324e00,
                +1.6234254583398267e00,
                +3.4651507628617878e01,
                +3.4923486847229228e01,
                +1.1136757306621792e01,
                +1.9680032604563024e01,
                +8.3004740679372276e01,
                +1.0042992019072095e02,
                +3.6962950298763566e01,
                +1.0919772339795125e01,
                +1.0352898443989817e01,
                +9.8943978385701605e00,
                +2.1539554554162795e01,
                +9.1879170429484873e01,
                +3.7897574429129875e01,
                +7.3621267055024324e00,
                +9.3031746769707865e01,
                +1.1206049984338996e01,
                +8.1388552398630072e00,
                +8.5961975589898643e00,
                +2.4609116715906052e00,
                +3.8746391243000158e00,
                +1.2500821517587408e01,
                +1.8653608543109257e01,
                +8.7118494250595724e00,
                +2.4150564158071561e00,
                +2.2877009236807915e00,
                +2.1864075234915714e00,
                +5.8355457277838685e00,
                +1.9310583681270991e01,
                +8.9322087944903910e00,
                +1.6234254583398267e00,
                +1.1206049984338996e01,
                +2.4790453596084148e00,
            ],
            dtype=torch.double,
        ).reshape(16, 16),
    },
}


samples: dict[str, Record] = merge_nested_dicts(mols, refs)
