# This file is part of tad-dftd4.
#
# SPDX-Identifier: LGPL-3.0
# Copyright (C) 2022 Marvin Friede
#
# tad-dftd4 is free software: you can redistribute it and/or modify it under
# the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# tad-dftd4 is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with tad-dftd4. If not, see <https://www.gnu.org/licenses/>.
"""
Data for testing repulsion taken from https://github.com/grimme-lab/mstore.
"""
from __future__ import annotations

import torch

from tad_dftd4._typing import Molecule
from tad_dftd4.utils import to_number


def merge_nested_dicts(a: dict, b: dict) -> dict:
    """
    Merge nested dictionaries. Dictionary `a` remains unaltered, while
    the corresponding keys of it are added to `b`.

    Parameters
    ----------
    a : dict
        First dictionary (not changed).
    b : dict
        Second dictionary (changed).

    Returns
    -------
    dict
        Merged dictionary `b`.
    """
    for key in b:
        if key in a:
            b[key].update(a[key])
    return b


mols: dict[str, Molecule] = {
    "H": {
        "numbers": to_number(["H"]),
        "positions": torch.tensor(
            [[0.00000000000000, 0.00000000000000, 0.00000000000000]],
        ),
    },
    "He": {
        "numbers": to_number(["He"]),
        "positions": torch.tensor(
            [[0.00000000000000, 0.00000000000000, 0.00000000000000]],
        ),
    },
    "C": {
        "numbers": to_number(["C"]),
        "positions": torch.tensor(
            [[0.00000000000000, 0.00000000000000, 0.00000000000000]],
        ),
    },
    "Rn": {
        "numbers": to_number(["Rn"]),
        "positions": torch.tensor(
            [[0.00000000000000, 0.00000000000000, 0.00000000000000]],
        ),
    },
    "H2": {
        "numbers": to_number(["H", "H"]),
        "positions": torch.tensor(
            [
                [+0.00000000000000, +0.00000000000000, -0.70252931147690],
                [+0.00000000000000, +0.00000000000000, +0.70252931147690],
            ],
        ),
    },
    "LiH": {
        "numbers": to_number(["Li", "H"]),
        "positions": torch.tensor(
            [
                [+0.00000000000000, +0.00000000000000, -1.50796743897235],
                [+0.00000000000000, +0.00000000000000, +1.50796743897235],
            ]
        ),
    },
    "HLi": {
        "numbers": to_number(["H", "Li"]),
        "positions": torch.tensor(
            [
                [0.00000000000000, 0.00000000000000, -1.50796743897235],
                [0.00000000000000, 0.00000000000000, 1.50796743897235],
            ],
        ),
    },
    "HC": {
        "numbers": to_number(["H", "C"]),
        "positions": torch.tensor(
            [
                [0.00000000000000, 0.00000000000000, 0.00000000000000],
                [0.00000000000000, 0.00000000000000, 1.40000000000000],
            ]
        ),
    },
    "HHe": {
        "numbers": to_number(["H", "He"]),
        "positions": torch.tensor(
            [
                [0.00000000000000, 0.00000000000000, 0.00000000000000],
                [0.00000000000000, 0.00000000000000, 1.70000000000000],
            ]
        ),
    },
    "SCl": {
        "numbers": to_number(["S", "Cl"]),
        "positions": torch.tensor(
            [
                [0.00000000000000, 0.00000000000000, 0.00000000000000],
                [0.00000000000000, 0.00000000000000, 2.10000000000000],
            ]
        ),
    },
    "S2": {
        "numbers": to_number(["S", "S"]),
        "positions": torch.tensor(
            [
                [+0.00000000000000, +0.00000000000000, -1.80281271474629],
                [+0.00000000000000, +0.00000000000000, +1.80281271474629],
            ],
        ),
    },
    "H2O": {
        "numbers": torch.tensor([8, 1, 1]),
        "positions": torch.tensor(
            [
                [+0.00000000000000, +0.00000000000000, -0.74288549752983],
                [-1.43472674945442, +0.00000000000000, +0.37144274876492],
                [+1.43472674945442, +0.00000000000000, +0.37144274876492],
            ]
        ),
    },
    "NO2": {
        "numbers": to_number("N O O".split()),
        "positions": torch.tensor(
            [
                [-0.18522935692720, +0.16338284167819, -0.21364928184270],
                [-0.88525635077826, -1.85115058687488, -0.86821060517558],
                [+0.56323030352441, +2.04159061614098, -1.15650936924419],
            ]
        ),
    },
    "NH3": {
        "numbers": to_number("N H H H".split()),
        "positions": torch.tensor(
            [
                [+0.00000000000000, +0.00000000000000, -0.54524837997150],
                [-0.88451840382282, +1.53203081565085, +0.18174945999050],
                [-0.88451840382282, -1.53203081565085, +0.18174945999050],
                [+1.76903680764564, +0.00000000000000, +0.18174945999050],
            ]
        ),
    },
    "NH3-dimer": {
        "numbers": to_number("N N H H H H H H".split()),
        "positions": torch.tensor(
            [
                [-2.98334550857544, -0.08808205276728, +0.00000000000000],
                [+2.98334550857544, +0.08808205276728, +0.00000000000000],
                [-4.07920360565186, +0.25775116682053, +1.52985656261444],
                [-1.60526800155640, +1.24380481243134, +0.00000000000000],
                [-4.07920360565186, +0.25775116682053, -1.52985656261444],
                [+4.07920360565186, -0.25775116682053, -1.52985656261444],
                [+1.60526800155640, -1.24380481243134, +0.00000000000000],
                [+4.07920360565186, -0.25775116682053, +1.52985656261444],
            ]
        ),
    },
    "CH4": {
        "numbers": to_number(["C", "H", "H", "H", "H"]),
        "positions": torch.tensor(
            [
                [+0.00000000000000, +0.00000000000000, +0.00000000000000],
                [-1.19077691784446, -1.19077691784446, -1.19077691784446],
                [+1.19077691784446, +1.19077691784446, -1.19077691784446],
                [+1.19077691784446, -1.19077691784446, +1.19077691784446],
                [-1.19077691784446, +1.19077691784446, +1.19077691784446],
            ]
        ),
    },
    "SiH4": {
        "numbers": to_number(["Si", "H", "H", "H", "H"]),
        "positions": torch.tensor(
            [
                [+0.00000000000000, -0.00000000000000, +0.00000000000000],
                [+1.61768389755830, +1.61768389755830, -1.61768389755830],
                [-1.61768389755830, -1.61768389755830, -1.61768389755830],
                [+1.61768389755830, -1.61768389755830, +1.61768389755830],
                [-1.61768389755830, +1.61768389755830, +1.61768389755830],
            ],
        ),
    },
    "MB16_43_01": {
        "numbers": to_number("Na H O H F H H O N H H Cl B B N Al".split()),
        "positions": torch.tensor(
            [
                [-1.85528263484662, +3.58670515364616, -2.41763729306344],
                [+4.40178023537845, +0.02338844412653, -4.95457749372945],
                [-2.98706033463438, +4.76252065456814, +1.27043301573532],
                [+0.79980886075526, +1.41103455609189, -5.04655321620119],
                [-4.20647469409936, +1.84275767548460, +4.55038084858449],
                [-3.54356121843970, -3.18835665176557, +1.46240021785588],
                [+2.70032160109941, +1.06818452504054, -1.73234650374438],
                [+3.73114088824361, -2.07001543363453, +2.23160937604731],
                [-1.75306819230397, +0.35951417150421, +1.05323406177129],
                [+5.41755788583825, -1.57881830078929, +1.75394002750038],
                [-2.23462868255966, -2.13856505054269, +4.10922285746451],
                [+1.01565866207568, -3.21952154552768, -3.36050963020778],
                [+2.42119255723593, +0.26626435093114, -3.91862474360560],
                [-3.02526098819107, +2.53667889095925, +2.31664984740423],
                [-2.00438948664892, -2.29235136977220, +2.19782807357059],
                [+1.12226554109716, -1.36942007032045, +0.48455055461782],
            ],
        ),
    },
    "MB16_43_02": {
        "numbers": to_number("H S B O Mg H H H Si H B Li F H H S".split()),
        "positions": torch.tensor(
            [
                [-1.79537625851198, -3.77866422935275, -1.07883558363403],
                [-2.68278833302782, +0.38892666265890, +1.66214865238427],
                [+0.11484649791305, +1.48857933226955, +3.65660396510375],
                [-1.07998879593946, -0.16259121615748, -4.55703065871422],
                [+0.60302832999383, +4.08816149622342, -0.02589373148029],
                [-1.22534089315880, -1.79981382478068, -3.70773173318592],
                [-1.33460982049866, -4.24819082475503, +2.72791902701083],
                [-0.16278082578516, +2.41267994179303, +5.69030695190570],
                [+2.87802444057103, -0.33120525058830, +1.88311373530297],
                [+0.68489327931487, +0.32790204044961, -4.20547693710673],
                [-1.20919773588330, -2.87253762561437, +0.94064204223101],
                [-3.25572604597922, +2.21241092990940, -2.86715549314771],
                [-1.83147468262373, +5.20527293771933, -2.26976270603341],
                [+4.90885865772880, -1.92576561961811, +2.99069919443735],
                [+1.26806242248758, -2.60409341782411, +0.55162805282247],
                [+4.11956976339902, +1.59892866766766, -1.39117477789609],
            ],
        ),
    },
    "MB16_43_03": {
        "numbers": to_number("C O H Li Mg Al C H H H F S C H Na H".split()),
        "positions": torch.tensor(
            [
                [-0.02148551327524, -0.67161751504297, -4.75078512817560],
                [+1.37792545875526, -3.24818416423144, +3.83896600631495],
                [-2.23986953822894, +1.64550402751694, +3.42773272178522],
                [-0.87622711432790, -2.74068400827752, +1.43723692979592],
                [+1.29492470653815, +1.86470311043681, -1.04536500695239],
                [-3.65768365013010, +0.45437052179208, -1.41566056087159],
                [-0.23245910487384, -1.83274112101585, -2.43395808606122],
                [+0.30373451850419, -3.84228931776777, -2.44882782867802],
                [-3.36159503902161, +4.20056392581975, +1.63352684198071],
                [+0.49372989648081, -1.56245253044952, -6.53610501083288],
                [+4.38566058812996, +1.86127331114460, +0.56178822055152],
                [-1.17545963764009, +2.49456345795141, -4.90195191215762],
                [-1.86623614216854, +2.76329843590746, +1.71572598870213],
                [+1.02361259176985, -4.24377370348987, +5.32418288889440],
                [+4.71194535010347, -1.03648125005561, +3.35573062118779],
                [-0.16051737061546, +3.89394681976155, +2.23776331451663],
            ],
        ),
    },
    "MB16_43_07": {
        "numbers": to_number("C H B H H Cl F N C H S H H O F Mg".split()),
        "positions": torch.tensor(
            [
                [-3.75104222741336, -5.81308736205268, -1.22507366840233],
                [-1.45226572768296, -3.01878767879831, +2.38723142561073],
                [-1.99423317853240, -3.52953889999752, -1.30301724065129],
                [-4.33750965171233, -6.65936981001909, +0.55979831484564],
                [-4.51833920602637, -6.72398616322561, -2.90031439001886],
                [-1.25657105633503, -2.39389339457851, -4.58765484136593],
                [-0.14864209579028, +4.40065007854051, +1.35717716022989],
                [-0.91662354168326, -2.22680612180354, +0.71122632634918],
                [+1.83282041695179, +5.36061635978157, +3.22095765094686],
                [+0.66518416413161, +6.30980889882630, +4.62705414435961],
                [+3.68701623423530, +2.79957532381681, +4.21336212424745],
                [+1.69373321407504, +0.01030275402386, -3.74820290941150],
                [+3.35791986589808, +2.52513229318111, -3.46078430541625],
                [+2.79199182665654, +1.01759578021447, -2.59243571461852],
                [+3.05358934464082, +7.15252337445235, +1.82164153773112],
                [+1.29297161858681, +0.78926456763834, +0.91903438556425],
            ],
        ),
    },
    "MB16_43_08": {
        "numbers": to_number("C O B F H Al H H O B Be C H H B F".split()),
        "positions": torch.tensor(
            [
                [-1.27823293129313, +0.06442674490989, +2.76980447300615],
                [+2.05039033278229, +0.64690940303039, -0.29571013189632],
                [-0.07388472989895, +2.46033979750309, -1.30590420482375],
                [+1.10019432741349, +4.43501067437330, -2.64796515354449],
                [-1.89008873387150, +0.02064696008121, +4.74727599156952],
                [+0.81013963557610, +1.41165582964016, -6.35835508532445],
                [+2.51638337449170, +1.74086425451198, +3.45340860505386],
                [+2.62048878651566, -1.58024532804571, +2.87415150030394],
                [-0.92472602392464, -3.37659091509259, -0.68138826965952],
                [-2.19962829538645, -2.53092502025386, +1.35654623095955],
                [+0.92594749614406, -1.61669775704536, -1.93872059141561],
                [+1.63141903847248, +0.18081362275364, +2.42899361614054],
                [-3.96336280784845, -3.68611886004249, +2.18920954455515],
                [-1.17097381446263, +1.08303722364990, -3.04753977323348],
                [-2.18263847972349, +2.31604957286801, +1.11461091308323],
                [+2.02857282501340, -1.56917620284149, -4.65841766477431],
            ],
        ),
    },
    "PbH4-BiH3": {
        "numbers": to_number("Pb H H H H Bi H H H".split()),
        "positions": torch.tensor(
            [
                [-0.00000020988889, -4.98043478877778, +0.00000000000000],
                [+3.06964045311111, -6.06324400177778, +0.00000000000000],
                [-1.53482054188889, -6.06324400177778, -2.65838526500000],
                [-1.53482054188889, -6.06324400177778, +2.65838526500000],
                [-0.00000020988889, -1.72196703577778, +0.00000000000000],
                [-0.00000020988889, +4.77334244722222, +0.00000000000000],
                [+1.35700257511111, +6.70626379422222, -2.35039772300000],
                [-2.71400388988889, +6.70626379422222, +0.00000000000000],
                [+1.35700257511111, +6.70626379422222, +2.35039772300000],
            ]
        ),
    },
    "C6H5I-CH3SH": {
        "numbers": to_number("C C C C C C I H H H H H S H C H H H".split()),
        "positions": torch.tensor(
            [
                [-1.42754169820131, -1.50508961850828, -1.93430551124333],
                [+1.19860572924150, -1.66299114873979, -2.03189643761298],
                [+2.65876001301880, +0.37736955363609, -1.23426391650599],
                [+1.50963368042358, +2.57230374419743, -0.34128058818180],
                [-1.12092277855371, +2.71045691257517, -0.25246348639234],
                [-2.60071517756218, +0.67879949508239, -1.04550707592673],
                [-2.86169588073340, +5.99660765711210, +1.08394899986031],
                [+2.09930989272956, -3.36144811062374, -2.72237695164263],
                [+2.64405246349916, +4.15317840474646, +0.27856972788526],
                [+4.69864865613751, +0.26922271535391, -1.30274048619151],
                [-4.63786461351839, +0.79856258572808, -0.96906659938432],
                [-2.57447518692275, -3.08132039046931, -2.54875517521577],
                [-5.88211879210329, 11.88491819358157, +2.31866455902233],
                [-8.18022701418703, 10.95619984550779, +1.83940856333092],
                [-5.08172874482867, 12.66714386256482, -0.92419491629867],
                [-3.18311711399702, 13.44626574330220, -0.86977613647871],
                [-5.07177399637298, 10.99164969235585, -2.10739192258756],
                [-6.35955320518616, 14.08073002965080, -1.68204314084441],
            ]
        ),
    },
    "C4H5NCS": {
        "numbers": to_number("C C C C N C S H H H H H".split()),
        "positions": torch.tensor(
            [
                [-2.56745685564671, -0.02509985979910, 0.00000000000000],
                [-1.39177582455797, +2.27696188880014, 0.00000000000000],
                [+1.27784995624894, +2.45107479759386, 0.00000000000000],
                [+2.62801937615793, +0.25927727028120, 0.00000000000000],
                [+1.41097033661123, -1.99890996077412, 0.00000000000000],
                [-1.17186102298849, -2.34220576284180, 0.00000000000000],
                [-2.39505990368378, -5.22635838332362, 0.00000000000000],
                [+2.41961980455457, -3.62158019253045, 0.00000000000000],
                [-2.51744374846065, +3.98181713686746, 0.00000000000000],
                [+2.24269048384775, +4.24389473203647, 0.00000000000000],
                [+4.66488984573956, +0.17907568006409, 0.00000000000000],
                [-4.60044244782237, -0.17794734637413, 0.00000000000000],
            ]
        ),
    },
    "LYS_xao": {
        "numbers": to_number(
            "N C C O C C H H H H H N H C H H H C O C H H H H C H H C H H N H H".split()
        ),
        "positions": torch.tensor(
            [
                [-3.08629288118877, -2.19561127309795, -0.77217893055321],
                [-1.06879606686991, -0.61608658924481, +0.23634288443356],
                [-1.53102984802615, +2.13141574530499, -0.56939167294673],
                [-0.40983742362732, +3.10958800397462, -2.34867178991298],
                [+1.47971484502131, -1.55589129675000, -0.67464620457756],
                [+3.67080881782305, -0.22160454696147, +0.62879195584726],
                [-2.83661151752225, -2.99270493895025, -2.48021099590057],
                [-1.20083247698939, -0.74777619088370, +2.28964206874229],
                [+1.58937150127564, -3.58727643425421, -0.32481723884977],
                [+3.54409665680710, -0.53132058661176, +2.66860726830112],
                [+3.50974930263916, +1.80833214591191, +0.30677605183608],
                [-3.30361007773525, +3.31485339209112, +0.83839008978212],
                [-4.34880376214547, +2.20488174175011, +1.99552260527961],
                [-4.18868179160437, +5.81154501452241, +0.15747382156164],
                [-5.61569230213706, +5.74761437268114, -1.32808061177308],
                [-2.59583748873156, +6.90533962601390, -0.53230790566849],
                [-4.98328471638590, +6.72600829931603, +1.81417357766629],
                [-5.35207705391596, -2.42418436069169, +0.40031295493727],
                [-5.84409781413938, -1.31026328950066, +2.38293392048313],
                [-7.23017418928331, -4.16191765723655, -0.85113016451490],
                [-7.68462505102300, -5.67776148707435, +0.46000324484997],
                [-6.54293162706832, -4.97137068473344, -2.60950586255879],
                [-8.95780193409875, -3.11008592668001, -1.21083256960746],
                [+1.60633375878005, -1.24066001627663, -2.71011415264373],
                [+6.22896524570903, -1.13514352758590, -0.31993549624624],
                [+6.36077259797159, -0.80721133290823, -2.35609502036389],
                [+6.39494201117646, -3.17620769071747, -0.03322800857001],
                [+8.43206186995541, +0.18178876655507, +0.98995969125296],
                [+8.26394746388151, +2.21871263547301, +0.71486715223124],
                [+8.30590057110490, -0.13981983839829, +3.02371506696762],
                [10.95680155885510, -0.57943795611462, +0.16732903266954],
                [11.15655125465616, -2.46919075399082, +0.40005961272418],
                [11.12834218297359, -0.27087148385594, -1.71428160151084],
            ]
        ),
    },
    "LYS_xao_dist": {
        "numbers": to_number(
            "N C C O C C H H H H H N H C H H H C O C H H H H C H H C H H N H H".split()
        ),
        "positions": torch.tensor(
            [
                [-2.41888912376960, -1.33181283222188, -0.87717874170716],
                [-0.92066133983480, -0.22582038216665, -0.01273108490160],
                [-0.92239610841721, +1.50792396868026, -1.40987174925180],
                [+0.00146075829434, +2.60315064900349, -2.29974945051645],
                [+1.54492858565268, -1.61165481511321, -0.98973838859437],
                [+3.93803995535626, -0.18382499849910, +0.04845068810928],
                [-3.62699292496898, -2.78150111091954, -2.82981006956948],
                [-2.14100678686951, -1.65801924558091, +1.33783160992881],
                [+1.48209141255662, -4.36903546177803, -0.12446303174623],
                [+3.84589690951951, +0.56785325181942, +3.08297100930722],
                [+3.46276627610838, +1.72559207234527, -0.58360978824655],
                [-2.56989902811457, +2.99064466003638, +0.45439031584464],
                [-5.12462355544840, +1.82051490531135, +1.77050708287538],
                [-4.15837257285700, +5.28436021503689, +0.09911235578437],
                [-5.38560418188984, +5.93215266138023, -2.07031024282113],
                [-3.06209898426073, +6.54322961884822, -0.09870795439370],
                [-5.23266108771938, +6.39708197982191, +1.90384993768122],
                [-4.47603931385884, -3.03301609920273, +0.40615883596582],
                [-7.26815123696816, -1.54763656318927, +2.33175574188923],
                [-7.35875793546559, -5.46242343858200, -1.06743258848224],
                [-7.42421048951813, -5.45296157987600, +1.10766674740164],
                [-6.84990382298311, -6.12831001256864, -2.25850995729874],
                [-8.42477322935436, -3.33802545462183, -1.22060433951378],
                [+0.70446155336862, -0.62174257171537, -2.59144001620918],
                [+5.38776791830254, -1.06978340778127, +0.48495096673209],
                [+5.67591713723773, -1.20372908522087, -1.95548292458437],
                [+4.89282219009731, -3.78336020287727, +0.58678830758817],
                [+7.59875504301919, +0.80332824475679, +0.42990324472174],
                [+7.77529514730785, +2.65584944144093, +1.44602787919426],
                [+9.85171676441603, -0.96299120502642, +3.54969934701954],
                [+11.10423101927024, +0.35599038680925, -0.57037036701742],
                [+11.08920391712721, -0.79837338285802, +0.64856534432831],
                [+12.86126624460316, +0.24428300640425, -1.29780532088312],
            ]
        ),
    },
    "C60": {
        "numbers": torch.tensor(60 * [6]),
        "positions": torch.tensor(
            [
                [-2.33619927251900, +0.00000000000000, +6.29274336133455],
                [-0.72192527745476, -2.22185754149319, +6.29274336133455],
                [+1.89002491371426, -1.37318347880308, +6.29274336133455],
                [+1.89002491371426, +1.37318347880308, +6.29274336133455],
                [-0.72192527745476, +2.22185754149319, +6.29274336133455],
                [+4.43015231296584, -4.91604218629124, +1.12371070968761],
                [+6.04442630803008, -2.69418464479805, +1.12371070968761],
                [+5.59825194922534, -1.32100116599497, +3.45990998220660],
                [+3.70822703551107, -2.69418464479805, +4.90376053711613],
                [+2.98630175805632, -4.91604218629124, +3.45990998220660],
                [+6.04442630803008, +2.69418464479805, +1.12371070968761],
                [+4.43015231296584, +4.91604218629124, +1.12371070968761],
                [+2.98630175805632, +4.91604218629124, +3.45990998220660],
                [+3.70822703551107, +2.69418464479805, +4.90376053711613],
                [+5.59825194922534, +1.32100116599497, +3.45990998220660],
                [-0.69449141210920, +6.58113986874450, +1.12371070968761],
                [-3.30644160327823, +5.73246580605438, +1.12371070968761],
                [-3.75261596208297, +4.35928232725129, +3.45990998220660],
                [-1.41641668956397, +4.35928232725129, +4.90376053711613],
                [+0.47360822415029, +5.73246580605438, +3.45990998220660],
                [-6.47364560560848, +1.37318347880308, +1.12371070968761],
                [-6.47364560560848, -1.37318347880308, +1.12371070968761],
                [-5.30554596934898, -2.22185754149319, +3.45990998220660],
                [-4.58362069189423, +0.00000000000000, +4.90376053711613],
                [-5.30554596934898, +2.22185754149319, +3.45990998220660],
                [-3.30644160327823, -5.73246580605438, +1.12371070968761],
                [-0.69449141210920, -6.58113986874450, +1.12371070968761],
                [+0.47360822415029, -5.73246580605438, +3.45990998220660],
                [-1.41641668956397, -4.35928232725129, +4.90376053711613],
                [-3.75261596208297, -4.35928232725129, +3.45990998220660],
                [-0.47360822415029, -5.73246580605438, -3.45990998220660],
                [+1.41641668956397, -4.35928232725129, -4.90376053711613],
                [+3.75261596208297, -4.35928232725129, -3.45990998220660],
                [+3.30644160327823, -5.73246580605438, -1.12371070968761],
                [+0.69449141210920, -6.58113986874450, -1.12371070968761],
                [+5.30554596934897, -2.22185754149319, -3.45990998220660],
                [+4.58362069189423, +0.00000000000000, -4.90376053711613],
                [+5.30554596934897, +2.22185754149319, -3.45990998220660],
                [+6.47364560560848, +1.37318347880308, -1.12371070968761],
                [+6.47364560560848, -1.37318347880308, -1.12371070968761],
                [+3.75261596208297, +4.35928232725129, -3.45990998220660],
                [+1.41641668956397, +4.35928232725129, -4.90376053711613],
                [-0.47360822415029, +5.73246580605438, -3.45990998220660],
                [+0.69449141210920, +6.58113986874450, -1.12371070968761],
                [+3.30644160327823, +5.73246580605438, -1.12371070968761],
                [-2.98630175805631, +4.91604218629124, -3.45990998220660],
                [-3.70822703551107, +2.69418464479805, -4.90376053711613],
                [-5.59825194922534, +1.32100116599497, -3.45990998220660],
                [-6.04442630803008, +2.69418464479805, -1.12371070968761],
                [-4.43015231296584, +4.91604218629124, -1.12371070968761],
                [-5.59825194922534, -1.32100116599497, -3.45990998220660],
                [-3.70822703551107, -2.69418464479805, -4.90376053711613],
                [-2.98630175805631, -4.91604218629124, -3.45990998220660],
                [-4.43015231296584, -4.91604218629124, -1.12371070968761],
                [-6.04442630803008, -2.69418464479805, -1.12371070968761],
                [-1.89002491371426, -1.37318347880308, -6.29274336133455],
                [-1.89002491371426, +1.37318347880308, -6.29274336133455],
                [+0.72192527745475, +2.22185754149319, -6.29274336133455],
                [+2.33619927251900, +0.00000000000000, -6.29274336133455],
                [+0.72192527745475, -2.22185754149319, -6.29274336133455],
            ]
        ),
    },
    "vancoh2": {
        "numbers": to_number(
            "h c c c c c c o c c c c c c o c h c h o c c h h c n h h c h h h c o h h c c h h h h o h c h o h c o h h c c o h h h o h o c c c c c c h h c o h h c n c o c c c c c c c h h o h c c c c c c o h h o h h c n c o h c o o h h h n c o c n c o c n c o c n c o c n c h h h h c h h c c h h h c h h h h h h h c o h h h c c o n h h h h h h h h h h h h h h h h h h".split()
        ),
        "positions": torch.tensor(
            [
                [-4.12538769921434, -3.21158183919580, -3.19682767130024],
                [-6.02009769693417, -2.74628139051134, -2.59279580890343],
                [-7.53961002992393, -1.20824253149235, -4.05346489611364],
                [-9.93839752891803, -0.51684590202428, -3.22140336743590],
                [-10.89371868077148, -1.63730630215238, -1.04863154445230],
                [-9.38211933998691, -3.17163882595432, 0.43565272218101],
                [-6.89776628094592, -3.61056092284554, -0.26809604767662],
                [-5.30602747700687, -4.86701842960269, 1.38286103491508],
                [-3.14746403023824, -3.62614664239547, 2.04213305448606],
                [-2.82249744649441, -1.05043432150309, 1.59989972989473],
                [-0.56661707604629, 0.11333153069150, 2.17425742826420],
                [1.31538264982188, -1.23829051671937, 3.38822900541345],
                [0.99795536530488, -3.78592717369249, 3.93985523751875],
                [-1.22557431398636, -5.02429244738392, 3.20408142294163],
                [-1.43457398764986, -7.58314508747779, 3.62079804314904],
                [-1.56088433519223, -9.06017517767015, 1.35190146479088],
                [-2.49525974148085, -7.95566385415148, -0.13677988729630],
                [1.14653566631218, -9.66944865536715, 0.46029795026662],
                [2.30779288503030, -7.97119372603625, 0.74223530488766],
                [1.00144274364859, -10.21047371395700, -2.15749244501547],
                [3.07306317160088, -9.30996551813451, -3.62232092064510],
                [2.42241518774420, -9.57284969127321, -6.42191721388004],
                [1.66185842356912, -11.45089401156576, -6.76976263238613],
                [4.17842367112950, -9.35309233477430, -7.47757416924928],
                [0.54242158810500, -7.57761404823026, -7.38911159183500],
                [0.58042084144901, -7.71034311122052, -10.13597995678340],
                [-0.73017220062605, -6.51498652030035, -10.86468981623193],
                [2.29541701936119, -7.13964394855851, -10.77633665922019],
                [-2.15573033333502, -8.18120460180402, -6.54409994776056],
                [-2.24681515763441, -8.57209047538204, -4.53503618653510],
                [-2.78423571683368, -9.85274813558471, -7.55709768931847],
                [-3.42812477844995, -6.62935808843030, -6.99930470776903],
                [1.38614393354765, -4.91857904981788, -6.44151377123438],
                [3.46987623337461, -4.15333923353392, -7.89981359636450],
                [3.81644190492405, -2.39097181666417, -7.55342443037267],
                [-0.19360585027358, -3.58226478184608, -6.72685217505694],
                [1.97486326942205, -4.96345102669530, -3.59012572401289],
                [3.12280304388212, -2.50234102440084, -2.63956501038029],
                [4.99168119014415, -2.28010814336697, -3.45824340366162],
                [3.32805336856430, -2.59093715015863, -0.60001350316477],
                [1.92952991415854, -0.90462517348401, -3.12318061921686],
                [0.18608502135866, -5.33248263028990, -2.58841441794475],
                [3.76425781168699, -6.86654228057390, -2.96233421011170],
                [4.75707597112518, -10.45247350665901, -3.17598873760725],
                [2.32550268345175, -11.81444356180777, 2.02196698083523],
                [3.99248733984138, -12.53184710205960, 1.01092574974729],
                [3.19338890689257, -10.93067774001129, 4.36532012389965],
                [2.15905525600708, -9.52868802893586, 4.92192645995491],
                [0.45683124395131, -14.04763618136973, 2.23996725971216],
                [0.30311759153612, -15.31604864318488, -0.08759416192854],
                [-0.19042399197421, -14.10137434832007, -1.36362350974563],
                [1.21198674166504, -15.41481972917823, 3.59648314916987],
                [-2.17657098762293, -13.20386414776909, 3.12281127975626],
                [-2.38093033342728, -12.66150846293209, 5.97175864878338],
                [-4.81937706534524, -11.83064705646286, 6.59037417761133],
                [-5.23254430994185, -10.48047378079703, 5.43061531540030],
                [-2.05724277598030, -14.40722997904393, 7.02289869916862],
                [-0.97339682985422, -11.25762263891393, 6.55633888011614],
                [-3.10605556698939, -11.12043842727380, 1.66973430522971],
                [-3.52296310506918, -14.70894975773220, 2.65602762996701],
                [2.82422292196965, -5.20194479760726, 5.07591963832577],
                [4.99972480224581, -3.84440517073019, 5.65165822138973],
                [5.07947413613981, -2.38420005784892, 7.83055172684290],
                [7.06202949713153, -0.72226792057295, 8.18400595214435],
                [8.96841505109351, -0.49641483892247, 6.38394670128930],
                [8.96387042224573, -2.11830451251091, 4.31538629731478],
                [6.98344443722817, -3.78471111374411, 3.93886715530369],
                [6.92311156237639, -4.97678322541409, 2.28504922834613],
                [10.45968289088891, -2.02787855201256, 2.93398371568008],
                [10.91870925103756, 1.58818848599210, 6.68370054467061],
                [13.41326755764172, 0.78511332957411, 6.41693471614361],
                [13.66234685647971, 0.28105127994951, 4.66948984575125],
                [10.78474942669380, 2.31437541368890, 8.62150757854088],
                [10.31488178355000, 3.83227894053164, 4.86367742757487],
                [7.62551800650366, 4.38750951139158, 4.80082808265715],
                [6.74859648827824, 6.71827117608407, 4.29236970909342],
                [8.09796397361027, 8.57745165986264, 4.20353396374424],
                [3.96261968566901, 6.91914126424039, 3.50683791359349],
                [4.10050956874645, 6.80103803589393, 0.64804557780660],
                [5.54786279219640, 4.98757343632648, -0.58943381880572],
                [6.07542897485738, 5.15007691018714, -3.15068524388173],
                [5.01988499730318, 7.12690463903716, -4.56531063118022],
                [3.47606105481541, 8.88098211272767, -3.35111552927876],
                [3.05564019026471, 8.74170695734120, -0.78113132099485],
                [1.92765177524478, 10.17757698047684, 0.13306135117373],
                [2.70731369574914, 10.40427087572663, -4.46842014829144],
                [5.46101617264093, 7.44376345861709, -7.04658880430021],
                [6.80650489276129, 6.34197000565844, -7.61670822772278],
                [7.82745318080644, 3.28601416826792, -4.29700993731229],
                [10.38690846608574, 3.28883954852532, -3.63359391813343],
                [12.01109470081403, 1.49468161381780, -4.61924566704241],
                [11.12613203555479, -0.32743692431837, -6.29689662527737],
                [8.60978655493948, -0.30196136897065, -7.02648811827772],
                [6.95867864499421, 1.48213753855579, -6.03314587201915],
                [4.52502909819850, 1.37951500494576, -6.82893732154513],
                [3.40572120726528, 2.41871864611720, -5.80427345938205],
                [7.93282646774051, -1.69166616438693, -8.35133234847149],
                [12.64972915480483, -2.12016315164210, -7.28183692086886],
                [14.33237673978119, -1.97054621179221, -6.59383793434962],
                [13.97891101912674, 1.50259271748724, -4.07692693770869],
                [11.49248142097661, 5.40682268200406, -2.03861056185756],
                [10.80738132397245, 5.36857656861684, 0.61406866027980],
                [11.29104362284186, 3.40004682351786, 2.13660334773830],
                [12.35970444097817, 1.48810873781552, 1.43955737097236],
                [10.16390605102951, 7.01480263716682, 1.35014609733861],
                [10.76976279107487, 7.91727042343226, -3.24710293842966],
                [10.68702963275411, 8.24274025492312, -5.49167297439561],
                [10.30311968764884, 9.73382356311540, -1.56954000700145],
                [9.89745825581469, 11.31451106508548, -2.40576691203645],
                [13.56387929377399, 5.26583032863039, -2.19508168830943],
                [6.32211583558799, 3.39873930815410, 0.42913972305930],
                [2.29338770220927, 5.16051041461001, 4.83057025382569],
                [0.00563492503721, 4.25029884799799, 4.02320437948733],
                [-1.82230056528111, 4.26248177170781, 5.39395336856523],
                [-0.03124431262372, 2.84903921941416, 1.49456902192360],
                [-1.85605209494782, 3.90958127976028, -0.23666451019758],
                [-1.38398497536428, 4.01605119317095, -2.71867380077511],
                [0.57513542510151, 3.25418447603116, -3.67276899724561],
                [-3.44948010497427, 5.21240447767652, -4.34780134549938],
                [-5.92543981161170, 4.19013306694201, -3.77789824365942],
                [-8.05264127335910, 5.08119310657999, -4.89176522456555],
                [-8.01391061396359, 6.54956758548931, -6.64869873859410],
                [-10.49633983217443, 4.19837069042028, -3.64992250038762],
                [-10.15224576580139, 4.35782494802021, -0.92461731157512],
                [-11.92938770015905, 3.92146805663835, 0.85149517006631],
                [-14.00955268507380, 3.04455806597626, 0.39508305711487],
                [-11.03217075988150, 4.54541150618380, 3.53840992074204],
                [-10.33544415191386, 7.17418268718763, 3.87727504523588],
                [-12.30524929318105, 8.95338692725477, 3.22029492245233],
                [-13.98427959388165, 8.51952437957154, 4.33127631397334],
                [-11.67144207987738, 10.85013365288306, 3.70945784163437],
                [-12.84346480280234, 8.92979821427431, 1.21130319450921],
                [-8.71003994391634, 7.57393729259373, 2.93797832874096],
                [-8.82114733289762, 2.78295617271263, 4.20955223937224],
                [-7.05383488400150, 3.63671188177863, 3.57132424245791],
                [-9.08382427593198, 0.99807112674712, 3.20292223656293],
                [-8.57327146865768, 2.21370625747439, 7.03765901663534],
                [-7.81074946992166, 4.57317320363496, 8.50474082318675],
                [-9.12709882208965, 6.10424197857600, 8.14349632109357],
                [-7.77319891440508, 4.17973545879407, 10.52132231744340],
                [-5.93338640588897, 5.16473227991146, 7.91777827210490],
                [-6.60524859327706, 0.13713197881133, 7.42839720797189],
                [-4.76880746167127, 0.78194909544825, 6.76791678932449],
                [-6.44964558315062, -0.33152882228249, 9.42233869793237],
                [-7.10930120991543, -1.57186564084265, 6.40263330080602],
                [-10.40339365831910, 1.53926292541477, 7.73189500826730],
                [-12.63983090818957, 4.16202867514895, 4.78912105041731],
                [-8.60316631654959, 5.32279046391419, -0.34644347901924],
                [-11.99164013009787, 5.50632136703657, -4.26443491592593],
                [-11.38267545719506, 1.52666501801739, -4.60878513708355],
                [-13.99285329462420, 1.31254511075970, -4.45046475030065],
                [-14.50601469359559, 1.78231892756185, -2.73132053488874],
                [-10.93046995599579, 1.43913888476886, -6.63181143026238],
                [-6.07014886462780, 2.82649254713678, -2.45601855996842],
                [-3.43721667472228, 8.11312702759942, -4.03348383965781],
                [-4.79573146524593, 9.02568146378643, -1.68527965200730],
                [-5.56732106028568, 7.64016117857187, -0.00276297360742],
                [-5.10641608874635, 11.53533709716254, -1.57000720448867],
                [-6.05977558010407, 12.28633392191823, -0.10151487007323],
                [-4.59948378277943, 12.67794181785443, -2.99941084163650],
                [-1.48584183667929, 8.78121941409039, -4.00796046183135],
                [-4.39616214850520, 8.93016601049508, -5.66844693731519],
                [-2.96530288427253, 4.80398499499406, -6.32474878116254],
                [-3.41030746956679, 4.78395951899312, 0.48223484689858],
                [1.80971446093211, 2.92643909363068, 0.56467819564263],
                [2.16486052529519, 5.67271092567087, 6.67989712895833],
                [3.37533969710816, 8.85500145834930, 3.99001446473051],
                [6.41875138538940, 2.91053314260861, 4.88645167716006],
                [11.27580608944315, 5.52912476704497, 5.56899837637730],
                [7.09907338587011, 0.45579147413627, 9.85029678323477],
                [3.54947217624135, -2.50031979131901, 9.17336234467577],
                [3.06242791653076, -0.30102621361350, 3.85606934835183],
                [-4.34337383748494, 0.02084092992115, 0.77713774458883],
                [-10.08639436277542, -3.97269062183610, 2.17249015944311],
                [-12.80870839912841, -1.26067715243202, -0.46347194114657],
                [-6.84434849946216, -0.50808148356371, -5.84053973638459],
            ]
        ),
    },
    "br2nh3": {
        "numbers": to_number(["Br", "Br", "N", "H", "H", "H"]),
        "positions": torch.tensor(
            [
                [+0.00000000000000, +0.00000000000000, +3.11495251300000],
                [+0.00000000000000, +0.00000000000000, -1.25671880600000],
                [+0.00000000000000, +0.00000000000000, -6.30201130100000],
                [+0.00000000000000, +1.78712709700000, -6.97470840000000],
                [-1.54769692500000, -0.89356260400000, -6.97470840000000],
                [+1.54769692500000, -0.89356260400000, -6.97470840000000],
            ],
        ),
    },
    "br2nh2o": {
        "numbers": to_number(["Br", "Br", "N", "H", "H", "O"]),
        "positions": torch.tensor(
            [
                [+0.00000000000000, +0.00000000000000, +3.11495251300000],
                [+0.00000000000000, +0.00000000000000, -1.25671880600000],
                [+0.00000000000000, +0.00000000000000, -6.30201130100000],
                [+0.00000000000000, +1.78712709700000, -6.97470840000000],
                [-1.54769692500000, -0.89356260400000, -6.97470840000000],
                [+1.54769692500000, -0.89356260400000, -6.97470840000000],
            ]
        ),
    },
    "br2och2": {
        "numbers": to_number(["Br", "Br", "O", "C", "H", "H"]),
        "positions": torch.tensor(
            [
                [-1.78533374700000, -3.12608299900000, +0.00000000000000],
                [+0.00000000000000, +0.81604226400000, +0.00000000000000],
                [+2.65828699900000, +5.29707580600000, +0.00000000000000],
                [+4.88597158600000, +4.86116137300000, +0.00000000000000],
                [+5.61550975300000, +2.90822215900000, +0.00000000000000],
                [+6.28907612600000, +6.39963643500000, +0.00000000000000],
            ],
        ),
    },
    "finch": {
        "numbers": to_number(["F", "I", "N", "C", "H"]),
        "positions": torch.tensor(
            [
                [0.00000000000000, 0.00000000000000, 4.37637862700000],
                [0.00000000000000, 0.00000000000000, 0.69981844700000],
                [0.00000000000000, 0.00000000000000, -4.24181123900000],
                [0.00000000000000, 0.00000000000000, -6.39520691700000],
                [0.00000000000000, 0.00000000000000, -8.41387269200000],
            ],
        ),
    },
    "tmpda": {
        "numbers": to_number(
            "C C C C C C C C C C C C C C C C C C C C C C C C F F F F F F F F F F F F F F F I I I N C C C H H H H H H C H H C H H N H H C C H H H H H H".split()
        ),
        "positions": torch.tensor(
            [
                [+0.5290878, +1.2013531, +2.2700441],
                [+1.3161209, +0.0507214, +2.3239534],
                [+0.6223940, -1.1595194, +2.3736772],
                [-0.7685078, -1.2647904, +2.3639976],
                [-1.4744377, -0.0650938, +2.3059391],
                [-0.8667556, +1.1909427, +2.2412522],
                [-1.6478959, +2.4482878, +2.1800916],
                [-1.6343150, +3.3163383, +3.2738759],
                [-2.3370934, +4.5191279, +3.2717126],
                [-3.0798735, +4.8731320, +2.1492348],
                [-3.1090458, +4.0231925, +1.0444267],
                [-2.4006738, +2.8250420, +1.0521578],
                [-1.4554005, -2.5773075, 2.4728359],
                [-1.8794846, -3.3087945, 1.3501617],
                [-2.5160427, -4.5322786, 1.5338947],
                [-2.7396783, -5.0486298, 2.8091412],
                [-2.3210977, -4.3306654, 3.9255251],
                [-1.6849255, -3.1054668, 3.7437103],
                [2.7974930, 0.1052463, 2.3657096],
                [3.5738583, 0.5124953, 1.2651516],
                [4.9590315, 0.5446458, 1.3823259],
                [5.5966983, 0.1764207, 2.5658240],
                [4.8362413, -0.2339276, 3.6567451],
                [3.4489314, -0.2648897, 3.5439542],
                [1.1486090, 2.3979578, 2.2468071],
                [-2.8213958, -0.1191155, 2.3332240],
                [1.3392361, -2.2986085, 2.4302239],
                [-0.9371683, 2.9951050, 4.3780669],
                [-2.3073302, 5.3287079, 4.3380634],
                [-3.7593254, 6.0274999, 2.1330375],
                [-3.8292371, 4.4026522, -0.0250375],
                [-2.9374890, -5.2640344, 0.4827576],
                [-3.3540400, -6.2307782, 2.9684280],
                [-2.5319117, -4.8175816, 5.1572286],
                [-1.2905847, -2.4258816, 4.8393372],
                [5.7366659, 0.9258618, 0.3483358],
                [6.9347859, 0.2114601, 2.6604289],
                [5.4381467, -0.5873956, 4.8021967],
                [2.7349247, -0.6534853, 4.6193825],
                [-2.4600738, 1.6175327, -0.6577418],
                [-1.5804839, -2.6131434, -0.6415977],
                [2.6992284, 1.0429354, -0.6072158],
                [1.7595400, 1.8721176, -3.1125965],
                [2.8105299, 1.4358390, -4.0406870],
                [1.7389539, 3.3380404, -3.0161015],
                [0.4310606, 1.3606491, -3.4932700],
                [2.8666704, 0.3456466, -4.0583812],
                [2.6236843, 1.8022505, -5.0651462],
                [3.7727965, 1.8214787, -3.6950512],
                [0.9906252, 3.6392908, -2.2786620],
                [2.7200177, 3.6895915, -2.6861906],
                [1.4983812, 3.8101467, -3.9842083],
                [0.2587515, -0.1356473, -3.2321337],
                [-0.3064854, 1.9132184, -2.9021733],
                [+0.2318430, +1.5866051, -4.5581650],
                [-1.0877227, -0.6121098, -3.7724130],
                [+1.0701060, -0.6941223, -3.7091894],
                [+0.3161864, -0.3289624, -2.1556392],
                [-1.4190038, -1.9979209, -3.4038086],
                [-1.8898030, +0.0215661, -3.3729522],
                [-1.1108075, -0.4956079, -4.8731789],
                [-2.7797088, -2.3259584, -3.8496917],
                [-0.4558655, -2.9689092, -3.9390583],
                [-3.0333933, -3.3362405, -3.5190230],
                [-3.4863100, -1.6233648, -3.3994733],
                [-2.8747529, -2.2735036, -4.9480640],
                [-0.7761393, -3.9740022, -3.6542041],
                [-0.3889685, -2.9144663, -5.0399768],
                [0.5326283, -2.7908957, -3.5118282],
            ]
        ),
    },
    "tmpda_mod": {
        "numbers": to_number(
            "c c c c c c c c c c c c c c c c c c c c c c c c f f f f f f f f f f f f f f f i i br n c c c h h h h h h c h h c o h n h h c c h h h h h h h".split()
        ),
        "positions": torch.tensor(
            [
                [+0.99983141582600, +2.27022814899754, +4.28976145084998],
                [+2.48710823686859, +0.09584879876714, +4.39163469650243],
                [+1.17615420161033, -2.19117334629995, +4.48559943832332],
                [-1.45226964458390, -2.39010670516543, +4.46730877916307],
                [-2.78628400774097, -0.12300983235639, +4.35759317009342],
                [-1.63793145687613, +2.25055610004019, +4.23535245626976],
                [-3.11407212186631, +4.62659379420778, +4.11977680648764],
                [-3.08840775136777, +6.26697055668918, +6.18672900598532],
                [-4.41646569778001, +8.53991424212781, +6.18264152837775],
                [-5.82011835835567, +9.20888484914983, +4.06146552746007],
                [-5.87524544886325, +7.60273291651153, +1.97368098716452],
                [-4.53661637450985, +5.33855567056503, +1.98829045983400],
                [-2.75030929150647, -4.87040436908087, +4.67298279111509],
                [-3.55171190534227, -6.25271446280500, +2.55143640387698],
                [-4.75463218778180, -8.56476603039276, +2.89864145393285],
                [-5.17724108966248, -9.54052800456940, +5.30850713545736],
                [-4.38623952841663, -8.18377078750246, +7.41816714537158],
                [-3.18404868026120, -5.86848211906322, +7.07458659002274],
                [+5.28649560555772, +0.19888611571236, +4.47054399028843],
                [+6.75361282830281, +0.96847519024008, +2.39079078602254],
                [+9.37121232325518, +1.02923177487293, +2.61221755494944],
                [10.57622642224080, +0.33338737263260, +4.84870464399179],
                [+9.13917096268626, -0.44205985288146, +6.91024655759466],
                [+6.51753501273168, -0.50056955315212, +6.69710245827200],
                [+2.17055643428028, +4.53148387835536, +4.24584988489205],
                [-5.33166572911465, -0.22509472733480, +4.40915434740384],
                [+2.53078925623932, -4.34373958787379, +4.59245778149254],
                [-1.77099085276328, +5.65992816449727, +8.27334758526197],
                [-4.36022177913278, 10.06979871810234, +8.19775098137244],
                [-7.10409466345877, 11.39032421618183, +4.03085574369338],
                [-7.23620919628361, +8.31980650203590, -0.04731496270838],
                [-5.55104970410083, -9.94758257071829, +0.91228040447209],
                [-6.33821701103982, -11.77446396334351, 5.60951594067063],
                [-4.78462025165348, -9.10391056292687, +9.74575037197764],
                [-2.43885219055015, -4.58425259065941, +9.14502155476810],
                [10.84072760845242, +1.74962560919826, +0.65825963934764],
                [13.10484627288905, +0.39960148631337, +5.02748218401201],
                [10.27660845545526, -1.11001756670068, +9.07483712649950],
                [+5.16825922139213, -1.23490767655107, +8.72936873475216],
                [-4.64886610631262, +3.05669436754430, -1.24295224066360],
                [-2.98668190435304, -4.93812459448296, -1.21244450210764],
                [+5.10080166792137, +1.97086151578658, -1.14747193849076],
                [+3.32504870532403, +3.53779029298215, -5.88195397660567],
                [+5.31113196504447, +2.71334246905654, -7.63579178533573],
                [+3.28614680332248, +6.30798139304581, -5.69960485420991],
                [+0.81458723300731, +2.57125396174593, -6.60132357937146],
                [+5.41722118968096, +0.65317816579852, -7.66922859938486],
                [+4.95804419756264, +3.40575890810679, -9.57173872124372],
                [+7.12955305380973, +3.44209645175722, -6.98263440652458],
                [+1.87200994220740, +6.87726327781544, -4.30604711059201],
                [+5.14008907405234, +6.97231839161025, -5.07616530843462],
                [+2.83152972034289, +7.20013432456451, -7.52906194354299],
                [+0.48897041419917, -0.25633567962711, -6.10784805809119],
                [-0.57917271130593, +3.61545803670427, -5.48431213628355],
                [+0.43811977391161, +2.99824891796187, -8.61368348085482],
                [-2.05549856945632, -1.15672025814468, -7.12882739897787],
                [+2.02220726431878, -1.31170047707749, -7.00935135447454],
                [+0.59750494444092, -0.62164808540914, -4.07356733356217],
                [-2.68152892974847, -3.77552350863844, -6.43226679053632],
                [-3.57121009949615, +0.04075383360368, -6.37395551150874],
                [-2.09912100731718, -0.93656338517353, -9.20897366627769],
                [-5.25288871615737, -4.39542359738231, -7.27486354416283],
                [-0.86145999980253, -5.61042489893657, -7.44374080901614],
                [-5.73228199835694, -6.30458177574760, -6.64998969625895],
                [-6.58817108554407, -3.06771525030311, -6.42407293805994],
                [-5.43249584594631, -4.29629990324119, -9.35048580712029],
                [-1.46669014464092, -7.50977539871506, -6.90544476351199],
                [-0.73504299124344, -5.50754253953357, -9.52417620441302],
                [+1.00652104630717, -5.27402908231356, -6.63639311679227],
                [10.97892875661261, -1.03282225450972, -8.78226972788494],
            ]
        ),
    },
    "Ag2Cl22-": {
        "numbers": to_number("Ag Ag Cl Cl Cl Cl".split()),
        "positions": torch.tensor(
            [
                [+0.00000000000000, -3.10725336044816, +0.00000000000000],
                [+0.00000000000000, +3.10725336044816, +0.00000000000000],
                [-3.35382227578130, +0.00000000000000, +0.00000000000000],
                [+0.00000000000000, -7.48233203321965, +0.00000000000000],
                [+0.00000000000000, +7.48233203321965, +0.00000000000000],
                [+3.35382227578130, +0.00000000000000, +0.00000000000000],
            ],
        ),
    },
    "Al3+Ar6": {
        "numbers": to_number("Al Ar Ar Ar Ar Ar Ar".split()),
        "positions": torch.tensor(
            [
                [+0.00000000000000, +0.00000000000000, +0.00000000000000],
                [-4.83361416932971, +0.00000000000000, -0.00000000000000],
                [+4.83361416932971, -0.00000000000000, +0.00000000000000],
                [+0.00000000000000, -0.00000000000000, -4.83361416932971],
                [+0.00000000000000, -0.00000000000000, +4.83361416932971],
                [+0.00000000000000, +4.83361416932971, +0.00000000000000],
                [-0.00000000000000, -4.83361416932971, +0.00000000000000],
            ],
        ),
    },
    "AD7en+": {
        "numbers": to_number(
            "c c c c c c c h h h h h h h h h h h h h h h h c c c c c c c h h h h h h h h h".split()
        ),
        "positions": torch.tensor(
            [
                [-3.20868398218451, +0.81338576920721, +0.00000000000000],
                [-2.74527806469722, -0.72094646635770, +2.39932976171515],
                [-2.74527806469722, -0.72094646635770, -2.39932976171515],
                [-3.08987706868245, +0.88032712755882, +4.78050218429059],
                [-3.08987706868245, +0.88032712755882, -4.78050218429059],
                [-2.24991924121552, -0.49947849598937, +7.15458157802629],
                [-2.24991924121552, -0.49947849598937, -7.15458157802629],
                [-1.96613220283514, +2.46542593163266, +0.00000000000000],
                [-5.14977393132991, +1.50790279463433, +0.00000000000000],
                [-0.82880270075724, -1.49583753208091, +2.35732742755509],
                [-4.05166617831319, -2.31485469244108, +2.45612685682732],
                [-0.82880270075724, -1.49583753208091, -2.35732742755509],
                [-4.05166617831319, -2.31485469244108, -2.45612685682732],
                [-2.02552326389203, +2.64082825102477, +4.58976889067168],
                [-5.07842495481751, +1.39500213203172, +4.96726265317654],
                [-2.02552326389203, +2.64082825102477, -4.58976889067168],
                [-5.07842495481751, +1.39500213203172, -4.96726265317654],
                [-0.19533746504355, -0.83162969141366, +7.10835455754267],
                [-3.17654001815395, -2.32666104233015, +7.30973305003174],
                [-2.66710444394322, +0.60519698375962, +8.83560037797207],
                [-0.19533746504355, -0.83162969141366, -7.10835455754267],
                [-3.17654001815395, -2.32666104233015, -7.30973305003174],
                [-2.66710444394322, +0.60519698375962, -8.83560037797207],
                [+4.44203589533023, -0.61191376271149, +0.00000000000000],
                [+3.87926904943266, +0.51011020933180, -2.29251661260312],
                [+3.87926904943266, +0.51011020933180, +2.29251661260312],
                [+4.24448794558471, -0.68450742413506, -4.64062864339737],
                [+4.24448794558471, -0.68450742413506, +4.64062864339737],
                [+3.52165215472345, +0.39151447005582, -6.85288366051549],
                [+3.52165215472345, +0.39151447005582, +6.85288366051549],
                [+5.27500886512691, -2.48123073761105, +0.00000000000000],
                [+3.05171150894116, +2.38307971640241, -2.29573309948118],
                [+3.05171150894116, +2.38307971640241, +2.29573309948118],
                [+5.06297978890177, -2.55837272686144, -4.64288433189127],
                [+5.06297978890177, -2.55837272686144, +4.64288433189127],
                [+2.78315503816840, +2.29140452560769, -6.91431707203879],
                [+3.86656025085084, -0.52052884030713, -8.64049586076960],
                [+2.78315503816840, +2.29140452560769, +6.91431707203879],
                [+3.86656025085084, -0.52052884030713, +8.64049586076960],
            ],
        ),
    },
    "C2H4F+": {
        "numbers": to_number("C C F H H H H".split()),
        "positions": torch.tensor(
            [
                [-0.00000000000000, +1.38825959869068, -0.28096500207358],
                [-0.00000000000000, -1.38825959869068, -0.28096500207358],
                [+0.00000000000000, +0.00000000000000, +2.33689103853566],
                [-1.78408589032842, +2.42537442958257, -0.44374025859713],
                [+1.78408589032842, +2.42537442958257, -0.44374025859713],
                [-1.78408589032842, -2.42537442958257, -0.44374025859713],
                [+1.78408589032842, -2.42537442958257, -0.44374025859713],
            ],
        ),
    },
    "ZnOOH-": {
        "numbers": to_number("Zn O O H".split()),
        "positions": torch.tensor(
            [
                [-0.30631629283878, -1.11507514203552, +0.00000000000000],
                [-0.06543072660074, -4.32862093666082, +0.00000000000000],
                [-0.64012239724097, +2.34966763895920, +0.00000000000000],
                [+1.01186941668051, +3.09402843973713, +0.00000000000000],
            ],
        ),
    },
}
