"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InstrumentedServiceMultiAZObservability = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const CanaryMetrics_1 = require("./CanaryMetrics");
const Operation_1 = require("./Operation");
const OperationMetricDetails_1 = require("./OperationMetricDetails");
const MetricDimensions_1 = require("./props/MetricDimensions");
const OperationAlarmsAndRules_1 = require("../alarmsandrules/OperationAlarmsAndRules");
const ServiceAlarmsAndRules_1 = require("../alarmsandrules/ServiceAlarmsAndRules");
const AvailabilityZoneMapper_1 = require("../azmapper/AvailabilityZoneMapper");
const CanaryFunction_1 = require("../canaries/CanaryFunction");
const CanaryTest_1 = require("../canaries/CanaryTest");
const OperationAvailabilityAndLatencyDashboard_1 = require("../dashboards/OperationAvailabilityAndLatencyDashboard");
const ServiceAvailabilityAndLatencyDashboard_1 = require("../dashboards/ServiceAvailabilityAndLatencyDashboard");
const OutlierDetectionFunction_1 = require("../outlier-detection/OutlierDetectionFunction");
const OutlierDetectionAlgorithm_1 = require("../utilities/OutlierDetectionAlgorithm");
const StackWithDynamicSource_1 = require("../utilities/StackWithDynamicSource");
/**
 * An service that implements its own instrumentation to record
 * availability and latency metrics that can be used to create
 * alarms, rules, and dashboards from
 */
class InstrumentedServiceMultiAZObservability extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        let outlierThreshold;
        if (!props.outlierThreshold) {
            switch (props.outlierDetectionAlgorithm) {
                case OutlierDetectionAlgorithm_1.OutlierDetectionAlgorithm.CHI_SQUARED:
                    outlierThreshold = 0.05;
                    break;
                case OutlierDetectionAlgorithm_1.OutlierDetectionAlgorithm.IQR:
                    outlierThreshold = 1.5;
                    break;
                case OutlierDetectionAlgorithm_1.OutlierDetectionAlgorithm.MAD:
                    outlierThreshold = 3;
                    break;
                case OutlierDetectionAlgorithm_1.OutlierDetectionAlgorithm.STATIC:
                    outlierThreshold = 0.7;
                    break;
                case OutlierDetectionAlgorithm_1.OutlierDetectionAlgorithm.Z_SCORE:
                    outlierThreshold = 2;
                    break;
            }
        }
        else {
            outlierThreshold = props.outlierThreshold;
        }
        this.azMapper = new AvailabilityZoneMapper_1.AvailabilityZoneMapper(this, 'AZMapper', {
            availabilityZoneNames: props.service.availabilityZoneNames,
        });
        if (props.service.canaryTestProps !== undefined) {
            let canaryStack = new StackWithDynamicSource_1.StackWithDynamicSource(this, 'Canary', {
                assetsBucketsParameterName: props.assetsBucketParameterName,
                assetsBucketPrefixParameterName: props.assetsBucketPrefixParameterName,
            });
            let canary = new CanaryFunction_1.CanaryFunction(canaryStack, 'CanaryFunction', {
                vpc: props.service.canaryTestProps.networkConfiguration?.vpc,
                subnetSelection: props.service.canaryTestProps.networkConfiguration?.subnetSelection,
                httpTimeout: props.service.canaryTestProps.timeout
                    ? props.service.canaryTestProps.timeout
                    : aws_cdk_lib_1.Duration.seconds(2),
                ignoreTlsErrors: props.service.canaryTestProps.ignoreTlsErrors
                    ? props.service.canaryTestProps.ignoreTlsErrors
                    : false,
            });
            this.canaryLogGroup = canary.logGroup;
            for (let i = 0; i < props.service.operations.length; i++) {
                let operation = props.service.operations[i];
                if (operation.optOutOfServiceCreatedCanary != true) {
                    let testProps = operation.canaryTestProps
                        ? operation.canaryTestProps
                        : props.service.canaryTestProps;
                    let nestedStack = new aws_cdk_lib_1.NestedStack(this, operation.operationName + 'CanaryTest', {});
                    let test = new CanaryTest_1.CanaryTest(nestedStack, operation.operationName, {
                        function: canary.function,
                        requestCount: testProps.requestCount,
                        regionalRequestCount: testProps.regionalRequestCount
                            ? testProps.regionalRequestCount
                            : testProps.requestCount,
                        schedule: testProps.schedule,
                        operation: operation,
                        loadBalancer: testProps.loadBalancer,
                        headers: testProps.headers,
                        postData: testProps.postData,
                        azMapper: this.azMapper,
                    });
                    let defaultAvailabilityMetricDetails;
                    let defaultLatencyMetricDetails;
                    if (operation.canaryMetricDetails?.canaryAvailabilityMetricDetails) {
                        defaultAvailabilityMetricDetails = new OperationMetricDetails_1.OperationMetricDetails({
                            operationName: operation.operationName,
                            metricDimensions: new MetricDimensions_1.MetricDimensions({ Operation: operation.operationName }, 'AZ-ID', 'Region'),
                            alarmStatistic: operation.canaryMetricDetails.canaryAvailabilityMetricDetails
                                .alarmStatistic,
                            datapointsToAlarm: operation.canaryMetricDetails.canaryAvailabilityMetricDetails
                                .datapointsToAlarm,
                            evaluationPeriods: operation.canaryMetricDetails.canaryAvailabilityMetricDetails
                                .evaluationPeriods,
                            faultAlarmThreshold: operation.canaryMetricDetails.canaryAvailabilityMetricDetails
                                .faultAlarmThreshold,
                            faultMetricNames: operation.canaryMetricDetails.canaryAvailabilityMetricDetails
                                .faultMetricNames,
                            graphedFaultStatistics: operation.canaryMetricDetails.canaryAvailabilityMetricDetails
                                .graphedFaultStatistics,
                            graphedSuccessStatistics: operation.canaryMetricDetails.canaryAvailabilityMetricDetails
                                .graphedSuccessStatistics,
                            metricNamespace: operation.canaryMetricDetails.canaryAvailabilityMetricDetails
                                .metricNamespace,
                            period: operation.canaryMetricDetails.canaryAvailabilityMetricDetails
                                .period,
                            successAlarmThreshold: operation.canaryMetricDetails.canaryAvailabilityMetricDetails
                                .successAlarmThreshold,
                            successMetricNames: operation.canaryMetricDetails.canaryAvailabilityMetricDetails
                                .successMetricNames,
                            unit: operation.canaryMetricDetails
                                .canaryAvailabilityMetricDetails.unit,
                        }, operation.service.defaultAvailabilityMetricDetails);
                    }
                    else if (operation.canaryTestAvailabilityMetricsOverride) {
                        defaultAvailabilityMetricDetails = new OperationMetricDetails_1.OperationMetricDetails({
                            operationName: operation.operationName,
                            metricNamespace: test.metricNamespace,
                            successMetricNames: ['Success'],
                            faultMetricNames: ['Fault', 'Failure'],
                            metricDimensions: new MetricDimensions_1.MetricDimensions({ Operation: operation.operationName }, 'AZ-ID', 'Region'),
                            alarmStatistic: operation.canaryTestAvailabilityMetricsOverride
                                .alarmStatistic,
                            datapointsToAlarm: operation.canaryTestAvailabilityMetricsOverride
                                .datapointsToAlarm,
                            evaluationPeriods: operation.canaryTestAvailabilityMetricsOverride
                                .evaluationPeriods,
                            faultAlarmThreshold: operation.canaryTestAvailabilityMetricsOverride
                                .faultAlarmThreshold,
                            period: operation.canaryTestAvailabilityMetricsOverride.period,
                            successAlarmThreshold: operation.canaryTestAvailabilityMetricsOverride
                                .successAlarmThreshold,
                        }, props.service.defaultAvailabilityMetricDetails);
                    }
                    else {
                        defaultAvailabilityMetricDetails = new OperationMetricDetails_1.OperationMetricDetails({
                            operationName: operation.operationName,
                            metricNamespace: test.metricNamespace,
                            successMetricNames: ['Success'],
                            faultMetricNames: ['Fault', 'Failure'],
                            metricDimensions: new MetricDimensions_1.MetricDimensions({ Operation: operation.operationName }, 'AZ-ID', 'Region'),
                        }, props.service.defaultAvailabilityMetricDetails);
                    }
                    if (operation.canaryMetricDetails?.canaryLatencyMetricDetails) {
                        defaultLatencyMetricDetails = new OperationMetricDetails_1.OperationMetricDetails({
                            operationName: operation.operationName,
                            metricDimensions: new MetricDimensions_1.MetricDimensions({ Operation: operation.operationName }, 'AZ-ID', 'Region'),
                            alarmStatistic: operation.canaryMetricDetails.canaryLatencyMetricDetails
                                .alarmStatistic,
                            datapointsToAlarm: operation.canaryMetricDetails.canaryLatencyMetricDetails
                                .datapointsToAlarm,
                            evaluationPeriods: operation.canaryMetricDetails.canaryLatencyMetricDetails
                                .evaluationPeriods,
                            faultAlarmThreshold: operation.canaryMetricDetails.canaryLatencyMetricDetails
                                .faultAlarmThreshold,
                            faultMetricNames: operation.canaryMetricDetails.canaryLatencyMetricDetails
                                .faultMetricNames,
                            graphedFaultStatistics: operation.canaryMetricDetails.canaryLatencyMetricDetails
                                .graphedFaultStatistics,
                            graphedSuccessStatistics: operation.canaryMetricDetails.canaryLatencyMetricDetails
                                .graphedSuccessStatistics,
                            metricNamespace: operation.canaryMetricDetails.canaryLatencyMetricDetails
                                .metricNamespace,
                            period: operation.canaryMetricDetails.canaryLatencyMetricDetails
                                .period,
                            successAlarmThreshold: operation.canaryMetricDetails.canaryLatencyMetricDetails
                                .successAlarmThreshold,
                            successMetricNames: operation.canaryMetricDetails.canaryLatencyMetricDetails
                                .successMetricNames,
                            unit: operation.canaryMetricDetails.canaryLatencyMetricDetails
                                .unit,
                        }, operation.service.defaultLatencyMetricDetails);
                    }
                    else if (operation.canaryTestLatencyMetricsOverride) {
                        defaultLatencyMetricDetails = new OperationMetricDetails_1.OperationMetricDetails({
                            operationName: operation.operationName,
                            metricNamespace: test.metricNamespace,
                            successMetricNames: ['SuccessLatency'],
                            faultMetricNames: ['FaultLatency'],
                            metricDimensions: new MetricDimensions_1.MetricDimensions({ Operation: operation.operationName }, 'AZ-ID', 'Region'),
                            alarmStatistic: operation.canaryTestLatencyMetricsOverride.alarmStatistic,
                            datapointsToAlarm: operation.canaryTestLatencyMetricsOverride.datapointsToAlarm,
                            evaluationPeriods: operation.canaryTestLatencyMetricsOverride.evaluationPeriods,
                            faultAlarmThreshold: operation.canaryTestLatencyMetricsOverride
                                .faultAlarmThreshold,
                            period: operation.canaryTestLatencyMetricsOverride.period,
                            successAlarmThreshold: operation.canaryTestLatencyMetricsOverride
                                .successAlarmThreshold,
                        }, props.service.defaultLatencyMetricDetails);
                    }
                    else {
                        defaultLatencyMetricDetails = new OperationMetricDetails_1.OperationMetricDetails({
                            operationName: operation.operationName,
                            metricNamespace: test.metricNamespace,
                            successMetricNames: ['SuccessLatency'],
                            faultMetricNames: ['FaultLatency'],
                            metricDimensions: new MetricDimensions_1.MetricDimensions({ Operation: operation.operationName }, 'AZ-ID', 'Region'),
                        }, props.service.defaultLatencyMetricDetails);
                    }
                    let newOperation = new Operation_1.Operation({
                        serverSideAvailabilityMetricDetails: operation.serverSideAvailabilityMetricDetails,
                        serverSideLatencyMetricDetails: operation.serverSideLatencyMetricDetails,
                        serverSideContributorInsightRuleDetails: operation.serverSideContributorInsightRuleDetails,
                        service: operation.service,
                        operationName: operation.operationName,
                        path: operation.path,
                        critical: operation.critical,
                        httpMethods: operation.httpMethods,
                        canaryMetricDetails: new CanaryMetrics_1.CanaryMetrics({
                            canaryAvailabilityMetricDetails: defaultAvailabilityMetricDetails,
                            canaryLatencyMetricDetails: defaultLatencyMetricDetails,
                        }),
                    });
                    props.service.operations[i] = newOperation;
                }
            }
        }
        if (props.outlierDetectionAlgorithm != OutlierDetectionAlgorithm_1.OutlierDetectionAlgorithm.STATIC) {
            let outlierDetectionStack = new StackWithDynamicSource_1.StackWithDynamicSource(this, 'OutlierDetectionStack', {
                assetsBucketsParameterName: props.assetsBucketParameterName,
                assetsBucketPrefixParameterName: props.assetsBucketPrefixParameterName,
            });
            this.outlierDetectionFunction = new OutlierDetectionFunction_1.OutlierDetectionFunction(outlierDetectionStack, 'OutlierDetectionFunction', {}).function;
        }
        this.perOperationAlarmsAndRules = Object.fromEntries(props.service.operations.map((operation) => {
            let nestedStack = new aws_cdk_lib_1.NestedStack(this, operation.operationName + 'OperationAlarmsAndRulesNestedStack');
            return [
                operation.operationName,
                new OperationAlarmsAndRules_1.OperationAlarmsAndRules(nestedStack, operation.operationName, {
                    operation: operation,
                    outlierDetectionAlgorithm: props.outlierDetectionAlgorithm,
                    outlierThreshold: outlierThreshold,
                    loadBalancer: props.service.loadBalancer,
                    azMapper: this.azMapper,
                    outlierDetectionFunction: this.outlierDetectionFunction,
                }),
            ];
        }));
        this.perOperationZonalImpactAlarms = Object.fromEntries(Object.entries(this.perOperationAlarmsAndRules).map(([key, value]) => {
            return [key, value.aggregateZonalAlarmsMap];
        }));
        let serviceAlarmsStack = new aws_cdk_lib_1.NestedStack(this, 'ServiceAlarmsNestedStack');
        this.serviceAlarms = new ServiceAlarmsAndRules_1.ServiceAlarmsAndRules(serviceAlarmsStack, props.service.serviceName, {
            perOperationAlarmsAndRules: this.perOperationAlarmsAndRules,
            service: props.service,
            azMapper: this.azMapper,
        });
        if (props.createDashboards) {
            this.operationDashboards = [];
            props.service.operations.forEach((x) => {
                let dashboardStack = new aws_cdk_lib_1.NestedStack(this, x.operationName + 'Dashboard', {});
                if (this.operationDashboards) {
                    this.operationDashboards.push(new OperationAvailabilityAndLatencyDashboard_1.OperationAvailabilityAndLatencyDashboard(dashboardStack, x.operationName, {
                        operation: x,
                        azMapper: this.azMapper,
                        interval: props.interval
                            ? props.interval
                            : aws_cdk_lib_1.Duration.minutes(60),
                        loadBalancer: props.service.loadBalancer,
                        regionalEndpointCanaryAvailabilityAlarm: this.perOperationAlarmsAndRules[x.operationName]
                            .canaryRegionalAlarmsAndRules?.availabilityAlarm,
                        regionalEndpointCanaryLatencyAlarm: this.perOperationAlarmsAndRules[x.operationName]
                            .canaryRegionalAlarmsAndRules?.latencyAlarm,
                        regionalEndpointServerAvailabilityAlarm: this.perOperationAlarmsAndRules[x.operationName]
                            .serverSideRegionalAlarmsAndRules.availabilityAlarm,
                        regionalEndpointServerLatencyAlarm: this.perOperationAlarmsAndRules[x.operationName]
                            .serverSideRegionalAlarmsAndRules.latencyAlarm,
                        zonalEndpointCanaryAvailabilityAlarms: this.perOperationAlarmsAndRules[x.operationName].canaryZonalAlarmsAndRules?.map((a) => a.availabilityAlarm),
                        zonalEndpointCanaryLatencyAlarms: this.perOperationAlarmsAndRules[x.operationName].canaryZonalAlarmsAndRules?.map((a) => a.latencyAlarm),
                        zonalEndpointServerAvailabilityAlarms: this.perOperationAlarmsAndRules[x.operationName].serverSideZonalAlarmsAndRules.map((a) => a.availabilityAlarm),
                        zonalEndpointServerLatencyAlarms: this.perOperationAlarmsAndRules[x.operationName].serverSideZonalAlarmsAndRules.map((a) => a.latencyAlarm),
                        isolatedAZImpactAlarms: this.perOperationAlarmsAndRules[x.operationName]
                            .aggregateZonalAlarms,
                        regionalImpactAlarm: this.perOperationAlarmsAndRules[x.operationName]
                            .aggregateRegionalAlarm,
                        instanceContributorsToFaults: this.perOperationAlarmsAndRules[x.operationName]
                            .serverSideRegionalAlarmsAndRules
                            .instanceContributorsToRegionalFaults,
                        instanceContributorsToHighLatency: this.perOperationAlarmsAndRules[x.operationName]
                            .serverSideRegionalAlarmsAndRules
                            .instanceContributorsToRegionalHighLatency,
                    }).dashboard);
                }
            });
            let dashboardStack = new aws_cdk_lib_1.NestedStack(this, 'ServiceDashboard', {});
            this.serviceDashboard = new ServiceAvailabilityAndLatencyDashboard_1.ServiceAvailabilityAndLatencyDashboard(dashboardStack, props.service.serviceName.toLowerCase(), {
                interval: props.interval ? props.interval : aws_cdk_lib_1.Duration.minutes(60),
                service: props.service,
                aggregateRegionalAlarm: this.serviceAlarms.regionalFaultCountServerSideAlarm,
                zonalAggregateAlarms: this.serviceAlarms.zonalAggregateIsolatedImpactAlarms,
                azMapper: this.azMapper,
            }).dashboard;
        }
    }
}
exports.InstrumentedServiceMultiAZObservability = InstrumentedServiceMultiAZObservability;
_a = JSII_RTTI_SYMBOL_1;
InstrumentedServiceMultiAZObservability[_a] = { fqn: "@cdklabs/multi-az-observability.InstrumentedServiceMultiAZObservability", version: "0.0.1-alpha.14" };
//# sourceMappingURL=data:application/json;base64,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