"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OutlierDetectionFunction = void 0;
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
const MetricsHelper_1 = require("../utilities/MetricsHelper");
class OutlierDetectionFunction extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        let xrayManagedPolicy = new aws_iam_1.ManagedPolicy(this, 'xrayManagedPolicy', {
            path: '/outlier-detection/',
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: [
                        'xray:PutTraceSegments',
                        'xray:PutTelemetryRecords',
                        'xray:GetSamplingRules',
                        'xray:GetSamplingTargets',
                        'xray:GetSamplingStatisticSummaries',
                    ],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: ['*'],
                }),
            ],
        });
        let cwManagedPolicy = new aws_iam_1.ManagedPolicy(this, 'CWManagedPolicy', {
            path: '/outlier-detection/',
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: ['cloudwatch:GetMetricData', 'cloduwatch:PutMetricData'],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: ['*'],
                }),
            ],
        });
        let executionRole = new aws_iam_1.Role(this, 'executionRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            path: '/outlier-detection/',
            managedPolicies: [xrayManagedPolicy, cwManagedPolicy],
        });
        let sciPyLayer = new aws_lambda_1.LayerVersion(this, 'SciPyLayer', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, 'src/scipy-layer.zip')),
            compatibleArchitectures: [aws_lambda_1.Architecture.ARM_64],
            compatibleRuntimes: [MetricsHelper_1.MetricsHelper.PythonRuntime],
        });
        let monitoringLayer = new aws_lambda_1.LayerVersion(this, 'MonitoringLayer', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '../monitoring/src/monitoring-layer.zip')),
            compatibleArchitectures: [aws_lambda_1.Architecture.ARM_64],
            compatibleRuntimes: [MetricsHelper_1.MetricsHelper.PythonRuntime],
        });
        if (props.vpc !== undefined && props.vpc != null) {
            let sg = new aws_ec2_1.SecurityGroup(this, 'OutlierDetectionSecurityGroup', {
                description: 'Allow outlier detection function to communicate with CW',
                vpc: props.vpc,
                allowAllOutbound: true,
            });
            this.function = new aws_lambda_1.Function(this, 'OutlierDetection', {
                runtime: MetricsHelper_1.MetricsHelper.PythonRuntime,
                code: aws_lambda_1.Code.fromAsset(path.join(__dirname, 'src/outlier-detection.zip')),
                handler: 'index.handler',
                role: executionRole,
                architecture: aws_lambda_1.Architecture.ARM_64,
                tracing: aws_lambda_1.Tracing.ACTIVE,
                timeout: aws_cdk_lib_1.Duration.seconds(5),
                memorySize: 512,
                layers: [sciPyLayer, monitoringLayer],
                environment: {
                    REGION: aws_cdk_lib_1.Aws.REGION,
                    PARTITION: aws_cdk_lib_1.Aws.PARTITION,
                },
                vpc: props.vpc,
                securityGroups: [sg],
                vpcSubnets: props.subnetSelection,
            });
        }
        else {
            this.function = new aws_lambda_1.Function(this, 'OutlierDetection', {
                runtime: MetricsHelper_1.MetricsHelper.PythonRuntime,
                code: aws_lambda_1.Code.fromAsset(path.join(__dirname, 'src/outlier-detection.zip')),
                handler: 'index.handler',
                role: executionRole,
                architecture: aws_lambda_1.Architecture.ARM_64,
                tracing: aws_lambda_1.Tracing.ACTIVE,
                timeout: aws_cdk_lib_1.Duration.seconds(5),
                memorySize: 512,
                layers: [sciPyLayer, monitoringLayer],
                environment: {
                    REGION: aws_cdk_lib_1.Aws.REGION,
                    PARTITION: aws_cdk_lib_1.Aws.PARTITION,
                },
            });
        }
        aws_cdk_lib_1.Tags.of(this.function).add('cloudwatch:datasource', 'custom', {
            includeResourceTypes: ['AWS::Lambda::Function'],
        });
        this.function.addPermission('invokePermission', {
            action: 'lambda:InvokeFunction',
            principal: new aws_iam_1.ServicePrincipal('lambda.datasource.cloudwatch.amazonaws.com'),
            sourceAccount: aws_cdk_lib_1.Fn.ref('AWS::AccountId'),
        });
        this.logGroup = new aws_logs_1.LogGroup(this, 'logGroup', {
            logGroupName: `/aws/lambda/${this.function.functionName}`,
            retention: aws_logs_1.RetentionDays.ONE_WEEK,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        new aws_iam_1.ManagedPolicy(this, 'cwLogsManagedPolicy', {
            path: '/outlier-detection/',
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: ['logs:CreateLogStream', 'logs:PutLogEvents'],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: [this.logGroup.logGroupArn],
                }),
            ],
            roles: [executionRole],
        });
    }
}
exports.OutlierDetectionFunction = OutlierDetectionFunction;
//# sourceMappingURL=data:application/json;base64,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