"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ZonalAvailabilityMetrics = void 0;
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const AvailabilityAndLatencyMetrics_1 = require("./AvailabilityAndLatencyMetrics");
const AvailabilityMetricType_1 = require("../utilities/AvailabilityMetricType");
class ZonalAvailabilityMetrics {
    /**
     * Creates a zonal service level availability metrics, one metric for
     * each operation at the zonal level and the service.
     * @param props
     * @returns The metric at index 0 is the metric math expression for the whole service. The following metrics
     * are the metrics for each operation included in the request availability metric props.
     */
    static createZonalServiceAvailabilityMetrics(props) {
        let usingMetrics = {};
        let operationMetrics = [];
        let counter = 0;
        props.availabilityMetricProps.forEach((prop) => {
            let keyPrefix = (prop.keyPrefix === undefined || prop.keyPrefix == ''
                ? ''
                : prop.keyPrefix.toLowerCase() + '_') +
                prop.metricDetails.operationName.toLowerCase() +
                '_' +
                prop.metricType.toString().toLowerCase();
            let zonalOperationAvailabilityMetric = this.createZonalAvailabilityMetric(prop);
            operationMetrics.push(zonalOperationAvailabilityMetric);
            usingMetrics[`${keyPrefix}${counter++}`] =
                zonalOperationAvailabilityMetric;
        });
        let expression = '';
        switch (props.availabilityMetricProps[0].metricType) {
            case AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE:
                expression = `(${Object.keys(usingMetrics).join('+')}) / ${props.availabilityMetricProps.length}`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.REQUEST_COUNT:
                expression = `${Object.keys(usingMetrics).join('+')}`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT:
                expression = `${Object.keys(usingMetrics).join('+')}`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.FAULT_RATE:
                expression = `(${Object.keys(usingMetrics).join('+')}) / ${props.availabilityMetricProps.length}`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_COUNT:
                expression = `${Object.keys(usingMetrics).join('+')}`;
                break;
        }
        let math = new aws_cloudwatch_1.MathExpression({
            usingMetrics: usingMetrics,
            period: props.period,
            label: props.label,
            expression: expression,
        });
        operationMetrics.splice(0, 0, math);
        return operationMetrics;
    }
    /**
     * Creates a zonal availability metric
     * @param props
     * @returns
     */
    static createZonalAvailabilityMetric(props) {
        return AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.createAvailabilityMetric(props, props.metricDetails.metricDimensions.zonalDimensions(props.availabilityZoneId, aws_cdk_lib_1.Aws.REGION));
    }
}
exports.ZonalAvailabilityMetrics = ZonalAvailabilityMetrics;
//# sourceMappingURL=data:application/json;base64,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