"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RegionalLatencyMetrics = void 0;
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const AvailabilityAndLatencyMetrics_1 = require("./AvailabilityAndLatencyMetrics");
const MetricsHelper_1 = require("../utilities/MetricsHelper");
class RegionalLatencyMetrics {
    /**
     * Creates a metrics for regional latency, one metric per metric name for the
     * specified type of latency metric. You will need to perform some aggregation
     * of these metrics if there is more than 1 metric name that correponds to
     * SuccessLatency or FaultLatency, like doing an average.
     * @param props
     * @returns
     */
    static createRegionalLatencyMetrics(props) {
        return AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.createLatencyMetrics(props, props.metricDetails.metricDimensions.regionalDimensions(aws_cdk_lib_1.Aws.REGION));
    }
    /**
     * Creates a regional average latency metric, averages the values from all of the
     * metric names that represent either SuccessLatency or FaultLatency
     * @param props
     * @returns
     */
    static createRegionalAverageLatencyMetric(props) {
        return AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.createAverageLatencyMetric(props, props.metricDetails.metricDimensions.regionalDimensions(aws_cdk_lib_1.Aws.REGION));
    }
    /**
     * Creates a count of high latency metrics for either SuccessLatency or FaultLatency
     * @param props
     * @returns
     */
    static createRegionalLatencyCountMetric(props) {
        return AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.createLatencyCountMetric(props, props.metricDetails.metricDimensions.regionalDimensions(aws_cdk_lib_1.Aws.REGION));
    }
    /**
     * Creates a count of high latency responses for all critical operations
     * @param props
     * @returns
     */
    static createRegionalServiceLatencyCountMetrics(props, includeRegionCountAggregate) {
        if (includeRegionCountAggregate) {
            let usingMetrics = {};
            let operationMetrics = [];
            let keyPrefix = MetricsHelper_1.MetricsHelper.nextChar();
            props.latencyMetricProps.forEach((prop, index) => {
                let operationRegionalMetric = this.createRegionalLatencyCountMetric(prop);
                operationMetrics.push(operationRegionalMetric);
                usingMetrics[`${keyPrefix}${index}`] = operationRegionalMetric;
                keyPrefix = MetricsHelper_1.MetricsHelper.nextChar(keyPrefix);
            });
            if (Object.keys(usingMetrics).length == 1) {
                operationMetrics.push(Object.values(usingMetrics)[0]);
            }
            else {
                let math = new aws_cloudwatch_1.MathExpression({
                    usingMetrics: usingMetrics,
                    period: props.period,
                    label: props.label,
                    expression: Object.keys(usingMetrics).join('+'),
                });
                operationMetrics.splice(0, 0, math);
            }
            return operationMetrics;
        }
        else {
            return props.latencyMetricProps.map((prop) => {
                return this.createRegionalLatencyCountMetric(prop);
            });
        }
    }
}
exports.RegionalLatencyMetrics = RegionalLatencyMetrics;
//# sourceMappingURL=data:application/json;base64,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