"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RegionalAvailabilityMetrics = void 0;
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const AvailabilityAndLatencyMetrics_1 = require("./AvailabilityAndLatencyMetrics");
const AvailabilityMetricType_1 = require("../utilities/AvailabilityMetricType");
class RegionalAvailabilityMetrics {
    /**
     * Creates a regional availability metric
     * @param props
     * @returns
     */
    static createRegionalAvailabilityMetric(props) {
        return AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.createAvailabilityMetric(props, props.metricDetails.metricDimensions.regionalDimensions(aws_cdk_lib_1.Fn.ref('AWS::Region')));
    }
    /**
     * Creates a regional service level availability metrics, one metric for
     * each operation at the regional level and the service.
     * @param props
     * @returns The metric at index 0 is the metric math expression for the whole service. The following metrics
     * are the metrics for each operation included in the request availability metric props.
     */
    static createRegionalServiceAvailabilityMetrics(props) {
        let usingMetrics = {};
        let operationMetrics = [];
        let counter = 0;
        props.availabilityMetricProps.forEach((prop) => {
            let keyPrefix = (prop.keyPrefix === undefined || prop.keyPrefix == ''
                ? ''
                : prop.keyPrefix.toLowerCase() + '_') +
                prop.metricDetails.operationName.toLowerCase() +
                '_' +
                prop.metricType.toString().toLowerCase();
            let regionalOperationAvailabilityMetric = this.createRegionalAvailabilityMetric(prop);
            operationMetrics.push(regionalOperationAvailabilityMetric);
            usingMetrics[`${keyPrefix}${counter++}`] =
                regionalOperationAvailabilityMetric;
        });
        let expression = '';
        switch (props.availabilityMetricProps[0].metricType) {
            case AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE:
                expression = `(${Object.keys(usingMetrics).join('+')}) / ${props.availabilityMetricProps.length}`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.REQUEST_COUNT:
                expression = `${Object.keys(usingMetrics).join('+')}`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT:
                expression = `${Object.keys(usingMetrics).join('+')}`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.FAULT_RATE:
                expression = `(${Object.keys(usingMetrics).join('+')}) / ${props.availabilityMetricProps.length}`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_COUNT:
                expression = `${Object.keys(usingMetrics).join('+')}`;
                break;
        }
        let math = new aws_cloudwatch_1.MathExpression({
            usingMetrics: usingMetrics,
            period: props.period,
            label: props.label,
            expression: expression,
        });
        operationMetrics.splice(0, 0, math);
        return operationMetrics;
    }
}
exports.RegionalAvailabilityMetrics = RegionalAvailabilityMetrics;
//# sourceMappingURL=data:application/json;base64,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