"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AvailabilityAndLatencyMetrics = void 0;
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const AvailabilityMetricType_1 = require("../utilities/AvailabilityMetricType");
const LatencyMetricType_1 = require("../utilities/LatencyMetricType");
/**
 * Class for creating availability and latency metrics that can be used in alarms and graphs
 */
class AvailabilityAndLatencyMetrics {
    /**
     * General purpose method to create availability metrics
     * @param props
     * @param dimensions
     * @returns
     */
    static createAvailabilityMetric(props, dimensions) {
        let counter = 0;
        let key = '';
        let usingMetrics = {};
        let successKeys = [];
        let faultKeys = [];
        if (props.metricDetails.successMetricNames !== undefined &&
            props.metricType != AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT) {
            props.metricDetails.successMetricNames.forEach((successMetric) => {
                let keyPrefix = (props.keyPrefix === undefined || props.keyPrefix == ''
                    ? ''
                    : props.keyPrefix.toLowerCase() + '_') +
                    props.metricDetails.operationName.toLowerCase() +
                    '_' +
                    successMetric.toLowerCase();
                key = keyPrefix + '_' + counter++;
                successKeys.push(key);
                usingMetrics[key] = new aws_cloudwatch_1.Metric({
                    namespace: props.metricDetails.metricNamespace,
                    metricName: successMetric,
                    unit: props.metricDetails.unit,
                    period: props.metricDetails.period,
                    statistic: props.metricDetails.alarmStatistic,
                    dimensionsMap: dimensions,
                    label: successMetric,
                });
            });
        }
        if (props.metricDetails.faultMetricNames !== undefined &&
            props.metricType != AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_COUNT) {
            props.metricDetails.faultMetricNames.forEach((faultMetric) => {
                let keyPrefix = (props.keyPrefix === undefined || props.keyPrefix == ''
                    ? ''
                    : props.keyPrefix.toLowerCase() + '_') +
                    props.metricDetails.operationName.toLowerCase() +
                    '_' +
                    faultMetric.toLowerCase();
                key = keyPrefix + '_' + counter++;
                faultKeys.push(key);
                usingMetrics[key] = new aws_cloudwatch_1.Metric({
                    namespace: props.metricDetails.metricNamespace,
                    metricName: faultMetric,
                    unit: props.metricDetails.unit,
                    period: props.metricDetails.period,
                    statistic: props.metricDetails.alarmStatistic,
                    dimensionsMap: dimensions,
                    label: faultMetric,
                });
            });
        }
        let expression = '';
        switch (props.metricType) {
            case AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE:
                expression = `((${successKeys.join('+')}) / (${successKeys.join('+')}+${faultKeys.join('+')})) * 100`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.REQUEST_COUNT:
                expression = `${successKeys.join('+')}+${faultKeys.join('+')}`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT:
                expression = `(${faultKeys.join('+')})`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.FAULT_RATE:
                expression = `((${faultKeys.join('+')}) / (${successKeys.join('+')}+${faultKeys.join('+')})) * 100`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_COUNT:
                expression = `(${successKeys.join('+')})`;
                break;
        }
        return new aws_cloudwatch_1.MathExpression({
            expression: expression,
            label: props.label,
            period: props.metricDetails.period,
            usingMetrics: usingMetrics,
        });
    }
    /**
     * General purpose method to create latency metrics, the reason this creates an array of metrics while the
     * equivalent availability metric method doesn't is because in availability, we can just sum the count of different
     * metric names while for latency we can't sum the count because that's not what's being measured. It allows the
     * caller to decide if they only want to take the first name, or average all of the names
     * (like SuccessLatency and BigItemSuccessLatency).
     *
     * @param props
     * @param dimensions
     * @returns
     */
    static createLatencyMetrics(props, dimensions) {
        let names;
        switch (props.metricType) {
            default:
            case LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY:
                names = props.metricDetails.successMetricNames;
                break;
            case LatencyMetricType_1.LatencyMetricType.FAULT_LATENCY:
                names = props.metricDetails.faultMetricNames;
                break;
        }
        return names.map((x) => new aws_cloudwatch_1.Metric({
            metricName: x,
            namespace: props.metricDetails.metricNamespace,
            unit: props.metricDetails.unit,
            period: props.metricDetails.period,
            statistic: props.statistic,
            dimensionsMap: dimensions,
            label: props.label,
        }));
    }
    /**
     * Takes all of the success or failure latency metric names and creates an average of those
     * names, if there's only 1 name, it just returns that metric
     * @param props
     * @param dimensions
     */
    static createAverageLatencyMetric(props, dimensions) {
        let latencyMetrics = AvailabilityAndLatencyMetrics.createLatencyMetrics(props, dimensions);
        if (latencyMetrics.length == 1) {
            return latencyMetrics[0];
        }
        else {
            let usingMetrics = {};
            latencyMetrics.forEach((metric, index) => {
                let keyPrefix = (props.keyPrefix === undefined || props.keyPrefix == ''
                    ? ''
                    : props.keyPrefix.toLowerCase() + '_') +
                    props.metricDetails.operationName.toLowerCase() +
                    '_' +
                    props.metricType.toString().toLowerCase();
                usingMetrics[keyPrefix + index] = metric;
            });
            return new aws_cloudwatch_1.MathExpression({
                expression: `(${Object.keys(usingMetrics).join('+')})/${Object.keys(usingMetrics).length}`,
                label: props.label,
                period: props.metricDetails.period,
                usingMetrics: usingMetrics,
            });
        }
    }
    /**
     * Creates a count of high latency metrics for either SuccessLatency or FaultLatency, will total
     * the count of requests that exceed a threshold you define in your statistic, like TC(200:) across
     * all metric names that are part of either Success or Fault latency.
     * @param props
     * @returns
     */
    static createLatencyCountMetric(props, dimensions) {
        let latencyMetrics = AvailabilityAndLatencyMetrics.createLatencyMetrics(props, dimensions);
        if (latencyMetrics.length == 1) {
            return latencyMetrics[0];
        }
        else {
            let usingMetrics = {};
            latencyMetrics.forEach((metric, index) => {
                let keyPrefix = (props.keyPrefix === undefined || props.keyPrefix == ''
                    ? ''
                    : props.keyPrefix.toLowerCase() + '_') +
                    props.metricDetails.operationName.toLowerCase() +
                    '_' +
                    props.metricType.toString().toLowerCase();
                usingMetrics[keyPrefix + index] = metric;
            });
            return new aws_cloudwatch_1.MathExpression({
                expression: Object.keys(usingMetrics).join('+'),
                label: props.label,
                period: props.metricDetails.period,
                usingMetrics: usingMetrics,
            });
        }
    }
}
exports.AvailabilityAndLatencyMetrics = AvailabilityAndLatencyMetrics;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiQXZhaWxhYmlsaXR5QW5kTGF0ZW5jeU1ldHJpY3MuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvbWV0cmljcy9BdmFpbGFiaWxpdHlBbmRMYXRlbmN5TWV0cmljcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxxRUFBcUU7QUFDckUsc0NBQXNDO0FBQ3RDLCtEQUE2RTtBQUc3RSxnRkFBNkU7QUFDN0Usc0VBQW1FO0FBRW5FOztHQUVHO0FBQ0gsTUFBYSw2QkFBNkI7SUFFeEM7Ozs7O09BS0c7SUFDSCxNQUFNLENBQUMsd0JBQXdCLENBQzdCLEtBQThCLEVBQzlCLFVBQXFDO1FBRXJDLElBQUksT0FBTyxHQUFXLENBQUMsQ0FBQztRQUN4QixJQUFJLEdBQUcsR0FBVyxFQUFFLENBQUM7UUFFckIsSUFBSSxZQUFZLEdBQStCLEVBQUUsQ0FBQztRQUVsRCxJQUFJLFdBQVcsR0FBYSxFQUFFLENBQUM7UUFDL0IsSUFBSSxTQUFTLEdBQWEsRUFBRSxDQUFDO1FBRTdCLElBQ0UsS0FBSyxDQUFDLGFBQWEsQ0FBQyxrQkFBa0IsS0FBSyxTQUFTO1lBQ3BELEtBQUssQ0FBQyxVQUFVLElBQUksK0NBQXNCLENBQUMsV0FBVyxFQUN0RCxDQUFDO1lBQ0QsS0FBSyxDQUFDLGFBQWEsQ0FBQyxrQkFBa0IsQ0FBQyxPQUFPLENBQzVDLENBQUMsYUFBcUIsRUFBRSxFQUFFO2dCQUN4QixJQUFJLFNBQVMsR0FDWCxDQUFDLEtBQUssQ0FBQyxTQUFTLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxTQUFTLElBQUksRUFBRTtvQkFDckQsQ0FBQyxDQUFDLEVBQUU7b0JBQ0osQ0FBQyxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsV0FBVyxFQUFFLEdBQUcsR0FBRyxDQUFDO29CQUN4QyxLQUFLLENBQUMsYUFBYSxDQUFDLGFBQWEsQ0FBQyxXQUFXLEVBQUU7b0JBQy9DLEdBQUc7b0JBQ0gsYUFBYSxDQUFDLFdBQVcsRUFBRSxDQUFDO2dCQUU5QixHQUFHLEdBQUcsU0FBUyxHQUFHLEdBQUcsR0FBRyxPQUFPLEVBQUUsQ0FBQztnQkFDbEMsV0FBVyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFFdEIsWUFBWSxDQUFDLEdBQUcsQ0FBQyxHQUFHLElBQUksdUJBQU0sQ0FBQztvQkFDN0IsU0FBUyxFQUFFLEtBQUssQ0FBQyxhQUFhLENBQUMsZUFBZTtvQkFDOUMsVUFBVSxFQUFFLGFBQWE7b0JBQ3pCLElBQUksRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLElBQUk7b0JBQzlCLE1BQU0sRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLE1BQU07b0JBQ2xDLFNBQVMsRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLGNBQWM7b0JBQzdDLGFBQWEsRUFBRSxVQUFVO29CQUN6QixLQUFLLEVBQUUsYUFBYTtpQkFDckIsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUNGLENBQUM7UUFDSixDQUFDO1FBRUQsSUFDRSxLQUFLLENBQUMsYUFBYSxDQUFDLGdCQUFnQixLQUFLLFNBQVM7WUFDbEQsS0FBSyxDQUFDLFVBQVUsSUFBSSwrQ0FBc0IsQ0FBQyxhQUFhLEVBQ3hELENBQUM7WUFDRCxLQUFLLENBQUMsYUFBYSxDQUFDLGdCQUFnQixDQUFDLE9BQU8sQ0FBQyxDQUFDLFdBQVcsRUFBRSxFQUFFO2dCQUMzRCxJQUFJLFNBQVMsR0FDWCxDQUFDLEtBQUssQ0FBQyxTQUFTLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxTQUFTLElBQUksRUFBRTtvQkFDckQsQ0FBQyxDQUFDLEVBQUU7b0JBQ0osQ0FBQyxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsV0FBVyxFQUFFLEdBQUcsR0FBRyxDQUFDO29CQUN4QyxLQUFLLENBQUMsYUFBYSxDQUFDLGFBQWEsQ0FBQyxXQUFXLEVBQUU7b0JBQy9DLEdBQUc7b0JBQ0gsV0FBVyxDQUFDLFdBQVcsRUFBRSxDQUFDO2dCQUU1QixHQUFHLEdBQUcsU0FBUyxHQUFHLEdBQUcsR0FBRyxPQUFPLEVBQUUsQ0FBQztnQkFDbEMsU0FBUyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFFcEIsWUFBWSxDQUFDLEdBQUcsQ0FBQyxHQUFHLElBQUksdUJBQU0sQ0FBQztvQkFDN0IsU0FBUyxFQUFFLEtBQUssQ0FBQyxhQUFhLENBQUMsZUFBZTtvQkFDOUMsVUFBVSxFQUFFLFdBQVc7b0JBQ3ZCLElBQUksRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLElBQUk7b0JBQzlCLE1BQU0sRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLE1BQU07b0JBQ2xDLFNBQVMsRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLGNBQWM7b0JBQzdDLGFBQWEsRUFBRSxVQUFVO29CQUN6QixLQUFLLEVBQUUsV0FBVztpQkFDbkIsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDO1FBRUQsSUFBSSxVQUFVLEdBQVcsRUFBRSxDQUFDO1FBRTVCLFFBQVEsS0FBSyxDQUFDLFVBQVUsRUFBRSxDQUFDO1lBQ3pCLEtBQUssK0NBQXNCLENBQUMsWUFBWTtnQkFDdEMsVUFBVSxHQUFHLEtBQUssV0FBVyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsUUFBUSxXQUFXLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLFNBQVMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQztnQkFDdEcsTUFBTTtZQUNSLEtBQUssK0NBQXNCLENBQUMsYUFBYTtnQkFDdkMsVUFBVSxHQUFHLEdBQUcsV0FBVyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxTQUFTLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUM7Z0JBQy9ELE1BQU07WUFDUixLQUFLLCtDQUFzQixDQUFDLFdBQVc7Z0JBQ3JDLFVBQVUsR0FBRyxJQUFJLFNBQVMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQztnQkFDeEMsTUFBTTtZQUNSLEtBQUssK0NBQXNCLENBQUMsVUFBVTtnQkFDcEMsVUFBVSxHQUFHLEtBQUssU0FBUyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsUUFBUSxXQUFXLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLFNBQVMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQztnQkFDcEcsTUFBTTtZQUNSLEtBQUssK0NBQXNCLENBQUMsYUFBYTtnQkFDdkMsVUFBVSxHQUFHLElBQUksV0FBVyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDO2dCQUMxQyxNQUFNO1FBQ1YsQ0FBQztRQUVELE9BQU8sSUFBSSwrQkFBYyxDQUFDO1lBQ3hCLFVBQVUsRUFBRSxVQUFVO1lBQ3RCLEtBQUssRUFBRSxLQUFLLENBQUMsS0FBSztZQUNsQixNQUFNLEVBQUUsS0FBSyxDQUFDLGFBQWEsQ0FBQyxNQUFNO1lBQ2xDLFlBQVksRUFBRSxZQUFZO1NBQzNCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0gsTUFBTSxDQUFDLG9CQUFvQixDQUN6QixLQUF5QixFQUN6QixVQUFxQztRQUVyQyxJQUFJLEtBQWUsQ0FBQztRQUVwQixRQUFRLEtBQUssQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUN6QixRQUFRO1lBQ1IsS0FBSyxxQ0FBaUIsQ0FBQyxlQUFlO2dCQUNwQyxLQUFLLEdBQUcsS0FBSyxDQUFDLGFBQWEsQ0FBQyxrQkFBa0IsQ0FBQztnQkFDL0MsTUFBTTtZQUNSLEtBQUsscUNBQWlCLENBQUMsYUFBYTtnQkFDbEMsS0FBSyxHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUMsZ0JBQWdCLENBQUM7Z0JBQzdDLE1BQU07UUFDVixDQUFDO1FBRUQsT0FBTyxLQUFLLENBQUMsR0FBRyxDQUNkLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FDSixJQUFJLHVCQUFNLENBQUM7WUFDVCxVQUFVLEVBQUUsQ0FBQztZQUNiLFNBQVMsRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLGVBQWU7WUFDOUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxhQUFhLENBQUMsSUFBSTtZQUM5QixNQUFNLEVBQUUsS0FBSyxDQUFDLGFBQWEsQ0FBQyxNQUFNO1lBQ2xDLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztZQUMxQixhQUFhLEVBQUUsVUFBVTtZQUN6QixLQUFLLEVBQUUsS0FBSyxDQUFDLEtBQUs7U0FDbkIsQ0FBQyxDQUNMLENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSCxNQUFNLENBQUMsMEJBQTBCLENBQy9CLEtBQXlCLEVBQ3pCLFVBQXFDO1FBRXJDLElBQUksY0FBYyxHQUNoQiw2QkFBNkIsQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFFeEUsSUFBSSxjQUFjLENBQUMsTUFBTSxJQUFJLENBQUMsRUFBRSxDQUFDO1lBQy9CLE9BQU8sY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzNCLENBQUM7YUFBTSxDQUFDO1lBQ04sSUFBSSxZQUFZLEdBQStCLEVBQUUsQ0FBQztZQUVsRCxjQUFjLENBQUMsT0FBTyxDQUFDLENBQUMsTUFBZSxFQUFFLEtBQWEsRUFBRSxFQUFFO2dCQUN4RCxJQUFJLFNBQVMsR0FDWCxDQUFDLEtBQUssQ0FBQyxTQUFTLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxTQUFTLElBQUksRUFBRTtvQkFDckQsQ0FBQyxDQUFDLEVBQUU7b0JBQ0osQ0FBQyxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsV0FBVyxFQUFFLEdBQUcsR0FBRyxDQUFDO29CQUN4QyxLQUFLLENBQUMsYUFBYSxDQUFDLGFBQWEsQ0FBQyxXQUFXLEVBQUU7b0JBQy9DLEdBQUc7b0JBQ0gsS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxXQUFXLEVBQUUsQ0FBQztnQkFFNUMsWUFBWSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsR0FBRyxNQUFNLENBQUM7WUFDM0MsQ0FBQyxDQUFDLENBQUM7WUFFSCxPQUFPLElBQUksK0JBQWMsQ0FBQztnQkFDeEIsVUFBVSxFQUFFLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssTUFBTSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQyxNQUFNLEVBQUU7Z0JBQzFGLEtBQUssRUFBRSxLQUFLLENBQUMsS0FBSztnQkFDbEIsTUFBTSxFQUFFLEtBQUssQ0FBQyxhQUFhLENBQUMsTUFBTTtnQkFDbEMsWUFBWSxFQUFFLFlBQVk7YUFDM0IsQ0FBQyxDQUFDO1FBQ0wsQ0FBQztJQUNILENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSCxNQUFNLENBQUMsd0JBQXdCLENBQzdCLEtBQXlCLEVBQ3pCLFVBQXFDO1FBRXJDLElBQUksY0FBYyxHQUNoQiw2QkFBNkIsQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFFeEUsSUFBSSxjQUFjLENBQUMsTUFBTSxJQUFJLENBQUMsRUFBRSxDQUFDO1lBQy9CLE9BQU8sY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzNCLENBQUM7YUFBTSxDQUFDO1lBQ04sSUFBSSxZQUFZLEdBQStCLEVBQUUsQ0FBQztZQUVsRCxjQUFjLENBQUMsT0FBTyxDQUFDLENBQUMsTUFBZSxFQUFFLEtBQWEsRUFBRSxFQUFFO2dCQUN4RCxJQUFJLFNBQVMsR0FDWCxDQUFDLEtBQUssQ0FBQyxTQUFTLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxTQUFTLElBQUksRUFBRTtvQkFDckQsQ0FBQyxDQUFDLEVBQUU7b0JBQ0osQ0FBQyxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsV0FBVyxFQUFFLEdBQUcsR0FBRyxDQUFDO29CQUN4QyxLQUFLLENBQUMsYUFBYSxDQUFDLGFBQWEsQ0FBQyxXQUFXLEVBQUU7b0JBQy9DLEdBQUc7b0JBQ0gsS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxXQUFXLEVBQUUsQ0FBQztnQkFFNUMsWUFBWSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsR0FBRyxNQUFNLENBQUM7WUFDM0MsQ0FBQyxDQUFDLENBQUM7WUFFSCxPQUFPLElBQUksK0JBQWMsQ0FBQztnQkFDeEIsVUFBVSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQztnQkFDL0MsS0FBSyxFQUFFLEtBQUssQ0FBQyxLQUFLO2dCQUNsQixNQUFNLEVBQUUsS0FBSyxDQUFDLGFBQWEsQ0FBQyxNQUFNO2dCQUNsQyxZQUFZLEVBQUUsWUFBWTthQUMzQixDQUFDLENBQUM7UUFDTCxDQUFDO0lBQ0gsQ0FBQztDQUNGO0FBak9ELHNFQWlPQyIsInNvdXJjZXNDb250ZW50IjpbIi8vIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuLy8gU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbmltcG9ydCB7IElNZXRyaWMsIE1ldHJpYywgTWF0aEV4cHJlc3Npb24gfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtY2xvdWR3YXRjaCc7XG5pbXBvcnQgeyBBdmFpbGFiaWxpdHlNZXRyaWNQcm9wcyB9IGZyb20gJy4vcHJvcHMvQXZhaWxhYmlsaXR5TWV0cmljUHJvcHMnO1xuaW1wb3J0IHsgTGF0ZW5jeU1ldHJpY1Byb3BzIH0gZnJvbSAnLi9wcm9wcy9MYXRlbmN5TWV0cmljUHJvcHMnO1xuaW1wb3J0IHsgQXZhaWxhYmlsaXR5TWV0cmljVHlwZSB9IGZyb20gJy4uL3V0aWxpdGllcy9BdmFpbGFiaWxpdHlNZXRyaWNUeXBlJztcbmltcG9ydCB7IExhdGVuY3lNZXRyaWNUeXBlIH0gZnJvbSAnLi4vdXRpbGl0aWVzL0xhdGVuY3lNZXRyaWNUeXBlJztcblxuLyoqXG4gKiBDbGFzcyBmb3IgY3JlYXRpbmcgYXZhaWxhYmlsaXR5IGFuZCBsYXRlbmN5IG1ldHJpY3MgdGhhdCBjYW4gYmUgdXNlZCBpbiBhbGFybXMgYW5kIGdyYXBoc1xuICovXG5leHBvcnQgY2xhc3MgQXZhaWxhYmlsaXR5QW5kTGF0ZW5jeU1ldHJpY3Mge1xuXG4gIC8qKlxuICAgKiBHZW5lcmFsIHB1cnBvc2UgbWV0aG9kIHRvIGNyZWF0ZSBhdmFpbGFiaWxpdHkgbWV0cmljc1xuICAgKiBAcGFyYW0gcHJvcHNcbiAgICogQHBhcmFtIGRpbWVuc2lvbnNcbiAgICogQHJldHVybnNcbiAgICovXG4gIHN0YXRpYyBjcmVhdGVBdmFpbGFiaWxpdHlNZXRyaWMoXG4gICAgcHJvcHM6IEF2YWlsYWJpbGl0eU1ldHJpY1Byb3BzLFxuICAgIGRpbWVuc2lvbnM6IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH0sXG4gICk6IElNZXRyaWMge1xuICAgIGxldCBjb3VudGVyOiBudW1iZXIgPSAwO1xuICAgIGxldCBrZXk6IHN0cmluZyA9ICcnO1xuXG4gICAgbGV0IHVzaW5nTWV0cmljczogeyBba2V5OiBzdHJpbmddOiBJTWV0cmljIH0gPSB7fTtcblxuICAgIGxldCBzdWNjZXNzS2V5czogc3RyaW5nW10gPSBbXTtcbiAgICBsZXQgZmF1bHRLZXlzOiBzdHJpbmdbXSA9IFtdO1xuXG4gICAgaWYgKFxuICAgICAgcHJvcHMubWV0cmljRGV0YWlscy5zdWNjZXNzTWV0cmljTmFtZXMgIT09IHVuZGVmaW5lZCAmJlxuICAgICAgcHJvcHMubWV0cmljVHlwZSAhPSBBdmFpbGFiaWxpdHlNZXRyaWNUeXBlLkZBVUxUX0NPVU5UXG4gICAgKSB7XG4gICAgICBwcm9wcy5tZXRyaWNEZXRhaWxzLnN1Y2Nlc3NNZXRyaWNOYW1lcy5mb3JFYWNoKFxuICAgICAgICAoc3VjY2Vzc01ldHJpYzogc3RyaW5nKSA9PiB7XG4gICAgICAgICAgbGV0IGtleVByZWZpeCA9XG4gICAgICAgICAgICAocHJvcHMua2V5UHJlZml4ID09PSB1bmRlZmluZWQgfHwgcHJvcHMua2V5UHJlZml4ID09ICcnXG4gICAgICAgICAgICAgID8gJydcbiAgICAgICAgICAgICAgOiBwcm9wcy5rZXlQcmVmaXgudG9Mb3dlckNhc2UoKSArICdfJykgK1xuICAgICAgICAgICAgcHJvcHMubWV0cmljRGV0YWlscy5vcGVyYXRpb25OYW1lLnRvTG93ZXJDYXNlKCkgK1xuICAgICAgICAgICAgJ18nICtcbiAgICAgICAgICAgIHN1Y2Nlc3NNZXRyaWMudG9Mb3dlckNhc2UoKTtcblxuICAgICAgICAgIGtleSA9IGtleVByZWZpeCArICdfJyArIGNvdW50ZXIrKztcbiAgICAgICAgICBzdWNjZXNzS2V5cy5wdXNoKGtleSk7XG5cbiAgICAgICAgICB1c2luZ01ldHJpY3Nba2V5XSA9IG5ldyBNZXRyaWMoe1xuICAgICAgICAgICAgbmFtZXNwYWNlOiBwcm9wcy5tZXRyaWNEZXRhaWxzLm1ldHJpY05hbWVzcGFjZSxcbiAgICAgICAgICAgIG1ldHJpY05hbWU6IHN1Y2Nlc3NNZXRyaWMsXG4gICAgICAgICAgICB1bml0OiBwcm9wcy5tZXRyaWNEZXRhaWxzLnVuaXQsXG4gICAgICAgICAgICBwZXJpb2Q6IHByb3BzLm1ldHJpY0RldGFpbHMucGVyaW9kLFxuICAgICAgICAgICAgc3RhdGlzdGljOiBwcm9wcy5tZXRyaWNEZXRhaWxzLmFsYXJtU3RhdGlzdGljLFxuICAgICAgICAgICAgZGltZW5zaW9uc01hcDogZGltZW5zaW9ucyxcbiAgICAgICAgICAgIGxhYmVsOiBzdWNjZXNzTWV0cmljLFxuICAgICAgICAgIH0pO1xuICAgICAgICB9LFxuICAgICAgKTtcbiAgICB9XG5cbiAgICBpZiAoXG4gICAgICBwcm9wcy5tZXRyaWNEZXRhaWxzLmZhdWx0TWV0cmljTmFtZXMgIT09IHVuZGVmaW5lZCAmJlxuICAgICAgcHJvcHMubWV0cmljVHlwZSAhPSBBdmFpbGFiaWxpdHlNZXRyaWNUeXBlLlNVQ0NFU1NfQ09VTlRcbiAgICApIHtcbiAgICAgIHByb3BzLm1ldHJpY0RldGFpbHMuZmF1bHRNZXRyaWNOYW1lcy5mb3JFYWNoKChmYXVsdE1ldHJpYykgPT4ge1xuICAgICAgICBsZXQga2V5UHJlZml4ID1cbiAgICAgICAgICAocHJvcHMua2V5UHJlZml4ID09PSB1bmRlZmluZWQgfHwgcHJvcHMua2V5UHJlZml4ID09ICcnXG4gICAgICAgICAgICA/ICcnXG4gICAgICAgICAgICA6IHByb3BzLmtleVByZWZpeC50b0xvd2VyQ2FzZSgpICsgJ18nKSArXG4gICAgICAgICAgcHJvcHMubWV0cmljRGV0YWlscy5vcGVyYXRpb25OYW1lLnRvTG93ZXJDYXNlKCkgK1xuICAgICAgICAgICdfJyArXG4gICAgICAgICAgZmF1bHRNZXRyaWMudG9Mb3dlckNhc2UoKTtcblxuICAgICAgICBrZXkgPSBrZXlQcmVmaXggKyAnXycgKyBjb3VudGVyKys7XG4gICAgICAgIGZhdWx0S2V5cy5wdXNoKGtleSk7XG5cbiAgICAgICAgdXNpbmdNZXRyaWNzW2tleV0gPSBuZXcgTWV0cmljKHtcbiAgICAgICAgICBuYW1lc3BhY2U6IHByb3BzLm1ldHJpY0RldGFpbHMubWV0cmljTmFtZXNwYWNlLFxuICAgICAgICAgIG1ldHJpY05hbWU6IGZhdWx0TWV0cmljLFxuICAgICAgICAgIHVuaXQ6IHByb3BzLm1ldHJpY0RldGFpbHMudW5pdCxcbiAgICAgICAgICBwZXJpb2Q6IHByb3BzLm1ldHJpY0RldGFpbHMucGVyaW9kLFxuICAgICAgICAgIHN0YXRpc3RpYzogcHJvcHMubWV0cmljRGV0YWlscy5hbGFybVN0YXRpc3RpYyxcbiAgICAgICAgICBkaW1lbnNpb25zTWFwOiBkaW1lbnNpb25zLFxuICAgICAgICAgIGxhYmVsOiBmYXVsdE1ldHJpYyxcbiAgICAgICAgfSk7XG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBsZXQgZXhwcmVzc2lvbjogc3RyaW5nID0gJyc7XG5cbiAgICBzd2l0Y2ggKHByb3BzLm1ldHJpY1R5cGUpIHtcbiAgICAgIGNhc2UgQXZhaWxhYmlsaXR5TWV0cmljVHlwZS5TVUNDRVNTX1JBVEU6XG4gICAgICAgIGV4cHJlc3Npb24gPSBgKCgke3N1Y2Nlc3NLZXlzLmpvaW4oJysnKX0pIC8gKCR7c3VjY2Vzc0tleXMuam9pbignKycpfSske2ZhdWx0S2V5cy5qb2luKCcrJyl9KSkgKiAxMDBgO1xuICAgICAgICBicmVhaztcbiAgICAgIGNhc2UgQXZhaWxhYmlsaXR5TWV0cmljVHlwZS5SRVFVRVNUX0NPVU5UOlxuICAgICAgICBleHByZXNzaW9uID0gYCR7c3VjY2Vzc0tleXMuam9pbignKycpfSske2ZhdWx0S2V5cy5qb2luKCcrJyl9YDtcbiAgICAgICAgYnJlYWs7XG4gICAgICBjYXNlIEF2YWlsYWJpbGl0eU1ldHJpY1R5cGUuRkFVTFRfQ09VTlQ6XG4gICAgICAgIGV4cHJlc3Npb24gPSBgKCR7ZmF1bHRLZXlzLmpvaW4oJysnKX0pYDtcbiAgICAgICAgYnJlYWs7XG4gICAgICBjYXNlIEF2YWlsYWJpbGl0eU1ldHJpY1R5cGUuRkFVTFRfUkFURTpcbiAgICAgICAgZXhwcmVzc2lvbiA9IGAoKCR7ZmF1bHRLZXlzLmpvaW4oJysnKX0pIC8gKCR7c3VjY2Vzc0tleXMuam9pbignKycpfSske2ZhdWx0S2V5cy5qb2luKCcrJyl9KSkgKiAxMDBgO1xuICAgICAgICBicmVhaztcbiAgICAgIGNhc2UgQXZhaWxhYmlsaXR5TWV0cmljVHlwZS5TVUNDRVNTX0NPVU5UOlxuICAgICAgICBleHByZXNzaW9uID0gYCgke3N1Y2Nlc3NLZXlzLmpvaW4oJysnKX0pYDtcbiAgICAgICAgYnJlYWs7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBNYXRoRXhwcmVzc2lvbih7XG4gICAgICBleHByZXNzaW9uOiBleHByZXNzaW9uLFxuICAgICAgbGFiZWw6IHByb3BzLmxhYmVsLFxuICAgICAgcGVyaW9kOiBwcm9wcy5tZXRyaWNEZXRhaWxzLnBlcmlvZCxcbiAgICAgIHVzaW5nTWV0cmljczogdXNpbmdNZXRyaWNzLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEdlbmVyYWwgcHVycG9zZSBtZXRob2QgdG8gY3JlYXRlIGxhdGVuY3kgbWV0cmljcywgdGhlIHJlYXNvbiB0aGlzIGNyZWF0ZXMgYW4gYXJyYXkgb2YgbWV0cmljcyB3aGlsZSB0aGVcbiAgICogZXF1aXZhbGVudCBhdmFpbGFiaWxpdHkgbWV0cmljIG1ldGhvZCBkb2Vzbid0IGlzIGJlY2F1c2UgaW4gYXZhaWxhYmlsaXR5LCB3ZSBjYW4ganVzdCBzdW0gdGhlIGNvdW50IG9mIGRpZmZlcmVudFxuICAgKiBtZXRyaWMgbmFtZXMgd2hpbGUgZm9yIGxhdGVuY3kgd2UgY2FuJ3Qgc3VtIHRoZSBjb3VudCBiZWNhdXNlIHRoYXQncyBub3Qgd2hhdCdzIGJlaW5nIG1lYXN1cmVkLiBJdCBhbGxvd3MgdGhlXG4gICAqIGNhbGxlciB0byBkZWNpZGUgaWYgdGhleSBvbmx5IHdhbnQgdG8gdGFrZSB0aGUgZmlyc3QgbmFtZSwgb3IgYXZlcmFnZSBhbGwgb2YgdGhlIG5hbWVzXG4gICAqIChsaWtlIFN1Y2Nlc3NMYXRlbmN5IGFuZCBCaWdJdGVtU3VjY2Vzc0xhdGVuY3kpLlxuICAgKlxuICAgKiBAcGFyYW0gcHJvcHNcbiAgICogQHBhcmFtIGRpbWVuc2lvbnNcbiAgICogQHJldHVybnNcbiAgICovXG4gIHN0YXRpYyBjcmVhdGVMYXRlbmN5TWV0cmljcyhcbiAgICBwcm9wczogTGF0ZW5jeU1ldHJpY1Byb3BzLFxuICAgIGRpbWVuc2lvbnM6IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH0sXG4gICk6IElNZXRyaWNbXSB7XG4gICAgbGV0IG5hbWVzOiBzdHJpbmdbXTtcblxuICAgIHN3aXRjaCAocHJvcHMubWV0cmljVHlwZSkge1xuICAgICAgZGVmYXVsdDpcbiAgICAgIGNhc2UgTGF0ZW5jeU1ldHJpY1R5cGUuU1VDQ0VTU19MQVRFTkNZOlxuICAgICAgICBuYW1lcyA9IHByb3BzLm1ldHJpY0RldGFpbHMuc3VjY2Vzc01ldHJpY05hbWVzO1xuICAgICAgICBicmVhaztcbiAgICAgIGNhc2UgTGF0ZW5jeU1ldHJpY1R5cGUuRkFVTFRfTEFURU5DWTpcbiAgICAgICAgbmFtZXMgPSBwcm9wcy5tZXRyaWNEZXRhaWxzLmZhdWx0TWV0cmljTmFtZXM7XG4gICAgICAgIGJyZWFrO1xuICAgIH1cblxuICAgIHJldHVybiBuYW1lcy5tYXAoXG4gICAgICAoeCkgPT5cbiAgICAgICAgbmV3IE1ldHJpYyh7XG4gICAgICAgICAgbWV0cmljTmFtZTogeCxcbiAgICAgICAgICBuYW1lc3BhY2U6IHByb3BzLm1ldHJpY0RldGFpbHMubWV0cmljTmFtZXNwYWNlLFxuICAgICAgICAgIHVuaXQ6IHByb3BzLm1ldHJpY0RldGFpbHMudW5pdCxcbiAgICAgICAgICBwZXJpb2Q6IHByb3BzLm1ldHJpY0RldGFpbHMucGVyaW9kLFxuICAgICAgICAgIHN0YXRpc3RpYzogcHJvcHMuc3RhdGlzdGljLFxuICAgICAgICAgIGRpbWVuc2lvbnNNYXA6IGRpbWVuc2lvbnMsXG4gICAgICAgICAgbGFiZWw6IHByb3BzLmxhYmVsLFxuICAgICAgICB9KSxcbiAgICApO1xuICB9XG5cbiAgLyoqXG4gICAqIFRha2VzIGFsbCBvZiB0aGUgc3VjY2VzcyBvciBmYWlsdXJlIGxhdGVuY3kgbWV0cmljIG5hbWVzIGFuZCBjcmVhdGVzIGFuIGF2ZXJhZ2Ugb2YgdGhvc2VcbiAgICogbmFtZXMsIGlmIHRoZXJlJ3Mgb25seSAxIG5hbWUsIGl0IGp1c3QgcmV0dXJucyB0aGF0IG1ldHJpY1xuICAgKiBAcGFyYW0gcHJvcHNcbiAgICogQHBhcmFtIGRpbWVuc2lvbnNcbiAgICovXG4gIHN0YXRpYyBjcmVhdGVBdmVyYWdlTGF0ZW5jeU1ldHJpYyhcbiAgICBwcm9wczogTGF0ZW5jeU1ldHJpY1Byb3BzLFxuICAgIGRpbWVuc2lvbnM6IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH0sXG4gICk6IElNZXRyaWMge1xuICAgIGxldCBsYXRlbmN5TWV0cmljczogSU1ldHJpY1tdID1cbiAgICAgIEF2YWlsYWJpbGl0eUFuZExhdGVuY3lNZXRyaWNzLmNyZWF0ZUxhdGVuY3lNZXRyaWNzKHByb3BzLCBkaW1lbnNpb25zKTtcblxuICAgIGlmIChsYXRlbmN5TWV0cmljcy5sZW5ndGggPT0gMSkge1xuICAgICAgcmV0dXJuIGxhdGVuY3lNZXRyaWNzWzBdO1xuICAgIH0gZWxzZSB7XG4gICAgICBsZXQgdXNpbmdNZXRyaWNzOiB7IFtrZXk6IHN0cmluZ106IElNZXRyaWMgfSA9IHt9O1xuXG4gICAgICBsYXRlbmN5TWV0cmljcy5mb3JFYWNoKChtZXRyaWM6IElNZXRyaWMsIGluZGV4OiBudW1iZXIpID0+IHtcbiAgICAgICAgbGV0IGtleVByZWZpeDogc3RyaW5nID1cbiAgICAgICAgICAocHJvcHMua2V5UHJlZml4ID09PSB1bmRlZmluZWQgfHwgcHJvcHMua2V5UHJlZml4ID09ICcnXG4gICAgICAgICAgICA/ICcnXG4gICAgICAgICAgICA6IHByb3BzLmtleVByZWZpeC50b0xvd2VyQ2FzZSgpICsgJ18nKSArXG4gICAgICAgICAgcHJvcHMubWV0cmljRGV0YWlscy5vcGVyYXRpb25OYW1lLnRvTG93ZXJDYXNlKCkgK1xuICAgICAgICAgICdfJyArXG4gICAgICAgICAgcHJvcHMubWV0cmljVHlwZS50b1N0cmluZygpLnRvTG93ZXJDYXNlKCk7XG5cbiAgICAgICAgdXNpbmdNZXRyaWNzW2tleVByZWZpeCArIGluZGV4XSA9IG1ldHJpYztcbiAgICAgIH0pO1xuXG4gICAgICByZXR1cm4gbmV3IE1hdGhFeHByZXNzaW9uKHtcbiAgICAgICAgZXhwcmVzc2lvbjogYCgke09iamVjdC5rZXlzKHVzaW5nTWV0cmljcykuam9pbignKycpfSkvJHtPYmplY3Qua2V5cyh1c2luZ01ldHJpY3MpLmxlbmd0aH1gLFxuICAgICAgICBsYWJlbDogcHJvcHMubGFiZWwsXG4gICAgICAgIHBlcmlvZDogcHJvcHMubWV0cmljRGV0YWlscy5wZXJpb2QsXG4gICAgICAgIHVzaW5nTWV0cmljczogdXNpbmdNZXRyaWNzLFxuICAgICAgfSk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBjb3VudCBvZiBoaWdoIGxhdGVuY3kgbWV0cmljcyBmb3IgZWl0aGVyIFN1Y2Nlc3NMYXRlbmN5IG9yIEZhdWx0TGF0ZW5jeSwgd2lsbCB0b3RhbFxuICAgKiB0aGUgY291bnQgb2YgcmVxdWVzdHMgdGhhdCBleGNlZWQgYSB0aHJlc2hvbGQgeW91IGRlZmluZSBpbiB5b3VyIHN0YXRpc3RpYywgbGlrZSBUQygyMDA6KSBhY3Jvc3NcbiAgICogYWxsIG1ldHJpYyBuYW1lcyB0aGF0IGFyZSBwYXJ0IG9mIGVpdGhlciBTdWNjZXNzIG9yIEZhdWx0IGxhdGVuY3kuXG4gICAqIEBwYXJhbSBwcm9wc1xuICAgKiBAcmV0dXJuc1xuICAgKi9cbiAgc3RhdGljIGNyZWF0ZUxhdGVuY3lDb3VudE1ldHJpYyhcbiAgICBwcm9wczogTGF0ZW5jeU1ldHJpY1Byb3BzLFxuICAgIGRpbWVuc2lvbnM6IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH0sXG4gICk6IElNZXRyaWMge1xuICAgIGxldCBsYXRlbmN5TWV0cmljczogSU1ldHJpY1tdID1cbiAgICAgIEF2YWlsYWJpbGl0eUFuZExhdGVuY3lNZXRyaWNzLmNyZWF0ZUxhdGVuY3lNZXRyaWNzKHByb3BzLCBkaW1lbnNpb25zKTtcblxuICAgIGlmIChsYXRlbmN5TWV0cmljcy5sZW5ndGggPT0gMSkge1xuICAgICAgcmV0dXJuIGxhdGVuY3lNZXRyaWNzWzBdO1xuICAgIH0gZWxzZSB7XG4gICAgICBsZXQgdXNpbmdNZXRyaWNzOiB7IFtrZXk6IHN0cmluZ106IElNZXRyaWMgfSA9IHt9O1xuXG4gICAgICBsYXRlbmN5TWV0cmljcy5mb3JFYWNoKChtZXRyaWM6IElNZXRyaWMsIGluZGV4OiBudW1iZXIpID0+IHtcbiAgICAgICAgbGV0IGtleVByZWZpeDogc3RyaW5nID1cbiAgICAgICAgICAocHJvcHMua2V5UHJlZml4ID09PSB1bmRlZmluZWQgfHwgcHJvcHMua2V5UHJlZml4ID09ICcnXG4gICAgICAgICAgICA/ICcnXG4gICAgICAgICAgICA6IHByb3BzLmtleVByZWZpeC50b0xvd2VyQ2FzZSgpICsgJ18nKSArXG4gICAgICAgICAgcHJvcHMubWV0cmljRGV0YWlscy5vcGVyYXRpb25OYW1lLnRvTG93ZXJDYXNlKCkgK1xuICAgICAgICAgICdfJyArXG4gICAgICAgICAgcHJvcHMubWV0cmljVHlwZS50b1N0cmluZygpLnRvTG93ZXJDYXNlKCk7XG5cbiAgICAgICAgdXNpbmdNZXRyaWNzW2tleVByZWZpeCArIGluZGV4XSA9IG1ldHJpYztcbiAgICAgIH0pO1xuXG4gICAgICByZXR1cm4gbmV3IE1hdGhFeHByZXNzaW9uKHtcbiAgICAgICAgZXhwcmVzc2lvbjogT2JqZWN0LmtleXModXNpbmdNZXRyaWNzKS5qb2luKCcrJyksXG4gICAgICAgIGxhYmVsOiBwcm9wcy5sYWJlbCxcbiAgICAgICAgcGVyaW9kOiBwcm9wcy5tZXRyaWNEZXRhaWxzLnBlcmlvZCxcbiAgICAgICAgdXNpbmdNZXRyaWNzOiB1c2luZ01ldHJpY3MsXG4gICAgICB9KTtcbiAgICB9XG4gIH1cbn1cbiJdfQ==