"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationLoadBalancerMetrics = void 0;
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const AvailabilityMetricType_1 = require("../utilities/AvailabilityMetricType");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const MetricsHelper_1 = require("../utilities/MetricsHelper");
class ApplicationLoadBalancerMetrics {
    /**
     * Gets the TargetResponseTime latency for the ALB
     * targets
     * @param props The request props
     * @returns The TargetResponseTime metric for the ALB in the specified AZ
     */
    static getPerAZLatencyMetric(props) {
        return props.alb.metrics.targetResponseTime({
            dimensionsMap: {
                AvailabilityZone: props.availabilityZone,
                LoadBalancer: props.alb
                    .loadBalancerFullName,
            },
            label: props.availabilityZoneId ? props.availabilityZoneId : props.availabilityZone + "-target-response-time",
            period: props.period,
            statistic: props.statistic,
            unit: aws_cloudwatch_1.Unit.SECONDS
        });
    }
    /**
     * Gets either the successful, fault, or total count metrics
     * for the load balancer targets across all of the provided
     * ALBs in each AZ.
     * @param albs The ALBs to aggregate the count of
     * @param props The request props
     * @returns The total count of sucess, faults, or total requests in the specified AZ
     */
    static getPerAZAvailabilityMetricCountAggregate(albs, props) {
        let keyprefix = props.keyprefix ? props.keyprefix : props.availabilityZone.substring(props.availabilityZone.length - 1);
        let usingMetrics = {};
        let metrics = [];
        albs.forEach((alb) => {
            let target5xx = alb.metrics.httpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_5XX_COUNT, {
                dimensionsMap: {
                    AvailabilityZone: props.availabilityZone,
                    LoadBalancer: alb
                        .loadBalancerFullName,
                },
                label: props.availabilityZoneId + "-" + alb.loadBalancerArn + "-target-5xx",
                period: props.period,
                statistic: aws_cloudwatch_1.Stats.SUM,
                unit: aws_cloudwatch_1.Unit.COUNT
            });
            let elb5xx = alb.metrics.httpCodeElb(aws_elasticloadbalancingv2_1.HttpCodeElb.ELB_5XX_COUNT, {
                dimensionsMap: {
                    AvailabilityZone: props.availabilityZone,
                    LoadBalancer: alb
                        .loadBalancerFullName,
                },
                label: props.availabilityZoneId + "-" + alb.loadBalancerArn + "-elb-5xx",
                period: props.period,
                statistic: aws_cloudwatch_1.Stats.SUM,
                unit: aws_cloudwatch_1.Unit.COUNT
            });
            let requestCount = alb.metrics.requestCount({
                dimensionsMap: {
                    AvailabilityZone: props.availabilityZone,
                    LoadBalancer: alb
                        .loadBalancerFullName,
                },
                label: props.availabilityZoneId + "-" + alb.loadBalancerArn + "-request-count",
                period: props.period,
                statistic: aws_cloudwatch_1.Stats.SUM,
                unit: aws_cloudwatch_1.Unit.COUNT
            });
            let target2xx = alb.metrics.httpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_2XX_COUNT, {
                dimensionsMap: {
                    AvailabilityZone: props.availabilityZone,
                    LoadBalancer: alb
                        .loadBalancerFullName,
                },
                label: props.availabilityZoneId + "-" + alb.loadBalancerArn + "-target-2xx",
                period: props.period,
                statistic: aws_cloudwatch_1.Stats.SUM,
                unit: aws_cloudwatch_1.Unit.COUNT
            });
            let elb3xx = alb.metrics.httpCodeElb(aws_elasticloadbalancingv2_1.HttpCodeElb.ELB_3XX_COUNT, {
                dimensionsMap: {
                    AvailabilityZone: props.availabilityZone,
                    LoadBalancer: alb
                        .loadBalancerFullName,
                },
                label: props.availabilityZoneId + "-" + alb.loadBalancerArn + "-elb-3xx",
                period: props.period,
                statistic: aws_cloudwatch_1.Stats.SUM,
                unit: aws_cloudwatch_1.Unit.COUNT
            });
            let target3xx = alb.metrics.httpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_3XX_COUNT, {
                dimensionsMap: {
                    AvailabilityZone: props.availabilityZone,
                    LoadBalancer: alb
                        .loadBalancerFullName,
                },
                label: props.availabilityZoneId + "-" + alb.loadBalancerArn + "-target-3xx",
                period: props.period,
                statistic: aws_cloudwatch_1.Stats.SUM,
                unit: aws_cloudwatch_1.Unit.COUNT
            });
            switch (props.metricType) {
                case AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT:
                    usingMetrics[`${keyprefix}1`] = target5xx;
                    usingMetrics[`${keyprefix}2`] = elb5xx;
                    metrics.push(new aws_cloudwatch_1.MathExpression({
                        expression: `FILL(${keyprefix}1, 0) + FILL(${keyprefix}2, 0)`,
                        usingMetrics: usingMetrics,
                        label: props.availabilityZoneId + "-" + alb.loadBalancerArn + "-fault-count",
                        period: props.period,
                    }));
                    break;
                case AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_COUNT:
                    usingMetrics[`${keyprefix}1`] = target2xx;
                    usingMetrics[`${keyprefix}2`] = target3xx;
                    usingMetrics[`${keyprefix}3`] = elb3xx;
                    metrics.push(new aws_cloudwatch_1.MathExpression({
                        expression: `${keyprefix}1+${keyprefix}2+${keyprefix}3`,
                        usingMetrics: usingMetrics,
                        label: props.availabilityZoneId + "-" + alb.loadBalancerArn + "-success-count",
                        period: props.period,
                    }));
                    break;
                case AvailabilityMetricType_1.AvailabilityMetricType.REQUEST_COUNT:
                    metrics.push(requestCount);
                    break;
                default:
                    throw new Error("This method only supports COUNT availability metrics.");
            }
            keyprefix = MetricsHelper_1.MetricsHelper.nextChar(keyprefix);
        });
        metrics.forEach((metric, index) => {
            usingMetrics[`${keyprefix}${index}`] = metric;
        });
        return new aws_cloudwatch_1.MathExpression({
            expression: Object.keys(usingMetrics).join("+"),
            usingMetrics: usingMetrics,
            label: props.label,
            period: props.period,
        });
    }
    /**
     * Gets a specified availability metric in the specified AZ for the ALB
     * @param alb The ALB
     * @param props The request props
     * @returns The metric requested
     */
    static getPerAZAvailabilityMetric(alb, props) {
        let keyprefix = props.keyprefix ? props.keyprefix : props.availabilityZone.substring(props.availabilityZone.length - 1);
        let target5xx = alb.metrics.httpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_5XX_COUNT, {
            dimensionsMap: {
                AvailabilityZone: props.availabilityZone,
                LoadBalancer: alb
                    .loadBalancerFullName,
            },
            label: props.availabilityZoneId + "-target-5xx",
            period: props.period,
            statistic: aws_cloudwatch_1.Stats.SUM,
            unit: aws_cloudwatch_1.Unit.COUNT
        });
        let elb5xx = alb.metrics.httpCodeElb(aws_elasticloadbalancingv2_1.HttpCodeElb.ELB_5XX_COUNT, {
            dimensionsMap: {
                AvailabilityZone: props.availabilityZone,
                LoadBalancer: alb
                    .loadBalancerFullName,
            },
            label: props.availabilityZoneId + "-elb-5xx",
            period: props.period,
            statistic: aws_cloudwatch_1.Stats.SUM,
            unit: aws_cloudwatch_1.Unit.COUNT
        });
        let requestCount = alb.metrics.requestCount({
            dimensionsMap: {
                AvailabilityZone: props.availabilityZone,
                LoadBalancer: alb
                    .loadBalancerFullName,
            },
            label: props.availabilityZoneId,
            period: props.period,
            statistic: aws_cloudwatch_1.Stats.SUM,
            unit: aws_cloudwatch_1.Unit.COUNT
        });
        let target2xx = alb.metrics.httpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_2XX_COUNT, {
            dimensionsMap: {
                AvailabilityZone: props.availabilityZone,
                LoadBalancer: alb
                    .loadBalancerFullName,
            },
            label: props.availabilityZoneId + "-target-2xx",
            period: props.period,
            statistic: aws_cloudwatch_1.Stats.SUM,
            unit: aws_cloudwatch_1.Unit.COUNT
        });
        let elb3xx = alb.metrics.httpCodeElb(aws_elasticloadbalancingv2_1.HttpCodeElb.ELB_3XX_COUNT, {
            dimensionsMap: {
                AvailabilityZone: props.availabilityZone,
                LoadBalancer: alb
                    .loadBalancerFullName,
            },
            label: props.availabilityZoneId + "-elb-3xx",
            period: props.period,
            statistic: aws_cloudwatch_1.Stats.SUM,
            unit: aws_cloudwatch_1.Unit.COUNT
        });
        let target3xx = alb.metrics.httpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_3XX_COUNT, {
            dimensionsMap: {
                AvailabilityZone: props.availabilityZone,
                LoadBalancer: alb
                    .loadBalancerFullName,
            },
            label: props.availabilityZoneId + "-target-3xx",
            period: props.period,
            statistic: aws_cloudwatch_1.Stats.SUM,
            unit: aws_cloudwatch_1.Unit.COUNT
        });
        let usingMetrics = {};
        switch (props.metricType) {
            case AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT:
                usingMetrics[`${keyprefix}1`] = target5xx;
                usingMetrics[`${keyprefix}2`] = elb5xx;
                return new aws_cloudwatch_1.MathExpression({
                    expression: `FILL(${keyprefix}1, 0) + FILL(${keyprefix}2, 0)`,
                    usingMetrics: usingMetrics,
                    label: props.label,
                    period: props.period,
                });
            case AvailabilityMetricType_1.AvailabilityMetricType.FAULT_RATE:
                // Request count only includes requests where a response was generated from a target
                usingMetrics[`${keyprefix}1`] = target5xx;
                usingMetrics[`${keyprefix}2`] = requestCount;
                return new aws_cloudwatch_1.MathExpression({
                    expression: `(${keyprefix}1/${keyprefix}2)*100`,
                    usingMetrics: usingMetrics,
                    label: props.label,
                    period: props.period,
                });
            case AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_COUNT:
                usingMetrics[`${keyprefix}1`] = target2xx;
                usingMetrics[`${keyprefix}2`] = target3xx;
                usingMetrics[`${keyprefix}3`] = elb3xx;
                return new aws_cloudwatch_1.MathExpression({
                    expression: `${keyprefix}1+${keyprefix}2+${keyprefix}3`,
                    usingMetrics: usingMetrics,
                    label: props.label,
                    period: props.period,
                });
            case AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE:
                usingMetrics[`${keyprefix}1`] = target2xx;
                usingMetrics[`${keyprefix}2`] = target3xx;
                usingMetrics[`${keyprefix}3`] = requestCount;
                return new aws_cloudwatch_1.MathExpression({
                    expression: `((${keyprefix}1+${keyprefix}2)/${keyprefix}3)*100`,
                    usingMetrics: usingMetrics,
                    label: props.label,
                    period: props.period,
                });
            case AvailabilityMetricType_1.AvailabilityMetricType.REQUEST_COUNT:
                return requestCount;
        }
    }
    /**
     * Gets the ALBs specified availability metric at the regional level (only looks
     * at the regional dimension for the targets, not per AZ)
     * @param alb The ALB
     * @param props The request props
     * @returns The regional availability metric requested
     */
    static getRegionalAvailabilityMetric(alb, props) {
        let keyprefix = props.keyprefix ? props.keyprefix : "a";
        let target5xx = alb.metrics.httpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_5XX_COUNT, {
            dimensionsMap: {
                LoadBalancer: alb
                    .loadBalancerFullName,
            },
            label: "target-5xx",
            period: props.period,
            statistic: aws_cloudwatch_1.Stats.SUM,
            unit: aws_cloudwatch_1.Unit.COUNT
        });
        let elb5xx = alb.metrics.httpCodeElb(aws_elasticloadbalancingv2_1.HttpCodeElb.ELB_5XX_COUNT, {
            dimensionsMap: {
                LoadBalancer: alb
                    .loadBalancerFullName,
            },
            label: "elb-5xx",
            period: props.period,
            statistic: aws_cloudwatch_1.Stats.SUM,
            unit: aws_cloudwatch_1.Unit.COUNT
        });
        let requestCount = alb.metrics.requestCount({
            dimensionsMap: {
                LoadBalancer: alb
                    .loadBalancerFullName,
            },
            label: "request-count",
            period: props.period,
            statistic: aws_cloudwatch_1.Stats.SUM,
            unit: aws_cloudwatch_1.Unit.COUNT
        });
        let target2xx = alb.metrics.httpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_2XX_COUNT, {
            dimensionsMap: {
                LoadBalancer: alb
                    .loadBalancerFullName,
            },
            label: "target-2xx",
            period: props.period,
            statistic: aws_cloudwatch_1.Stats.SUM,
            unit: aws_cloudwatch_1.Unit.COUNT
        });
        let elb3xx = alb.metrics.httpCodeElb(aws_elasticloadbalancingv2_1.HttpCodeElb.ELB_3XX_COUNT, {
            dimensionsMap: {
                LoadBalancer: alb
                    .loadBalancerFullName,
            },
            label: "elb-3xx",
            period: props.period,
            statistic: aws_cloudwatch_1.Stats.SUM,
            unit: aws_cloudwatch_1.Unit.COUNT
        });
        let target3xx = alb.metrics.httpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_3XX_COUNT, {
            dimensionsMap: {
                LoadBalancer: alb
                    .loadBalancerFullName,
            },
            label: "target-3xx",
            period: props.period,
            statistic: aws_cloudwatch_1.Stats.SUM,
            unit: aws_cloudwatch_1.Unit.COUNT
        });
        let usingMetrics = {};
        switch (props.metricType) {
            case AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT:
                usingMetrics[`${keyprefix}1`] = target5xx;
                usingMetrics[`${keyprefix}2`] = elb5xx;
                return new aws_cloudwatch_1.MathExpression({
                    expression: `FILL(${keyprefix}1, 0) + FILL(${keyprefix}2, 0)`,
                    usingMetrics: usingMetrics,
                    label: props.label,
                    period: props.period,
                });
            case AvailabilityMetricType_1.AvailabilityMetricType.FAULT_RATE:
                usingMetrics[`${keyprefix}1`] = target5xx;
                usingMetrics[`${keyprefix}2`] = requestCount;
                return new aws_cloudwatch_1.MathExpression({
                    expression: `(${keyprefix}1/${keyprefix}2)*100`,
                    usingMetrics: usingMetrics,
                    label: props.label,
                    period: props.period,
                });
            case AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_COUNT:
                usingMetrics[`${keyprefix}1`] = target2xx;
                usingMetrics[`${keyprefix}2`] = target3xx;
                usingMetrics[`${keyprefix}3`] = elb3xx;
                return new aws_cloudwatch_1.MathExpression({
                    expression: `${keyprefix}1+${keyprefix}2+${keyprefix}3`,
                    usingMetrics: usingMetrics,
                    label: props.label,
                    period: props.period,
                });
            case AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE:
                usingMetrics[`${keyprefix}1`] = target2xx;
                usingMetrics[`${keyprefix}2`] = target3xx;
                usingMetrics[`${keyprefix}3`] = requestCount;
                return new aws_cloudwatch_1.MathExpression({
                    expression: `((${keyprefix}1+${keyprefix}2)/${keyprefix}3)*100`,
                    usingMetrics: usingMetrics,
                    label: props.label,
                    period: props.period,
                });
            case AvailabilityMetricType_1.AvailabilityMetricType.REQUEST_COUNT:
                return requestCount;
        }
    }
    /**
     * Gets the ALB's TargetResponseTime metric at the regional level (only looks
     * at the regional dimension for the targets, not per AZ)
     * @param props  The request props
     * @returns
     */
    static getRegionalLatencyMetric(props) {
        return props.alb.metrics.targetResponseTime({
            dimensionsMap: {
                LoadBalancer: props.alb
                    .loadBalancerFullName,
            },
            label: aws_cdk_lib_1.Aws.REGION,
            period: props.period,
            statistic: props.statistic,
            unit: aws_cloudwatch_1.Unit.SECONDS
        });
    }
    /**
     * Calculates the total number of faults for all ALBs combined per AZ
     * @param albs The ALBs to aggregate
     * @param period The period of time to calculate the metrics
     * @param azMapper The AZ mapper function so the metrics are correctly labeled with their AZ ID
     * @returns
     */
    static getTotalAlbFaultCountPerZone(albs, period, azMapper, prefix) {
        let faultsPerZone = {};
        let metricsPerAZ = {};
        let keyprefix = prefix ? prefix : MetricsHelper_1.MetricsHelper.nextChar();
        albs.forEach((alb) => {
            alb.vpc.availabilityZones.forEach((availabilityZone) => {
                let azLetter = availabilityZone.substring(availabilityZone.length - 1);
                let availabilityZoneId = azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
                if (!(azLetter in metricsPerAZ)) {
                    metricsPerAZ[azLetter] = [];
                }
                // 5xx responses from targets
                let target5xx = alb.metrics.httpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_5XX_COUNT, {
                    dimensionsMap: {
                        AvailabilityZone: availabilityZone,
                        LoadBalancer: alb
                            .loadBalancerFullName,
                    },
                    label: availabilityZoneId,
                    period: period,
                    statistic: aws_cloudwatch_1.Stats.SUM,
                    unit: aws_cloudwatch_1.Unit.COUNT
                });
                // 5xx responses from ELB
                let elb5xx = alb.metrics.httpCodeElb(aws_elasticloadbalancingv2_1.HttpCodeElb.ELB_5XX_COUNT, {
                    dimensionsMap: {
                        AvailabilityZone: availabilityZone,
                        LoadBalancer: alb
                            .loadBalancerFullName,
                    },
                    label: availabilityZoneId,
                    period: period,
                    statistic: aws_cloudwatch_1.Stats.SUM,
                    unit: aws_cloudwatch_1.Unit.COUNT
                });
                let usingMetrics = {};
                usingMetrics[`${keyprefix}1`] = elb5xx;
                usingMetrics[`${keyprefix}2`] = target5xx;
                // This is the total number of faults per zone for this load balancer
                metricsPerAZ[azLetter].push(new aws_cloudwatch_1.MathExpression({
                    expression: `FILL(${keyprefix}1, 0) + FILL(${keyprefix}2, 0)`,
                    usingMetrics: usingMetrics,
                    period: period,
                    label: availabilityZoneId
                }));
                keyprefix = MetricsHelper_1.MetricsHelper.nextChar(keyprefix);
            });
        });
        // We can have multiple load balancers per zone, so add their fault count
        // metrics together
        Object.keys(metricsPerAZ).forEach((azLetter) => {
            let availabilityZoneId = azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
            if (metricsPerAZ[azLetter].length > 1) {
                let usingMetrics = {};
                metricsPerAZ[azLetter].forEach((metric, index) => {
                    usingMetrics[`${keyprefix}${index}`] = metric;
                });
                faultsPerZone[azLetter] = new aws_cloudwatch_1.MathExpression({
                    expression: Object.keys(usingMetrics).join("+"),
                    usingMetrics: usingMetrics,
                    label: availabilityZoneId,
                    period: period
                });
            }
            else {
                faultsPerZone[azLetter] = metricsPerAZ[azLetter][0];
            }
        });
        return faultsPerZone;
    }
    /**
     * Calculates the total number of faults for all ALBs combined per AZ
     * @param albs The ALBs to aggregate
     * @param period The period of time to calculate the metrics
     * @param azMapper The AZ mapper function so the metrics are correctly labeled with their AZ ID
     * @returns
     */
    static getTotalAlbSuccessCountPerZone(albs, period, azMapper) {
        let successPerZone = {};
        let metricsPerAZ = {};
        let keyprefix = MetricsHelper_1.MetricsHelper.nextChar();
        albs.forEach((alb) => {
            alb.vpc.availabilityZones.forEach((availabilityZone) => {
                let azLetter = availabilityZone.substring(availabilityZone.length - 1);
                let availabilityZoneId = azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
                if (!(azLetter in metricsPerAZ)) {
                    metricsPerAZ[azLetter] = [];
                }
                // 2xx responses from targets
                let target2xx = alb.metrics.httpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_2XX_COUNT, {
                    dimensionsMap: {
                        AvailabilityZone: availabilityZone,
                        LoadBalancer: alb
                            .loadBalancerFullName,
                    },
                    label: availabilityZoneId,
                    period: period,
                    statistic: aws_cloudwatch_1.Stats.SUM,
                    unit: aws_cloudwatch_1.Unit.COUNT
                });
                // 3xx responses from targets
                let target3xx = alb.metrics.httpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_3XX_COUNT, {
                    dimensionsMap: {
                        AvailabilityZone: availabilityZone,
                        LoadBalancer: alb
                            .loadBalancerFullName,
                    },
                    label: availabilityZoneId,
                    period: period,
                    statistic: aws_cloudwatch_1.Stats.SUM,
                    unit: aws_cloudwatch_1.Unit.COUNT
                });
                // 3xx responses from ELB
                let elb3xx = alb.metrics.httpCodeElb(aws_elasticloadbalancingv2_1.HttpCodeElb.ELB_3XX_COUNT, {
                    dimensionsMap: {
                        AvailabilityZone: availabilityZone,
                        LoadBalancer: alb
                            .loadBalancerFullName,
                    },
                    label: availabilityZoneId,
                    period: period,
                    statistic: aws_cloudwatch_1.Stats.SUM,
                    unit: aws_cloudwatch_1.Unit.COUNT
                });
                let usingMetrics = {};
                usingMetrics[`${keyprefix}1`] = elb3xx;
                usingMetrics[`${keyprefix}2`] = target2xx;
                usingMetrics[`${keyprefix}3`] = target3xx;
                // This is the total number of faults per zone for this load balancer
                metricsPerAZ[azLetter].push(new aws_cloudwatch_1.MathExpression({
                    expression: `FILL(${keyprefix}1, 0) + FILL(${keyprefix}2, 0) + FILL(${keyprefix}3, 0)`,
                    usingMetrics: usingMetrics,
                    period: period,
                    label: availabilityZoneId
                }));
                keyprefix = MetricsHelper_1.MetricsHelper.nextChar(keyprefix);
            });
        });
        // We can have multiple load balancers per zone, so add their success count
        // metrics together
        Object.keys(metricsPerAZ).forEach((azLetter) => {
            let availabilityZoneId = azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
            if (metricsPerAZ[azLetter].length > 1) {
                let usingMetrics = {};
                metricsPerAZ[azLetter].forEach((metric, index) => {
                    usingMetrics[`${keyprefix}${index}`] = metric;
                });
                successPerZone[azLetter] = new aws_cloudwatch_1.MathExpression({
                    expression: Object.keys(usingMetrics).join("+"),
                    usingMetrics: usingMetrics,
                    label: availabilityZoneId,
                    period: period
                });
            }
            else {
                successPerZone[azLetter] = metricsPerAZ[azLetter][0];
            }
        });
        return successPerZone;
    }
    /**
     * Calculates the total number of processed bytes for all ALBs in each zone
     * @param albs
     * @param period
     * @param azMapper
     * @returns
     */
    static getTotalAlbRequestsPerZone(albs, period, azMapper, prefix) {
        let requestsPerZone = {};
        let metricsPerAZ = {};
        let keyprefix = prefix ? prefix : MetricsHelper_1.MetricsHelper.nextChar();
        albs.forEach((alb) => {
            alb.vpc.availabilityZones.forEach((availabilityZone) => {
                let azLetter = availabilityZone.substring(availabilityZone.length - 1);
                let availabilityZoneId = azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
                if (!(azLetter in metricsPerAZ)) {
                    metricsPerAZ[azLetter] = [];
                }
                let requestCount = alb.metrics.requestCount({
                    dimensionsMap: {
                        AvailabilityZone: availabilityZone,
                        LoadBalancer: alb
                            .loadBalancerFullName,
                    },
                    label: availabilityZoneId,
                    period: period,
                    statistic: aws_cloudwatch_1.Stats.SUM,
                    unit: aws_cloudwatch_1.Unit.COUNT
                });
                metricsPerAZ[azLetter].push(requestCount);
            });
        });
        // We can have multiple load balancers per zone, so add their processed bytes together
        Object.keys(metricsPerAZ).forEach((azLetter) => {
            let availabilityZoneId = azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
            if (metricsPerAZ[azLetter].length > 1) {
                let usingMetrics = {};
                metricsPerAZ[azLetter].forEach((metric, index) => {
                    usingMetrics[`${keyprefix}${index}`] = metric;
                });
                keyprefix = MetricsHelper_1.MetricsHelper.nextChar(keyprefix);
                requestsPerZone[azLetter] = new aws_cloudwatch_1.MathExpression({
                    expression: Object.keys(usingMetrics).join("+"),
                    usingMetrics: usingMetrics,
                    label: availabilityZoneId,
                    period: period
                });
            }
            else {
                requestsPerZone[azLetter] = metricsPerAZ[azLetter][0];
            }
        });
        return requestsPerZone;
    }
    /**
     * Calculates the total number of processed bytes for all ALBs in each zone
     * @param albs
     * @param period
     * @param azMapper
     * @returns
     */
    static getTotalAlbProcessedBytesPerZone(albs, period, azMapper) {
        let processedBytesPerZone = {};
        let metricsPerAZ = {};
        let keyprefix = MetricsHelper_1.MetricsHelper.nextChar();
        albs.forEach((alb) => {
            alb.vpc.availabilityZones.forEach((availabilityZone) => {
                let azLetter = availabilityZone.substring(availabilityZone.length - 1);
                let availabilityZoneId = azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
                if (!(azLetter in metricsPerAZ)) {
                    metricsPerAZ[azLetter] = [];
                }
                let processedBytes = alb.metrics.processedBytes({
                    dimensionsMap: {
                        AvailabilityZone: availabilityZone,
                        LoadBalancer: alb
                            .loadBalancerFullName,
                    },
                    label: availabilityZoneId,
                    period: period,
                    statistic: aws_cloudwatch_1.Stats.SUM,
                    unit: aws_cloudwatch_1.Unit.COUNT
                });
                metricsPerAZ[azLetter].push(processedBytes);
            });
        });
        // We can have multiple load balancers per zone, so add their processed bytes together
        Object.keys(metricsPerAZ).forEach((azLetter) => {
            let availabilityZoneId = azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
            if (metricsPerAZ[azLetter].length > 1) {
                let usingMetrics = {};
                metricsPerAZ[azLetter].forEach((metric, index) => {
                    usingMetrics[`${keyprefix}${index}`] = metric;
                });
                keyprefix = MetricsHelper_1.MetricsHelper.nextChar(keyprefix);
                processedBytesPerZone[azLetter] = new aws_cloudwatch_1.MathExpression({
                    expression: Object.keys(usingMetrics).join("+"),
                    usingMetrics: usingMetrics,
                    label: availabilityZoneId,
                    period: period
                });
            }
            else {
                processedBytesPerZone[azLetter] = metricsPerAZ[azLetter][0];
            }
        });
        return processedBytesPerZone;
    }
    /**
     * Calculates a weighted approximation of the latency at the provided statistic for all ALBs
     * in each zone.
     * @param albs
     * @param statistic
     * @param period
     * @param azMapper
     * @returns
     */
    static getTotalAlbLatencyPerZone(albs, statistic, period, azMapper) {
        let latencyPerZone = {};
        let keyprefix = MetricsHelper_1.MetricsHelper.nextChar();
        //let requestCountsPerAZ: {[key: string]: IMetric[]} = {};
        let requestCountsPerAZMetricKeys = {};
        let weightedLatencyPerAZ = {};
        albs.forEach((alb) => {
            alb.vpc.availabilityZones.forEach((availabilityZone) => {
                let azLetter = availabilityZone.substring(availabilityZone.length - 1);
                if (!(azLetter in weightedLatencyPerAZ)) {
                    weightedLatencyPerAZ[azLetter] = [];
                }
                //if (!(azLetter in requestCountsPerAZ)) {
                //  requestCountsPerAZ[azLetter] = [];
                //}
                if (!(azLetter in requestCountsPerAZMetricKeys)) {
                    requestCountsPerAZMetricKeys[azLetter] = [];
                }
                let availabilityZoneId = azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
                let latency = alb.metrics.targetResponseTime({
                    dimensionsMap: {
                        AvailabilityZone: availabilityZone,
                        LoadBalancer: alb
                            .loadBalancerFullName,
                    },
                    label: availabilityZoneId,
                    period: period,
                    statistic: statistic,
                    unit: aws_cloudwatch_1.Unit.SECONDS
                });
                let requestCount = alb.metrics.requestCount({
                    dimensionsMap: {
                        AvailabilityZone: availabilityZone,
                        LoadBalancer: alb
                            .loadBalancerFullName,
                    },
                    label: availabilityZoneId,
                    period: period,
                    statistic: aws_cloudwatch_1.Stats.SUM,
                    unit: aws_cloudwatch_1.Unit.COUNT
                });
                let usingMetrics = {};
                usingMetrics[`${keyprefix}1`] = latency;
                usingMetrics[`${keyprefix}2`] = requestCount;
                let weightedLatency = new aws_cloudwatch_1.MathExpression({
                    expression: `${keyprefix}1*${keyprefix}2`,
                    usingMetrics: usingMetrics,
                    period: period,
                    label: availabilityZoneId
                });
                weightedLatencyPerAZ[azLetter].push(weightedLatency);
                //requestCountsPerAZ[azLetter].push(requestCount);
                requestCountsPerAZMetricKeys[azLetter].push(`${keyprefix}2`);
                keyprefix = MetricsHelper_1.MetricsHelper.nextChar(keyprefix);
            });
        });
        // We can have multiple load balancers per zone, combine their latency per zone
        // to get an average latency percentile latency, like average p99
        Object.keys(weightedLatencyPerAZ).forEach((azLetter) => {
            let availabilityZoneId = azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
            let usingMetrics = {};
            weightedLatencyPerAZ[azLetter].forEach((metric, index) => {
                usingMetrics[`${keyprefix}${index}`] = metric;
            });
            let numerator = "(" + Object.keys(usingMetrics).join("+") + ")";
            //let index = Object.keys(usingMetrics).length;
            keyprefix = MetricsHelper_1.MetricsHelper.nextChar(keyprefix);
            // TODO: We're duplicating the same metric with 2 different keys for request count
            //requestCountsPerAZ[azLetter].forEach((metric: IMetric, index: number) => {
            //  usingMetrics[`${keyprefix}${index}`] = metric;
            //}); 
            //let denominator: string = "(" + Object.keys(usingMetrics).slice(index).join("+") + ")";
            let denominator = "(" + requestCountsPerAZMetricKeys[azLetter].join("+") + ")";
            /**
             * We want to calculate this formula
             *
             * (p99_1 * requests_1 + p99_2 * requests_2 + p99_3 * requests_3) / (requests_1 + requests_2 + requests_3)
             *
             * This will provide a request weighted approximation of the p99
             * latency per AZ
             */
            latencyPerZone[azLetter] = new aws_cloudwatch_1.MathExpression({
                expression: "(" + numerator + "/" + denominator + ") * 1000",
                usingMetrics: usingMetrics,
                label: availabilityZoneId,
                period: period
            });
            keyprefix = MetricsHelper_1.MetricsHelper.nextChar(keyprefix);
        });
        return latencyPerZone;
    }
    /**
     * Calculates the fault rate per AZ
     * @param requestsPerZone
     * @param faultsPerZone
     * @param period
     * @param azMapper
     * @returns The fault rate per AZ using the AZ name letter as the key for each metric
     */
    static getTotalAlbFaultRatePerZone(albs, period, azMapper) {
        let faultRateMetrics = {};
        let requestsPerZone = this.getTotalAlbRequestsPerZone(albs, period, azMapper, "a");
        let faultsPerZone = this.getTotalAlbFaultCountPerZone(albs, period, azMapper, "e");
        Object.keys(requestsPerZone).forEach((key) => {
            if (key in faultsPerZone) {
                let usingMetrics = {};
                let keyprefix = 'z' + key;
                usingMetrics[`${keyprefix}1`] = faultsPerZone[key];
                usingMetrics[`${keyprefix}2`] = requestsPerZone[key];
                let zonalFaultRate = new aws_cloudwatch_1.MathExpression({
                    expression: `(${keyprefix}1/${keyprefix}2) * 100`,
                    usingMetrics: usingMetrics,
                    period: period,
                    label: azMapper.availabilityZoneIdFromAvailabilityZoneLetter(key)
                });
                faultRateMetrics[key] = zonalFaultRate;
            }
            else {
                throw new Error("The zone " + key + " is not present in the faultsPerZone parameter.");
            }
        });
        Object.keys(faultsPerZone).forEach((key) => {
            if (!(key in requestsPerZone)) {
                throw new Error("The zone " + key + " is not present in the requestsPerZone parameter.");
            }
        });
        return faultRateMetrics;
    }
    /**
     * Creates a zonal processed bytes metric for the specified load balancer
     * @param loadBalancerFullName
     * @param availabilityZoneName
     * @param period
     * @returns IMetric
     */
    static getPerAZProcessedBytesMetric(alb, availabilityZone, availabilityZoneId, period, addLoadBalancerArnToLabel) {
        return alb.metrics.processedBytes({
            period: period,
            dimensionsMap: {
                LoadBalancer: alb.loadBalancerFullName,
                AvailabilityZone: availabilityZone,
            },
            label: availabilityZoneId + (addLoadBalancerArnToLabel ? "-" + alb.loadBalancerArn : "")
        });
    }
    /**
     * Creates a zonal processed bytes metric for the specified load balancer
     * @param loadBalancerFullName
     * @param availabilityZoneName
     * @param period
     * @returns IMetric
     */
    static getRegionalProcessedBytesMetric(alb, period, addLoadBalancerArn) {
        return alb.metrics.processedBytes({
            period: period,
            dimensionsMap: {
                LoadBalancer: alb.loadBalancerFullName,
            },
            label: aws_cdk_lib_1.Aws.REGION + (addLoadBalancerArn ? "-" + alb.loadBalancerArn : "")
        });
    }
}
exports.ApplicationLoadBalancerMetrics = ApplicationLoadBalancerMetrics;
//# sourceMappingURL=data:application/json;base64,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