"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceAvailabilityAndLatencyDashboard = void 0;
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
const ApplicationLoadBalancerMetrics_1 = require("../metrics/ApplicationLoadBalancerMetrics");
const RegionalAvailabilityMetrics_1 = require("../metrics/RegionalAvailabilityMetrics");
const RegionalLatencyMetrics_1 = require("../metrics/RegionalLatencyMetrics");
const ZonalAvailabilityMetrics_1 = require("../metrics/ZonalAvailabilityMetrics");
const ZonalLatencyMetrics_1 = require("../metrics/ZonalLatencyMetrics");
const AvailabilityMetricType_1 = require("../utilities/AvailabilityMetricType");
const LatencyMetricType_1 = require("../utilities/LatencyMetricType");
const MetricsHelper_1 = require("../utilities/MetricsHelper");
/**
 * Creates a service level availability and latency dashboard
 */
class ServiceAvailabilityAndLatencyDashboard extends constructs_1.Construct {
    static generateTPSWidgets(props, availabilityZoneIds) {
        let widgets = [];
        widgets.push(new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: '**TPS Metrics**' }));
        widgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 6,
            width: 24,
            title: aws_cdk_lib_1.Fn.ref('AWS::Region') + ' TPS',
            region: aws_cdk_lib_1.Fn.ref('AWS::Region'),
            left: RegionalAvailabilityMetrics_1.RegionalAvailabilityMetrics.createRegionalServiceAvailabilityMetrics({
                label: aws_cdk_lib_1.Fn.ref('AWS::Region') + ' tps',
                period: props.service.period,
                availabilityMetricProps: props.service.operations
                    .filter((x) => x.critical)
                    .map((x) => {
                    return {
                        label: x.operationName,
                        metricDetails: x.serverSideAvailabilityMetricDetails,
                        metricType: AvailabilityMetricType_1.AvailabilityMetricType.REQUEST_COUNT,
                    };
                }),
            }),
            statistic: 'Sum',
            leftYAxis: {
                label: 'TPS',
                showUnits: false,
            },
        }));
        for (let i = 0; i < availabilityZoneIds.length; i++) {
            let availabilityZoneId = availabilityZoneIds[i];
            let zonalMetricProps = {
                availabilityMetricProps: props.service.operations
                    .filter((x) => x.critical)
                    .map((x) => {
                    return {
                        availabilityZoneId: availabilityZoneId,
                        label: x.operationName,
                        metricDetails: x.serverSideAvailabilityMetricDetails,
                        metricType: AvailabilityMetricType_1.AvailabilityMetricType.REQUEST_COUNT,
                    };
                }),
                period: props.service.period,
                label: availabilityZoneId + 'tps',
            };
            widgets.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 8,
                title: availabilityZoneId + ' TPS',
                region: aws_cdk_lib_1.Fn.ref('AWS::Region'),
                left: ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalServiceAvailabilityMetrics(zonalMetricProps),
                statistic: 'Sum',
                leftYAxis: {
                    label: 'TPS',
                    showUnits: false,
                },
            }));
        }
        return widgets;
    }
    static generateServerSideAndCanaryAvailabilityWidgets(props, availabilityZoneIds) {
        let widgets = [];
        widgets.push(new aws_cloudwatch_1.TextWidget({
            height: 2,
            width: 24,
            markdown: '**Server-side Availability**\n(Each critical operation is equally weighted regardless of request volume)',
        }));
        widgets = widgets.concat(ServiceAvailabilityAndLatencyDashboard.generateAvailabilityWidgets(props, false, availabilityZoneIds));
        if (props.service.operations.filter((x) => x.critical && x.canaryMetricDetails !== undefined).length > 0) {
            widgets.push(new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: '**Canary Measured Availability**\n(Each operation is equally weighted regardless of request volume)',
            }));
            widgets = widgets.concat(ServiceAvailabilityAndLatencyDashboard.generateAvailabilityWidgets(props, true, availabilityZoneIds));
        }
        return widgets;
    }
    static generateServerSideAndCanaryLatencyWidgets(props, availabilityZoneIds) {
        let widgets = [];
        widgets.push(new aws_cloudwatch_1.TextWidget({
            height: 2,
            width: 24,
            markdown: '**Server-side Latency**\n(Counts of requests exceeding the per-operation latency threshold)',
        }));
        widgets = widgets.concat(ServiceAvailabilityAndLatencyDashboard.generateLatencyMetricWidgets(props, false, availabilityZoneIds));
        if (props.service.operations.filter((x) => x.critical && x.canaryMetricDetails !== undefined).length > 0) {
            widgets.push(new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: '**Canary Measured Latency**\n(Counts of requests exceeding the per-operation latency threshold)',
            }));
            widgets = widgets.concat(ServiceAvailabilityAndLatencyDashboard.generateLatencyMetricWidgets(props, true, availabilityZoneIds));
        }
        return widgets;
    }
    static generateAvailabilityWidgets(props, isCanary, availabilityZoneIds) {
        let widgets = [];
        widgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 6,
            width: 24,
            title: aws_cdk_lib_1.Fn.ref('AWS::Region') + ' Availability',
            region: aws_cdk_lib_1.Fn.ref('AWS::Region'),
            left: RegionalAvailabilityMetrics_1.RegionalAvailabilityMetrics.createRegionalServiceAvailabilityMetrics({
                label: aws_cdk_lib_1.Fn.ref('AWS::Region') + ' availability',
                period: props.service.period,
                availabilityMetricProps: this.createRegionalAvailabilityMetricProps(props.service.operations.filter((x) => x.critical), isCanary, AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE),
            }),
            statistic: 'Sum',
            leftYAxis: {
                max: 100,
                min: 95,
                label: 'Availability',
                showUnits: false,
            },
            right: RegionalAvailabilityMetrics_1.RegionalAvailabilityMetrics.createRegionalServiceAvailabilityMetrics({
                label: aws_cdk_lib_1.Fn.ref('AWS::Region') + ' faults',
                period: props.service.period,
                availabilityMetricProps: this.createRegionalAvailabilityMetricProps(props.service.operations.filter((x) => x.critical), isCanary, AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT),
            }),
            rightYAxis: {
                label: 'Faults',
                showUnits: false,
                min: 0,
                max: Math.ceil(props.service.faultCountThreshold * 1.5),
            },
            rightAnnotations: [
                {
                    color: aws_cloudwatch_1.Color.RED,
                    label: 'High severity',
                    value: props.service.faultCountThreshold,
                },
            ],
        }));
        for (let i = 0; i < availabilityZoneIds.length; i++) {
            let availabilityZoneId = availabilityZoneIds[i];
            widgets.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 8,
                title: availabilityZoneId + ' Availability',
                region: aws_cdk_lib_1.Fn.ref('AWS::Region'),
                left: ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalServiceAvailabilityMetrics({
                    label: availabilityZoneId + ' availability',
                    period: props.service.period,
                    availabilityMetricProps: this.createZonalAvailabilityMetricProps(props.service.operations.filter((x) => x.critical), availabilityZoneId, isCanary, AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE),
                }),
                statistic: 'Sum',
                leftYAxis: {
                    max: 100,
                    min: 95,
                    label: 'Availability',
                    showUnits: false,
                },
                right: ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalServiceAvailabilityMetrics({
                    label: availabilityZoneId + ' faults',
                    period: props.service.period,
                    availabilityMetricProps: this.createZonalAvailabilityMetricProps(props.service.operations.filter((x) => x.critical), availabilityZoneId, isCanary, AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT),
                }),
                rightYAxis: {
                    label: 'Faults',
                    showUnits: false,
                    min: 0,
                    max: Math.ceil(props.service.faultCountThreshold * 1.5),
                },
                rightAnnotations: [
                    {
                        color: aws_cloudwatch_1.Color.RED,
                        label: 'High severity',
                        value: props.service.faultCountThreshold,
                    },
                ],
            }));
        }
        return widgets;
    }
    static generateLatencyMetricWidgets(props, isCanary, availabilityZoneIds) {
        let widgets = [];
        widgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 6,
            width: 24,
            title: aws_cdk_lib_1.Fn.ref('AWS::Region') + ' High Latency Count',
            region: aws_cdk_lib_1.Fn.ref('AWS::Region'),
            left: RegionalLatencyMetrics_1.RegionalLatencyMetrics.createRegionalServiceLatencyCountMetrics({
                label: aws_cdk_lib_1.Fn.ref('AWS::Region'),
                period: props.service.period,
                latencyMetricProps: this.createRegionalLatencyMetricProps(props.service.operations.filter((x) => x.critical), isCanary, LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY),
            }),
            statistic: 'Sum',
            leftYAxis: {
                max: props.service.faultCountThreshold * 1.5,
                min: 0,
                label: 'Sum',
                showUnits: false,
            },
            leftAnnotations: [
                {
                    color: aws_cloudwatch_1.Color.RED,
                    label: 'High severity',
                    value: props.service.faultCountThreshold,
                },
            ],
        }));
        for (let i = 0; i < availabilityZoneIds.length; i++) {
            let availabilityZoneId = availabilityZoneIds[i];
            widgets.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 8,
                title: availabilityZoneId + ' High Latency Count',
                region: aws_cdk_lib_1.Fn.ref('AWS::Region'),
                left: ZonalLatencyMetrics_1.ZonalLatencyMetrics.createZonalServiceLatencyMetrics({
                    label: availabilityZoneId,
                    period: props.service.period,
                    latencyMetricProps: this.createZonalLatencyMetricProps(props.service.operations.filter((x) => x.critical), availabilityZoneId, isCanary, LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY),
                }),
                statistic: 'Sum',
                leftYAxis: {
                    max: props.service.faultCountThreshold * 1.5,
                    min: 0,
                    label: 'Sum',
                    showUnits: false,
                },
                leftAnnotations: [
                    {
                        color: aws_cloudwatch_1.Color.RED,
                        label: 'High severity',
                        value: props.service.faultCountThreshold,
                    },
                ],
            }));
        }
        return widgets;
    }
    static createRegionalAvailabilityMetricProps(criticalOperations, isCanary, metricType) {
        return criticalOperations
            .reduce((filtered, value) => {
            if (isCanary && value.canaryMetricDetails) {
                filtered.push(value.canaryMetricDetails.canaryAvailabilityMetricDetails);
            }
            else if (!isCanary) {
                filtered.push(value.serverSideAvailabilityMetricDetails);
            }
            return filtered;
        }, [])
            .map((x) => {
            return {
                label: x.operationName + ' ' + metricType.toString().replace('_', ' '),
                metricDetails: x,
                metricType: metricType,
            };
        });
    }
    static createRegionalLatencyMetricProps(criticalOperations, isCanary, metricType) {
        return criticalOperations
            .reduce((filtered, value) => {
            if (isCanary && value.canaryMetricDetails) {
                filtered.push(value.canaryMetricDetails.canaryLatencyMetricDetails);
            }
            else if (!isCanary) {
                filtered.push(value.serverSideLatencyMetricDetails);
            }
            return filtered;
        }, [])
            .map((x) => {
            return {
                label: x.operationName,
                metricDetails: x,
                metricType: metricType,
                statistic: 'TC(' + x.successAlarmThreshold + ':)',
            };
        });
    }
    static createZonalAvailabilityMetricProps(criticalOperations, availabilityZoneId, isCanary, metricType) {
        return criticalOperations
            .reduce((filtered, value) => {
            if (isCanary &&
                value.canaryMetricDetails !== undefined &&
                value.canaryMetricDetails != null) {
                filtered.push(value.canaryMetricDetails.canaryAvailabilityMetricDetails);
            }
            else if (!isCanary) {
                filtered.push(value.serverSideAvailabilityMetricDetails);
            }
            return filtered;
        }, [])
            .map((x) => {
            return {
                label: x.operationName + ' ' + metricType.toString().replace('_', ' '),
                metricDetails: x,
                metricType: metricType,
                availabilityZoneId: availabilityZoneId,
            };
        });
    }
    static createZonalLatencyMetricProps(criticalOperations, availabilityZoneId, isCanary, metricType) {
        return criticalOperations
            .reduce((filtered, value) => {
            if (isCanary &&
                value.canaryMetricDetails !== undefined &&
                value.canaryMetricDetails != null) {
                filtered.push(value.canaryMetricDetails.canaryLatencyMetricDetails);
            }
            else if (!isCanary) {
                filtered.push(value.serverSideLatencyMetricDetails);
            }
            return filtered;
        }, [])
            .map((x) => {
            return {
                label: x.operationName,
                metricDetails: x,
                metricType: metricType,
                availabilityZoneId: availabilityZoneId,
                statistic: 'TC(' + x.successAlarmThreshold + ':)',
            };
        });
    }
    static generateLoadBalancerWidgets(props, title, availabilityZoneNames) {
        let albWidgets = [];
        albWidgets.push(new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: title }));
        let keyprefix = MetricsHelper_1.MetricsHelper.nextChar();
        let faultRateMetrics = [];
        let requestCountMetrics = [];
        let processedBytesMetrics = [];
        let latencyMetrics = [];
        availabilityZoneNames.forEach((availabilityZoneName) => {
            let availabilityZoneId = props.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(availabilityZoneName.substring(availabilityZoneName.length - 1));
            keyprefix = MetricsHelper_1.MetricsHelper.nextChar(keyprefix);
            let faultMetric = ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.getPerAZAvailabilityMetric(props.service.loadBalancer, {
                availabilityZone: availabilityZoneName,
                availabilityZoneId: availabilityZoneId,
                metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_RATE,
                label: availabilityZoneId,
                period: props.service.period,
                keyprefix: keyprefix
            });
            faultRateMetrics.push(faultMetric);
            let requestCountMetric = ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.getPerAZAvailabilityMetric(props.service.loadBalancer, {
                availabilityZone: availabilityZoneName,
                availabilityZoneId: availabilityZoneId,
                metricType: AvailabilityMetricType_1.AvailabilityMetricType.REQUEST_COUNT,
                label: availabilityZoneId,
                period: props.service.period,
                keyprefix: keyprefix
            });
            requestCountMetrics.push(requestCountMetric);
            let processedBytesMetric = ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.getPerAZProcessedBytesMetric(props.service.loadBalancer, availabilityZoneName, availabilityZoneId, props.service.period);
            processedBytesMetrics.push(processedBytesMetric);
            let latencyMetric = ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.getPerAZLatencyMetric({
                availabilityZone: availabilityZoneName,
                availabilityZoneId: availabilityZoneId,
                alb: props.service.loadBalancer,
                statistic: "p99",
                label: availabilityZoneId,
                period: props.service.period
            });
            latencyMetrics.push(latencyMetric);
        });
        albWidgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 8,
            width: 8,
            title: aws_cdk_lib_1.Fn.sub('${AWS::Region} Fault Rate'),
            region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
            left: faultRateMetrics,
            leftYAxis: {
                max: 35,
                min: 0,
                label: 'Fault Rate',
                showUnits: false,
            }
        }));
        albWidgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 8,
            width: 8,
            title: aws_cdk_lib_1.Fn.sub('${AWS::Region} Request Count'),
            region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
            left: requestCountMetrics,
            leftYAxis: {
                min: 0,
                label: 'Sum',
                showUnits: false,
            }
        }));
        albWidgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 8,
            width: 8,
            title: aws_cdk_lib_1.Fn.sub('${AWS::Region} Processed Bytes'),
            region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
            left: processedBytesMetrics,
            leftYAxis: {
                min: 0,
                showUnits: true,
            }
        }));
        albWidgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 8,
            width: 8,
            title: aws_cdk_lib_1.Fn.sub('${AWS::Region} Latency'),
            region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
            left: latencyMetrics,
            leftYAxis: {
                min: 0,
                showUnits: true,
            }
        }));
        return albWidgets;
    }
    constructor(scope, id, props) {
        super(scope, id);
        let topLevelAggregateAlarmWidgets = [];
        let availabilityZoneIds = props.service.availabilityZoneNames.map((x) => {
            return props.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(x.substring(x.length - 1));
        });
        topLevelAggregateAlarmWidgets.push(new aws_cloudwatch_1.TextWidget({
            height: 2,
            width: 24,
            markdown: '***Availability and Latency Alarms***',
        }));
        topLevelAggregateAlarmWidgets.push(new aws_cloudwatch_1.AlarmStatusWidget({
            height: 2,
            width: 24,
            alarms: [props.aggregateRegionalAlarm],
            title: 'Customer Experience - Regional Aggregate Impact Alarm (measures fault count in aggregate across all critical operations)',
        }));
        let keyPrefix = MetricsHelper_1.MetricsHelper.nextChar();
        let perOperationAZFaultsMetrics = [];
        for (let i = 0; i < availabilityZoneIds.length; i++) {
            let counter = 1;
            let availabilityZoneId = availabilityZoneIds[i];
            topLevelAggregateAlarmWidgets.push(new aws_cloudwatch_1.AlarmStatusWidget({
                height: 2,
                width: 8,
                alarms: [props.zonalAggregateAlarms[i]],
                title: availabilityZoneId +
                    ' Zonal Isolated Impact Alarm (any critical operation in this AZ shows impact from server-side or canary)',
            }));
            let usingMetrics = {};
            props.service.operations
                .filter((x) => x.critical == true)
                .forEach((x) => {
                usingMetrics[`${keyPrefix}${counter++}`] =
                    ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalAvailabilityMetric({
                        availabilityZoneId: availabilityZoneId,
                        metricDetails: x.serverSideAvailabilityMetricDetails,
                        label: availabilityZoneId + ' ' + x.operationName + ' fault count',
                        metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
                        keyPrefix: keyPrefix,
                    });
            });
            let zonalFaultCount = new aws_cloudwatch_1.MathExpression({
                expression: Object.keys(usingMetrics).join('+'),
                label: availabilityZoneId,
                usingMetrics: usingMetrics,
                period: props.service.period
            });
            perOperationAZFaultsMetrics.push(zonalFaultCount);
            keyPrefix = MetricsHelper_1.MetricsHelper.nextChar(keyPrefix);
        }
        let azContributorWidgets = [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: '**AZ Contributors To Faults**',
            }),
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 24,
                title: 'AZ Fault Count',
                period: props.service.period,
                left: perOperationAZFaultsMetrics,
            }),
        ];
        topLevelAggregateAlarmWidgets.concat(ServiceAvailabilityAndLatencyDashboard.generateTPSWidgets(props, availabilityZoneIds));
        this.dashboard = new aws_cloudwatch_1.Dashboard(this, 'TopLevelDashboard', {
            dashboardName: props.service.serviceName.toLowerCase() +
                aws_cdk_lib_1.Fn.sub('-availability-and-latency-${AWS::Region}'),
            defaultInterval: props.interval,
            periodOverride: aws_cloudwatch_1.PeriodOverride.AUTO,
            widgets: [
                topLevelAggregateAlarmWidgets,
                azContributorWidgets,
                ServiceAvailabilityAndLatencyDashboard.generateServerSideAndCanaryAvailabilityWidgets(props, availabilityZoneIds),
                ServiceAvailabilityAndLatencyDashboard.generateServerSideAndCanaryLatencyWidgets(props, availabilityZoneIds),
            ],
        });
        let lb = props.service.loadBalancer?.node
            .defaultChild;
        if (lb !== undefined && lb != null && lb.type == 'application') {
            this.dashboard.addWidgets(...ServiceAvailabilityAndLatencyDashboard.generateLoadBalancerWidgets(props, '**Application Load Balancer Metrics**', props.service.availabilityZoneNames));
        }
    }
}
exports.ServiceAvailabilityAndLatencyDashboard = ServiceAvailabilityAndLatencyDashboard;
//# sourceMappingURL=data:application/json;base64,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