"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OperationAvailabilityAndLatencyDashboard = void 0;
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
const ContributorInsightsWidget_1 = require("./ContributorInsightsWidget");
const ApplicationLoadBalancerMetrics_1 = require("../metrics/ApplicationLoadBalancerMetrics");
const NetworkLoadBalancerMetrics_1 = require("../metrics/NetworkLoadBalancerMetrics");
const RegionalAvailabilityMetrics_1 = require("../metrics/RegionalAvailabilityMetrics");
const RegionalLatencyMetrics_1 = require("../metrics/RegionalLatencyMetrics");
const ZonalAvailabilityMetrics_1 = require("../metrics/ZonalAvailabilityMetrics");
const ZonalLatencyMetrics_1 = require("../metrics/ZonalLatencyMetrics");
const AvailabilityMetricType_1 = require("../utilities/AvailabilityMetricType");
const LatencyMetricType_1 = require("../utilities/LatencyMetricType");
const MetricsHelper_1 = require("../utilities/MetricsHelper");
/**
 * Creates an operation level availability and latency dashboard
 */
class OperationAvailabilityAndLatencyDashboard extends constructs_1.Construct {
    static createTopLevelAggregateAlarmWidgets(props, title, availabilityZoneIds) {
        let topLevelAggregateAlarms = [
            new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: title }),
            new aws_cloudwatch_1.AlarmStatusWidget({
                height: 2,
                width: 24,
                alarms: [props.regionalImpactAlarm],
                title: props.operation.operationName + ' Regional Impact',
            }),
        ];
        for (let i = 0; i < availabilityZoneIds.length; i++) {
            let availabilityZoneId = availabilityZoneIds[i];
            topLevelAggregateAlarms.push(new aws_cloudwatch_1.AlarmStatusWidget({
                height: 2,
                width: 8,
                alarms: [props.isolatedAZImpactAlarms[i]],
                title: availabilityZoneId + ' Isolated Impact',
            }));
        }
        topLevelAggregateAlarms.push(new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: '**AZ Contributors**' }));
        let zonalServerSideHighLatencyMetrics = [];
        let zonalServerSideFaultCountMetrics = [];
        let zonalCanaryHighLatencyMetrics = [];
        let zonalCanaryFaultCountMetrics = [];
        let keyPrefix = MetricsHelper_1.MetricsHelper.nextChar();
        for (let i = 0; i < availabilityZoneIds.length; i++) {
            let availabilityZoneId = availabilityZoneIds[i];
            zonalServerSideHighLatencyMetrics.push(ZonalLatencyMetrics_1.ZonalLatencyMetrics.createZonalCountLatencyMetric({
                availabilityZoneId: availabilityZoneId,
                metricDetails: props.operation.serverSideLatencyMetricDetails,
                label: availabilityZoneId + ' high latency responses',
                metricType: LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY,
                statistic: `TC(${props.operation.serverSideLatencyMetricDetails.successAlarmThreshold}:)`,
                keyPrefix: keyPrefix,
            }));
            zonalServerSideFaultCountMetrics.push(ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalAvailabilityMetric({
                availabilityZoneId: availabilityZoneId,
                metricDetails: props.operation.serverSideAvailabilityMetricDetails,
                label: availabilityZoneId + ' fault count',
                metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
                keyPrefix: keyPrefix,
            }));
            if (props.operation.canaryMetricDetails !== undefined &&
                props.operation.canaryMetricDetails != null) {
                zonalCanaryHighLatencyMetrics.push(ZonalLatencyMetrics_1.ZonalLatencyMetrics.createZonalCountLatencyMetric({
                    availabilityZoneId: availabilityZoneId,
                    metricDetails: props.operation.canaryMetricDetails.canaryLatencyMetricDetails,
                    label: availabilityZoneId + ' high latency responses',
                    metricType: LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY,
                    statistic: `TC(${props.operation.canaryMetricDetails.canaryLatencyMetricDetails.successAlarmThreshold}:)`,
                    keyPrefix: keyPrefix,
                }));
                zonalCanaryFaultCountMetrics.push(ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalAvailabilityMetric({
                    availabilityZoneId: availabilityZoneId,
                    metricDetails: props.operation.canaryMetricDetails
                        .canaryAvailabilityMetricDetails,
                    label: availabilityZoneId + ' fault count',
                    metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
                    keyPrefix: keyPrefix,
                }));
            }
            keyPrefix = MetricsHelper_1.MetricsHelper.nextChar(keyPrefix);
        }
        topLevelAggregateAlarms.push(new aws_cloudwatch_1.GraphWidget({
            height: 6,
            width: 24,
            title: 'Server-side AZ Fault Contributors',
            left: zonalServerSideFaultCountMetrics,
        }));
        if (zonalCanaryFaultCountMetrics.length > 0) {
            topLevelAggregateAlarms.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 24,
                title: 'Canary AZ Fault Contributors',
                left: zonalCanaryFaultCountMetrics,
            }));
        }
        topLevelAggregateAlarms.push(new aws_cloudwatch_1.GraphWidget({
            height: 6,
            width: 24,
            title: 'Server-side High Latency Contributors',
            left: zonalServerSideHighLatencyMetrics,
        }));
        if (zonalCanaryHighLatencyMetrics.length > 0) {
            topLevelAggregateAlarms.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 24,
                title: 'Canary High Latency Contributors',
                left: zonalCanaryHighLatencyMetrics,
            }));
        }
        topLevelAggregateAlarms.push(new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: '**TPS Metrics**' }));
        topLevelAggregateAlarms.push(new aws_cloudwatch_1.GraphWidget({
            height: 6,
            width: 24,
            title: aws_cdk_lib_1.Fn.sub('${AWS::Region} TPS'),
            region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
            left: [
                RegionalAvailabilityMetrics_1.RegionalAvailabilityMetrics.createRegionalAvailabilityMetric({
                    label: aws_cdk_lib_1.Fn.ref('AWS::Region') + ' tps',
                    metricDetails: props.operation.serverSideAvailabilityMetricDetails,
                    metricType: AvailabilityMetricType_1.AvailabilityMetricType.REQUEST_COUNT,
                }),
            ],
            statistic: 'Sum',
            leftYAxis: {
                label: 'TPS',
                showUnits: false,
            },
        }));
        for (let i = 0; i < availabilityZoneIds.length; i++) {
            let availabilityZoneId = availabilityZoneIds[i];
            topLevelAggregateAlarms.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 8,
                title: availabilityZoneId + ' TPS',
                region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
                left: [
                    ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalAvailabilityMetric({
                        availabilityZoneId: availabilityZoneId,
                        label: availabilityZoneId + ' tps',
                        metricDetails: props.operation.serverSideAvailabilityMetricDetails,
                        metricType: AvailabilityMetricType_1.AvailabilityMetricType.REQUEST_COUNT,
                    }),
                ],
                statistic: 'Sum',
                leftYAxis: {
                    label: 'TPS',
                    showUnits: false,
                },
            }));
        }
        return topLevelAggregateAlarms;
    }
    static createAvailabilityWidgets(props, title) {
        let availabilityWidgets = [];
        availabilityWidgets.push(new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: title }));
        let rowTracker = 0;
        let keyPrefix1 = MetricsHelper_1.MetricsHelper.nextChar();
        let keyPrefix2 = MetricsHelper_1.MetricsHelper.nextChar(keyPrefix1);
        // Create regional availability and fault metrics and availability alarm widgets
        availabilityWidgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 8,
            width: 24,
            title: aws_cdk_lib_1.Fn.sub('${AWS::Region} Availability'),
            region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
            left: [
                RegionalAvailabilityMetrics_1.RegionalAvailabilityMetrics.createRegionalAvailabilityMetric({
                    label: aws_cdk_lib_1.Fn.ref('AWS::Region') + ' availability',
                    metricDetails: props.availabilityMetricDetails,
                    metricType: AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE,
                    keyPrefix: keyPrefix1,
                }),
            ],
            statistic: 'Sum',
            leftYAxis: {
                max: 100,
                min: 95,
                label: 'Availability',
                showUnits: false,
            },
            leftAnnotations: [
                {
                    value: props.availabilityMetricDetails.successAlarmThreshold,
                    visible: true,
                    color: aws_cloudwatch_1.Color.RED,
                    label: 'High Severity',
                },
            ],
            right: [
                RegionalAvailabilityMetrics_1.RegionalAvailabilityMetrics.createRegionalAvailabilityMetric({
                    label: aws_cdk_lib_1.Fn.ref('AWS::Region') + ' fault count',
                    metricDetails: props.availabilityMetricDetails,
                    metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
                    keyPrefix: keyPrefix2,
                }),
            ],
            rightYAxis: {
                label: 'Fault Count',
                showUnits: false,
            },
        }));
        availabilityWidgets.push(new aws_cloudwatch_1.AlarmWidget({
            height: 2,
            width: 24,
            region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
            alarm: props.regionalEndpointAvailabilityAlarm,
            title: 'Success Rate',
        }));
        for (let i = 0; i < props.availabilityZoneIds.length; i++) {
            let availabilityZoneId = props.availabilityZoneIds[i];
            let keyPrefix3 = MetricsHelper_1.MetricsHelper.nextChar();
            let keyPrefix4 = MetricsHelper_1.MetricsHelper.nextChar(keyPrefix3);
            availabilityWidgets.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 8,
                title: availabilityZoneId + ' Availability',
                region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
                left: [
                    ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalAvailabilityMetric({
                        availabilityZoneId: availabilityZoneId,
                        label: availabilityZoneId + ' availability',
                        metricDetails: props.availabilityMetricDetails,
                        metricType: AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE,
                        keyPrefix: keyPrefix3,
                    }),
                ],
                statistic: 'Sum',
                leftYAxis: {
                    max: 100,
                    min: 95,
                    label: 'Availability',
                    showUnits: false,
                },
                leftAnnotations: [
                    {
                        value: props.availabilityMetricDetails.successAlarmThreshold,
                        visible: true,
                        color: aws_cloudwatch_1.Color.RED,
                        label: 'High Severity',
                    },
                ],
                right: [
                    ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalAvailabilityMetric({
                        availabilityZoneId: availabilityZoneId,
                        label: availabilityZoneId + ' fault count',
                        metricDetails: props.availabilityMetricDetails,
                        metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
                        keyPrefix: keyPrefix4,
                    }),
                ],
                rightYAxis: {
                    label: 'Fault Count',
                    showUnits: false,
                },
            }));
            //We're on the third one for this set, add 3 alarms
            //or if we're at the end, at the necessary amount
            //of alarms, 1, 2, or 3
            if (i % 3 == 2 || i - 1 == props.availabilityZoneIds.length) {
                for (let k = rowTracker; k <= i; k++) {
                    availabilityWidgets.push(new aws_cloudwatch_1.AlarmWidget({
                        height: 2,
                        width: 8,
                        region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
                        alarm: props.zonalEndpointAvailabilityAlarms[k],
                        title: 'Success Rate',
                    }));
                }
                rowTracker += i + 1;
            }
        }
        if (!props.isCanary &&
            props.instanceContributorsToFaults !== undefined &&
            props.instanceContributorsToFaults != null) {
            availabilityWidgets.push(new ContributorInsightsWidget_1.ContributorInsightsWidget({
                height: 6,
                width: 24,
                title: 'Individual Instance Contributors to Fault Count',
                insightRule: props.instanceContributorsToFaults,
                period: props.availabilityMetricDetails.period,
                legendPosition: aws_cloudwatch_1.LegendPosition.BOTTOM,
                orderStatistic: 'Sum',
                accountId: aws_cdk_lib_1.Fn.ref('AWS::AccountId'),
                topContributors: 10,
            }));
        }
        return availabilityWidgets;
    }
    static createLatencyWidgets(props, title) {
        let latencyWidgets = [];
        latencyWidgets.push(new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: title }));
        let rowTracker = 0;
        let keyPrefix = '';
        let latencyMetrics = [];
        let stats = props.latencyMetricDetails.graphedSuccessStatistics !== undefined
            ? props.latencyMetricDetails.graphedSuccessStatistics
            : ['p99'];
        let latencySuccessMetrics = stats.map((x) => {
            keyPrefix = MetricsHelper_1.MetricsHelper.nextChar(keyPrefix);
            return RegionalLatencyMetrics_1.RegionalLatencyMetrics.createRegionalAverageLatencyMetric({
                label: x + ' Success Latency',
                metricDetails: props.latencyMetricDetails,
                metricType: LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY,
                statistic: x,
                keyPrefix: keyPrefix,
            });
        });
        stats =
            props.latencyMetricDetails.graphedFaultStatistics !== undefined
                ? props.latencyMetricDetails.graphedFaultStatistics
                : ['p99'];
        let latencyFaultMetrics = stats.map((x) => {
            keyPrefix = MetricsHelper_1.MetricsHelper.nextChar(keyPrefix);
            return RegionalLatencyMetrics_1.RegionalLatencyMetrics.createRegionalAverageLatencyMetric({
                label: x + ' Fault Latency',
                metricDetails: props.latencyMetricDetails,
                metricType: LatencyMetricType_1.LatencyMetricType.FAULT_LATENCY,
                statistic: x,
                keyPrefix: keyPrefix,
            });
        });
        latencyMetrics = latencySuccessMetrics.concat(latencyFaultMetrics);
        if (latencyMetrics.length > 0) {
            latencyWidgets.push(new aws_cloudwatch_1.GraphWidget({
                height: 8,
                width: 24,
                title: aws_cdk_lib_1.Fn.sub('${AWS::Region} Latency'),
                region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
                left: latencyMetrics,
                leftYAxis: {
                    max: props.latencyMetricDetails.successAlarmThreshold * 1.5,
                    min: 0,
                    label: 'Latency',
                    showUnits: false,
                },
                leftAnnotations: [
                    {
                        value: props.latencyMetricDetails.successAlarmThreshold,
                        visible: true,
                        color: aws_cloudwatch_1.Color.RED,
                        label: 'High Severity',
                    },
                ],
            }));
        }
        latencyWidgets.push(new aws_cloudwatch_1.AlarmWidget({
            height: 2,
            width: 24,
            region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
            alarm: props.regionalEndpointLatencyAlarm,
        }));
        keyPrefix = '';
        for (let i = 0; i < props.availabilityZoneIds.length; i++) {
            let availabilityZoneId = props.availabilityZoneIds[i];
            let latencyMetrics2 = [];
            let stats2 = props.latencyMetricDetails.graphedSuccessStatistics
                ? props.latencyMetricDetails.graphedSuccessStatistics
                : ['p99'];
            let zonalSuccessLatencyMetrics = stats2.map((x) => {
                keyPrefix = MetricsHelper_1.MetricsHelper.nextChar(keyPrefix);
                return ZonalLatencyMetrics_1.ZonalLatencyMetrics.createZonalAverageLatencyMetric({
                    label: x + ' Success Latency',
                    metricDetails: props.latencyMetricDetails,
                    metricType: LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY,
                    statistic: x,
                    availabilityZoneId: availabilityZoneId,
                    keyPrefix: keyPrefix,
                });
            });
            stats2 = props.latencyMetricDetails.graphedFaultStatistics
                ? props.latencyMetricDetails.graphedFaultStatistics
                : ['p99'];
            let zonalFaultLatencyMetrics = stats2.map((x) => {
                keyPrefix = MetricsHelper_1.MetricsHelper.nextChar(keyPrefix);
                return ZonalLatencyMetrics_1.ZonalLatencyMetrics.createZonalAverageLatencyMetric({
                    label: x + ' Fault Latency',
                    metricDetails: props.latencyMetricDetails,
                    metricType: LatencyMetricType_1.LatencyMetricType.FAULT_LATENCY,
                    statistic: x,
                    availabilityZoneId: availabilityZoneId,
                    keyPrefix: keyPrefix,
                });
            });
            latencyMetrics2 = zonalSuccessLatencyMetrics.concat(zonalFaultLatencyMetrics);
            if (latencyMetrics2.length > 0) {
                latencyWidgets.push(new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 8,
                    title: availabilityZoneId + ' Latency',
                    region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
                    left: latencyMetrics2,
                    leftAnnotations: [
                        {
                            value: props.latencyMetricDetails.successAlarmThreshold,
                            visible: true,
                            color: aws_cloudwatch_1.Color.RED,
                            label: 'High Severity',
                        },
                    ],
                    leftYAxis: {
                        max: props.latencyMetricDetails.successAlarmThreshold * 1.5,
                        min: 0,
                        label: 'Latency',
                        showUnits: false,
                    },
                }));
            }
            //We're on the third one for this set, add 3 alarms
            //or if we're at the end, at the necessary amount
            //of alarms, 1, 2, or 3
            if (i % 3 == 2 || i - 1 == props.availabilityZoneIds.length) {
                for (let k = rowTracker; k <= i; k++) {
                    latencyWidgets.push(new aws_cloudwatch_1.AlarmWidget({
                        height: 2,
                        width: 8,
                        region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
                        alarm: props.zonalEndpointLatencyAlarms[k],
                    }));
                }
                rowTracker += i + 1;
            }
        }
        if (!props.isCanary &&
            props.instanceContributorsToHighLatency !== undefined &&
            props.instanceContributorsToHighLatency != null) {
            latencyWidgets.push(new ContributorInsightsWidget_1.ContributorInsightsWidget({
                height: 6,
                width: 24,
                title: 'Individual Instance Contributors to High Latency',
                insightRule: props.instanceContributorsToHighLatency,
                period: props.latencyMetricDetails.period,
                legendPosition: aws_cloudwatch_1.LegendPosition.BOTTOM,
                orderStatistic: 'Sum',
                accountId: aws_cdk_lib_1.Fn.ref('AWS::AccountId'),
                topContributors: 10,
            }));
        }
        return latencyWidgets;
    }
    static createApplicationLoadBalancerWidgets(props, title, availabilityZoneNames) {
        let albWidgets = [];
        albWidgets.push(new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: title }));
        albWidgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 8,
            width: 24,
            title: aws_cdk_lib_1.Fn.sub('${AWS::Region} Processed Bytes'),
            region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
            left: [
                ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.getRegionalProcessedBytesMetric(props.loadBalancer, props.operation.serverSideAvailabilityMetricDetails.period),
            ],
            leftYAxis: {
                label: 'Processed Bytes',
                showUnits: true,
            },
        }));
        availabilityZoneNames.forEach((availabilityZoneName) => {
            let availabilityZoneId = props.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(availabilityZoneName.substring(availabilityZoneName.length - 1));
            albWidgets.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 8,
                title: availabilityZoneId + ' Processed Bytes',
                region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
                left: [
                    ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.getPerAZProcessedBytesMetric(props.loadBalancer, availabilityZoneName, availabilityZoneId, props.operation.serverSideAvailabilityMetricDetails.period),
                ],
                leftYAxis: {
                    label: 'Processed Bytes',
                    showUnits: true,
                },
            }));
        });
        return albWidgets;
    }
    static createNetworkLoadBalancerWidgets(props, title, availabilityZoneNames) {
        let nlbWidgets = [];
        let loadBalancerFullName = props.loadBalancer
            .loadBalancerFullName;
        nlbWidgets.push(new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: title }));
        nlbWidgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 8,
            width: 24,
            title: aws_cdk_lib_1.Fn.sub('${AWS::Region} Processed Bytes'),
            region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
            left: [
                NetworkLoadBalancerMetrics_1.NetworkLoadBalancerMetrics.createRegionalNetworkLoadBalancerProcessedBytesMetric(loadBalancerFullName, props.operation.serverSideAvailabilityMetricDetails.period),
            ],
            leftYAxis: {
                label: 'Processed Bytes',
                showUnits: true,
            },
        }));
        availabilityZoneNames.forEach((availabilityZoneName) => {
            let availabilityZoneId = props.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(availabilityZoneName.substring(availabilityZoneName.length - 1));
            nlbWidgets.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 8,
                title: availabilityZoneId + ' Processed Bytes',
                region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
                left: [
                    NetworkLoadBalancerMetrics_1.NetworkLoadBalancerMetrics.createZonalNetworkLoadBalancerProcessedBytesMetric(loadBalancerFullName, availabilityZoneName, props.operation.serverSideAvailabilityMetricDetails.period),
                ],
                leftYAxis: {
                    label: 'Processed Bytes',
                    showUnits: true,
                },
            }));
        });
        return nlbWidgets;
    }
    constructor(scope, id, props) {
        super(scope, id);
        let widgets = [];
        this.azMapper = props.azMapper;
        let availabilityZoneIds = props.operation.service.availabilityZoneNames.map((x) => {
            return this.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(x.substring(x.length - 1));
        });
        widgets.push(OperationAvailabilityAndLatencyDashboard.createTopLevelAggregateAlarmWidgets(props, '**Top Level Aggregate Alarms**', availabilityZoneIds));
        widgets.push(OperationAvailabilityAndLatencyDashboard.createAvailabilityWidgets({
            operation: props.operation,
            availabilityMetricDetails: props.operation.serverSideAvailabilityMetricDetails,
            availabilityZoneIds: availabilityZoneIds,
            isCanary: false,
            zonalEndpointAvailabilityAlarms: props.zonalEndpointServerAvailabilityAlarms,
            regionalEndpointAvailabilityAlarm: props.regionalEndpointServerAvailabilityAlarm,
            instanceContributorsToFaults: props.instanceContributorsToFaults,
        }, '**Server-side Availability**'));
        widgets.push(OperationAvailabilityAndLatencyDashboard.createLatencyWidgets({
            operation: props.operation,
            latencyMetricDetails: props.operation.serverSideLatencyMetricDetails,
            availabilityZoneIds: availabilityZoneIds,
            isCanary: false,
            zonalEndpointLatencyAlarms: props.zonalEndpointServerLatencyAlarms,
            regionalEndpointLatencyAlarm: props.regionalEndpointServerLatencyAlarm,
            instanceContributorsToHighLatency: props.instanceContributorsToHighLatency,
        }, '**Server-side Latency**'));
        let lb = props.loadBalancer?.node
            .defaultChild;
        if (lb !== undefined && lb != null) {
            if (lb.type == 'application') {
                widgets.push(OperationAvailabilityAndLatencyDashboard.createApplicationLoadBalancerWidgets(props, '**Application Load Balancer Metrics**', props.operation.service.availabilityZoneNames));
            }
            else if (lb.type == 'network') {
                widgets.push(OperationAvailabilityAndLatencyDashboard.createNetworkLoadBalancerWidgets(props, '**Network Load Balancer Metrics**', props.operation.service.availabilityZoneNames));
            }
        }
        if (props.operation.canaryMetricDetails !== undefined &&
            props.operation.canaryMetricDetails != null) {
            if (props.zonalEndpointCanaryAvailabilityAlarms &&
                props.zonalEndpointCanaryLatencyAlarms &&
                props.regionalEndpointCanaryAvailabilityAlarm &&
                props.regionalEndpointCanaryLatencyAlarm) {
                widgets.push(OperationAvailabilityAndLatencyDashboard.createAvailabilityWidgets({
                    operation: props.operation,
                    availabilityMetricDetails: props.operation.canaryMetricDetails
                        .canaryAvailabilityMetricDetails,
                    availabilityZoneIds: availabilityZoneIds,
                    isCanary: true,
                    zonalEndpointAvailabilityAlarms: props.zonalEndpointCanaryAvailabilityAlarms,
                    regionalEndpointAvailabilityAlarm: props.regionalEndpointCanaryAvailabilityAlarm,
                }, '**Canary Measured Availability**'));
                widgets.push(OperationAvailabilityAndLatencyDashboard.createLatencyWidgets({
                    operation: props.operation,
                    latencyMetricDetails: props.operation.canaryMetricDetails.canaryLatencyMetricDetails,
                    availabilityZoneIds: availabilityZoneIds,
                    isCanary: true,
                    zonalEndpointLatencyAlarms: props.zonalEndpointCanaryLatencyAlarms,
                    regionalEndpointLatencyAlarm: props.regionalEndpointCanaryLatencyAlarm,
                }, '**Canary Measured Latency**'));
            }
        }
        this.dashboard = new aws_cloudwatch_1.Dashboard(this, 'Dashboard', {
            dashboardName: props.operation.service.serviceName.toLowerCase() +
                '-' +
                props.operation.operationName.toLowerCase() +
                aws_cdk_lib_1.Fn.sub('-availability-and-latency-${AWS::Region}'),
            defaultInterval: props.interval,
            periodOverride: aws_cloudwatch_1.PeriodOverride.AUTO,
            widgets: widgets,
        });
    }
}
exports.OperationAvailabilityAndLatencyDashboard = OperationAvailabilityAndLatencyDashboard;
//# sourceMappingURL=data:application/json;base64,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