"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CanaryFunction = void 0;
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
const MetricsHelper_1 = require("../utilities/MetricsHelper");
class CanaryFunction extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        let xrayManagedPolicy = new aws_iam_1.ManagedPolicy(this, 'xrayManagedPolicy', {
            path: '/canary/',
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: [
                        'xray:PutTraceSegments',
                        'xray:PutTelemetryRecords',
                        'xray:GetSamplingRules',
                        'xray:GetSamplingTargets',
                        'xray:GetSamplingStatisticSummaries',
                    ],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: ['*'],
                }),
            ],
        });
        let ec2ManagedPolicy = new aws_iam_1.ManagedPolicy(this, 'ec2ManagedPolicy', {
            path: '/canary/',
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: [
                        'ec2:CreateNetworkInterface',
                        'ec2:DescribeNetworkInterfaces',
                        'ec2:DeleteNetworkInterface',
                    ],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: ['*'],
                }),
            ],
        });
        let executionRole = new aws_iam_1.Role(this, 'executionRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            path: '/canary/',
            managedPolicies: [xrayManagedPolicy, ec2ManagedPolicy],
        });
        /*
        let code: AssetCode = Code.fromAsset(path.join(__dirname, "src/"), {
          bundling: {
            //image: new Runtime('python3.13:latest-arm64', RuntimeFamily.PYTHON).bundlingImage,
            image: MetricsHelper.PythonRuntime.bundlingImage,
            command: [
              'bash', '-c',
              'pip install --no-cache -r requirements.txt -t /asset-output && cp --archive --update . /asset-output',
            ],
            platform: 'linux/arm64',
          },
        });
        */
        let monitoringLayer = new aws_lambda_1.LayerVersion(this, 'MonitoringLayer', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '../monitoring/src/monitoring-layer.zip')),
            compatibleArchitectures: [aws_lambda_1.Architecture.ARM_64],
            compatibleRuntimes: [MetricsHelper_1.MetricsHelper.PythonRuntime],
        });
        if (props.vpc !== undefined && props.vpc != null) {
            let sg = new aws_ec2_1.SecurityGroup(this, 'canarySecurityGroup', {
                description: 'Allow canary to communicate with load balancer',
                vpc: props.vpc,
                allowAllOutbound: true,
            });
            this.function = new aws_lambda_1.Function(this, 'canary', {
                runtime: MetricsHelper_1.MetricsHelper.PythonRuntime,
                code: aws_lambda_1.Code.fromAsset(path.join(__dirname, 'src/canary.zip')),
                handler: 'index.handler',
                role: executionRole,
                architecture: aws_lambda_1.Architecture.ARM_64,
                tracing: aws_lambda_1.Tracing.ACTIVE,
                timeout: aws_cdk_lib_1.Duration.seconds(240),
                memorySize: 512,
                layers: [monitoringLayer],
                environment: {
                    REGION: aws_cdk_lib_1.Fn.ref('AWS::Region'),
                    PARTITION: aws_cdk_lib_1.Fn.ref('AWS::Partition'),
                    TIMEOUT: props.httpTimeout !== undefined
                        ? props.httpTimeout.toSeconds().toString()
                        : '2',
                    IGNORE_SSL_ERRORS: (props.ignoreTlsErrors !== undefined && props.ignoreTlsErrors == true)
                        .toString()
                        .toLowerCase(),
                },
                vpc: props.vpc,
                securityGroups: [sg],
                vpcSubnets: props.subnetSelection,
            });
        }
        else {
            this.function = new aws_lambda_1.Function(this, 'canary', {
                runtime: MetricsHelper_1.MetricsHelper.PythonRuntime,
                code: aws_lambda_1.Code.fromAsset(path.join(__dirname, 'src/canary.zip')),
                handler: 'index.handler',
                role: executionRole,
                architecture: aws_lambda_1.Architecture.ARM_64,
                tracing: aws_lambda_1.Tracing.ACTIVE,
                timeout: aws_cdk_lib_1.Duration.seconds(240),
                memorySize: 512,
                layers: [monitoringLayer],
                environment: {
                    REGION: aws_cdk_lib_1.Fn.ref('AWS::Region'),
                    PARTITION: aws_cdk_lib_1.Fn.ref('AWS::Partition'),
                    TIMEOUT: props.httpTimeout !== undefined
                        ? props.httpTimeout.toSeconds().toString()
                        : '2',
                    IGNORE_SSL_ERRORS: (props.ignoreTlsErrors !== undefined && props.ignoreTlsErrors == true)
                        .toString()
                        .toLowerCase(),
                },
            });
        }
        this.function.addPermission('invokePermission', {
            action: 'lambda:InvokeFunction',
            principal: new aws_iam_1.ServicePrincipal('events.amazonaws.com'),
            sourceArn: aws_cdk_lib_1.Fn.sub('arn:${AWS::Partition}:events:${AWS::Region}:${AWS::AccountId}:rule/*'),
        });
        this.logGroup = new aws_logs_1.LogGroup(this, 'logGroup', {
            logGroupName: `/aws/lambda/${this.function.functionName}`,
            retention: aws_logs_1.RetentionDays.ONE_WEEK,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        new aws_iam_1.ManagedPolicy(this, 'cwManagedPolicy', {
            path: '/canary/',
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: ['cloudwatch:PutMetricData'],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: ['*'],
                }),
                new aws_iam_1.PolicyStatement({
                    actions: ['logs:CreateLogStream', 'logs:PutLogEvents'],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: [this.logGroup.logGroupArn],
                }),
            ],
            roles: [executionRole],
        });
    }
}
exports.CanaryFunction = CanaryFunction;
//# sourceMappingURL=data:application/json;base64,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