"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BasicServiceMultiAZObservability = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
const AvailabilityZoneMapper_1 = require("../azmapper/AvailabilityZoneMapper");
const BasicServiceDashboard_1 = require("./BasicServiceDashboard");
const ApplicationLoadBalancerMetrics_1 = require("../metrics/ApplicationLoadBalancerMetrics");
const AvailabilityMetricType_1 = require("../utilities/AvailabilityMetricType");
const MetricsHelper_1 = require("../utilities/MetricsHelper");
const ApplicationLoadBalancerLatencyOutlierCalculation_1 = require("./props/ApplicationLoadBalancerLatencyOutlierCalculation");
/**
 * Basic observability for a service using metrics from
 * ALBs and NAT Gateways
 */
class BasicServiceMultiAZObservability extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Initialize class properties
        this.serviceName = props.serviceName;
        this.applicationLoadBalancers = props.applicationLoadBalancers;
        this.natGateways = props.natGateways;
        this.aggregateZonalIsolatedImpactAlarms = {};
        this.albZonalIsolatedImpactAlarms = {};
        this.natGWZonalIsolatedImpactAlarms = {};
        // Create the AZ mapper resource to translate AZ names to ids
        this._azMapper = new AvailabilityZoneMapper_1.AvailabilityZoneMapper(this, 'availability-zone-mapper');
        // Create ALB metrics and alarms per AZ
        if (this.applicationLoadBalancers) {
            this.albZonalIsolatedImpactAlarms = this.doAlbMetrics(props);
        }
        // Create NAT Gateway metrics and alarms per AZ
        if (this.natGateways) {
            this.natGWZonalIsolatedImpactAlarms = this.createNatGatewayZonalImpactAlarms(props);
        }
        // Look through all of the per AZ ALB alarms, if there's also a NAT GW alarm
        // create a composite alarm if either of them trigger
        Object.keys(this.albZonalIsolatedImpactAlarms).forEach((azLetter) => {
            let availabilityZoneId = this._azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
            if (azLetter in this.natGWZonalIsolatedImpactAlarms) {
                this.aggregateZonalIsolatedImpactAlarms[azLetter] = new aws_cloudwatch_1.CompositeAlarm(this, azLetter + "-isolated-impact-alarm", {
                    alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(this.albZonalIsolatedImpactAlarms[azLetter], this.natGWZonalIsolatedImpactAlarms[azLetter]),
                    compositeAlarmName: availabilityZoneId + "-isolated-impact-alarm"
                });
            }
            else {
                this.aggregateZonalIsolatedImpactAlarms[azLetter] = new aws_cloudwatch_1.CompositeAlarm(this, azLetter + "-isolated-impact-alarm", {
                    alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(this.albZonalIsolatedImpactAlarms[azLetter]),
                    compositeAlarmName: availabilityZoneId + "-isolated-impact-alarm"
                });
            }
        });
        // Look through all of the per AZ NAT GW alarms. If there's an AZ we haven't seen in the ALB
        // alarms yet, then it will just be a NAT GW alarm that we'll turn into the same kind of
        // composite alarm
        Object.keys(this.natGWZonalIsolatedImpactAlarms).forEach((azLetter) => {
            if (!(azLetter in this.aggregateZonalIsolatedImpactAlarms)) {
                let availabilityZoneId = this._azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
                this.aggregateZonalIsolatedImpactAlarms[azLetter] = new aws_cloudwatch_1.CompositeAlarm(this, azLetter + "-isolated-impact-alarm", {
                    alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(this.natGWZonalIsolatedImpactAlarms[azLetter]),
                    compositeAlarmName: availabilityZoneId + "-isolated-impact-alarm"
                });
            }
        });
        // Should we create the dashboard
        if (props.createDashboard == true) {
            this.dashboard = new BasicServiceDashboard_1.BasicServiceDashboard(this, 'BasicServiceDashboard', {
                serviceName: props.serviceName.toLowerCase(),
                zonalAggregateIsolatedImpactAlarms: this.aggregateZonalIsolatedImpactAlarms,
                zonalLoadBalancerIsolatedImpactAlarms: this.albZonalIsolatedImpactAlarms,
                zonalNatGatewayIsolatedImpactAlarms: this.natGWZonalIsolatedImpactAlarms,
                interval: props.interval,
                albs: props.applicationLoadBalancers,
                /*zonalLoadBalancerFaultRateMetrics: ApplicationLoadBalancerMetrics.getTotalAlbFaultCountPerZone(
                  props.applicationLoadBalancers ? props.applicationLoadBalancers : [],
                  props.period ? props.period : Duration.minutes(1),
                  this._azMapper
                ),*/
                period: props.period ? props.period : aws_cdk_lib_1.Duration.minutes(1),
                zonalNatGatewayPacketDropMetrics: this.getTotalPacketDropsPerZone(props.natGateways ? props.natGateways : {}, props.period ? props.period : aws_cdk_lib_1.Duration.minutes(1)),
                azMapper: this._azMapper,
                latencyStatistic: props.latencyStatistic
            }).dashboard;
        }
    }
    static isThereAnAZAvailabilityImpactAlb(scope, alb, availabilityZoneId, availabilityZone, threshold, keyprefix, period, evaluationPeriods, datapointsToAlarm) {
        // Create a fault rate alarm for the ALB in the specified AZ
        return new aws_cloudwatch_1.Alarm(scope, keyprefix + '-fault-rate-alarm', {
            alarmName: availabilityZoneId + '-' + alb.loadBalancerArn + '-fault-rate',
            actionsEnabled: false,
            metric: ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.getPerAZAvailabilityMetric(alb, {
                period: period,
                label: availabilityZoneId + '-' + alb.loadBalancerArn + '-fault-rate',
                availabilityZone: availabilityZone,
                availabilityZoneId: availabilityZoneId,
                metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_RATE
            }),
            evaluationPeriods: evaluationPeriods,
            datapointsToAlarm: datapointsToAlarm,
            threshold: threshold,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE
        });
    }
    static isThereAZLatencyImpactAlb(scope, alb, availabilityZoneId, availabilityZone, threshold, statistic, keyprefix, period, evaluationPeriods, datapointsToAlarm) {
        // Create a fault rate alarm for the ALB in the specified AZ
        return new aws_cloudwatch_1.Alarm(scope, keyprefix + '-latency-alarm', {
            alarmName: availabilityZoneId + '-' + alb.loadBalancerArn + '-latency',
            actionsEnabled: false,
            metric: ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.getPerAZLatencyMetric({
                alb: alb,
                availabilityZone: availabilityZone,
                availabilityZoneId: availabilityZoneId,
                label: availabilityZoneId + "-" + alb.loadBalancerArn + "-target-latency",
                period: period,
                statistic: statistic
            }),
            evaluationPeriods: evaluationPeriods,
            datapointsToAlarm: datapointsToAlarm,
            threshold: threshold,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE
        });
    }
    static isAZAnOutlierForAvailabilityAlb(scope, alb, availabilityZoneId, availabilityZone, threshold, keyprefix, period, evaluationPeriods, datapointsToAlarm) {
        let usingMetrics = {};
        let azMetricId = "";
        alb.vpc.availabilityZones.forEach((az) => {
            let azFaultCount = ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.getPerAZAvailabilityMetric(alb, {
                metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
                availabilityZone: availabilityZone,
                availabilityZoneId: availabilityZoneId,
                period: period,
                label: availabilityZoneId + "-" + alb.loadBalancerArn + "-fault-count",
                keyprefix: keyprefix
            });
            keyprefix = MetricsHelper_1.MetricsHelper.nextChar(keyprefix);
            usingMetrics[`${keyprefix}1`] = azFaultCount;
            if (az == availabilityZone) {
                azMetricId = `${keyprefix}1`;
            }
            keyprefix = MetricsHelper_1.MetricsHelper.nextChar(keyprefix);
        });
        return new aws_cloudwatch_1.Alarm(scope, keyprefix + '-availability-outlier-alarm', {
            alarmName: availabilityZoneId + '-' + alb.loadBalancerArn + '-availability-impact-outlier',
            actionsEnabled: false,
            metric: new aws_cloudwatch_1.MathExpression({
                expression: `${azMetricId}/(${Object.keys(usingMetrics).join("+")})`,
                usingMetrics: usingMetrics,
                label: availabilityZoneId + '-' + alb.loadBalancerArn + '-percent-of-faults',
                period: period,
            }),
            evaluationPeriods: evaluationPeriods,
            datapointsToAlarm: datapointsToAlarm,
            threshold: threshold,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE
        });
    }
    static isAZAnOutlierForLatencyAlb(scope, alb, algorithm, availabilityZoneId, availabilityZone, statistic, threshold, period, evaluationPeriods, datapointsToAlarm, keyprefix) {
        let usingMetrics = {};
        let azMetricId = "";
        switch (algorithm) {
            case ApplicationLoadBalancerLatencyOutlierCalculation_1.ApplicationLoadBalancerLatencyOutlierCalculation.Z_SCORE:
            default:
                alb.vpc.availabilityZones.forEach((az, index) => {
                    // Target response time
                    let targetResponseTime = ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.getPerAZLatencyMetric({
                        alb: alb,
                        availabilityZone: az,
                        label: az + "-target-response-time",
                        statistic: statistic,
                        period: period
                    });
                    if (az == availabilityZone) {
                        azMetricId = `a${index}`;
                        usingMetrics[`a${index}`] = targetResponseTime;
                    }
                    else {
                        usingMetrics[`b${index}`] = targetResponseTime;
                    }
                });
                return new aws_cloudwatch_1.Alarm(scope, keyprefix + "-latency-outlier-alarm", {
                    alarmName: availabilityZoneId + '-' + alb.loadBalancerArn + '-latency-impact-outlier',
                    actionsEnabled: false,
                    metric: new aws_cloudwatch_1.MathExpression({
                        expression: `(${azMetricId} - AVG(METRICS("b"))) / AVG(STDDEV(METRICS("b")))`,
                        usingMetrics: usingMetrics,
                        label: availabilityZoneId + '-' + alb.loadBalancerArn + '-latency-z-score',
                        period: period,
                    }),
                    evaluationPeriods: evaluationPeriods,
                    datapointsToAlarm: datapointsToAlarm,
                    threshold: threshold,
                    comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
                    treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE
                });
            case ApplicationLoadBalancerLatencyOutlierCalculation_1.ApplicationLoadBalancerLatencyOutlierCalculation.STATIC:
                alb.vpc.availabilityZones.forEach((az, index) => {
                    // Target response time
                    let targetResponseTime = ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.getPerAZLatencyMetric({
                        alb: alb,
                        availabilityZone: az,
                        label: az + "-target-response-time",
                        statistic: `TC(${threshold}:)`,
                        period: period
                    });
                    if (az == availabilityZone) {
                        azMetricId = `a${index}`;
                        usingMetrics[`a${index}`] = targetResponseTime;
                    }
                    else {
                        usingMetrics[`b${index}`] = targetResponseTime;
                    }
                });
                return new aws_cloudwatch_1.Alarm(scope, keyprefix + "-latency-outlier-alarm", {
                    alarmName: availabilityZoneId + '-' + alb.loadBalancerArn + '-latency-impact-outlier',
                    actionsEnabled: false,
                    metric: new aws_cloudwatch_1.MathExpression({
                        expression: `${azMetricId}/(${Object.keys(usingMetrics).join("+")})`,
                        usingMetrics: usingMetrics,
                        label: availabilityZoneId + '-' + alb.loadBalancerArn + '-latency-static',
                        period: period,
                    }),
                    evaluationPeriods: evaluationPeriods,
                    datapointsToAlarm: datapointsToAlarm,
                    threshold: .66,
                    comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
                    treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE
                });
        }
    }
    /**
     *
     * @param props
     * @returns A composite alarm per AZ to indicate isolated zonal impact. The dictionary
     * is keyed by the az letter, like "a", "b", "c".
     */
    doAlbMetrics(props) {
        // Create impact alarms per AZ, with each ALB providing
        // an alarm for its AZs
        let perAZImpactAlarms = {};
        let keyPrefix = MetricsHelper_1.MetricsHelper.nextChar();
        // Iterate each ALB
        this.applicationLoadBalancers.forEach((alb) => {
            // Iterate each AZ in the VPC
            alb.vpc?.availabilityZones.forEach((az) => {
                if (!(az in perAZImpactAlarms)) {
                    perAZImpactAlarms[az] = [];
                }
                // Get AZ letter
                let azLetter = az.substring(az.length - 1);
                // Map letter to AZ ID
                let availabilityZoneId = this._azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
                // Is there availability impact in this AZ?
                let availabilityImpact = BasicServiceMultiAZObservability.isThereAnAZAvailabilityImpactAlb(this, alb, availabilityZoneId, az, props.faultCountPercentageThreshold ? props.faultCountPercentageThreshold : 0.05, keyPrefix, props.period ? props.period : aws_cdk_lib_1.Duration.minutes(1), props.evaluationPeriods, props.datapointsToAlarm);
                // Is there latency impact in this AZ?
                let latencyImpact = BasicServiceMultiAZObservability.isThereAZLatencyImpactAlb(this, alb, availabilityZoneId, az, props.latencyThreshold, props.latencyStatistic, keyPrefix, props.period ? props.period : aws_cdk_lib_1.Duration.minutes(1), props.evaluationPeriods, props.datapointsToAlarm);
                // Is the AZ an outlier for faults
                let availabilityOutlier = BasicServiceMultiAZObservability.isAZAnOutlierForAvailabilityAlb(this, alb, availabilityZoneId, az, props.faultCountPercentageThreshold ? props.faultCountPercentageThreshold : 0.05, keyPrefix, props.period ? props.period : aws_cdk_lib_1.Duration.minutes(1), props.evaluationPeriods, props.datapointsToAlarm);
                // Is the AZ an outlier for latency
                let latencyOutlier = BasicServiceMultiAZObservability.isAZAnOutlierForLatencyAlb(this, alb, props.latencyOutlierCalculation ? props.latencyOutlierCalculation : ApplicationLoadBalancerLatencyOutlierCalculation_1.ApplicationLoadBalancerLatencyOutlierCalculation.Z_SCORE, availabilityZoneId, az, props.latencyStatistic, props.latencyOutlierCalculation == ApplicationLoadBalancerLatencyOutlierCalculation_1.ApplicationLoadBalancerLatencyOutlierCalculation.Z_SCORE ? 3 : props.latencyThreshold, props.period ? props.period : aws_cdk_lib_1.Duration.minutes(1), props.evaluationPeriods, props.datapointsToAlarm, azLetter);
                // Alarm if the AZ shows impact and is an outlier
                let azImpactAlarm = new aws_cloudwatch_1.CompositeAlarm(this, azLetter + "-composite-impact-alarm", {
                    alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(aws_cloudwatch_1.AlarmRule.allOf(availabilityImpact, availabilityOutlier), aws_cloudwatch_1.AlarmRule.allOf(latencyImpact, latencyOutlier)),
                    compositeAlarmName: availabilityZoneId + "-" +
                        alb.loadBalancerName +
                        "-latency-or-availability-impact",
                    actionsEnabled: false
                });
                // Add this ALB's fault rate alarm
                perAZImpactAlarms[az].push(azImpactAlarm);
                // Get next unique key
                keyPrefix = MetricsHelper_1.MetricsHelper.nextChar(keyPrefix);
            });
        });
        let azCompositeAlarms = {};
        // Iterate AZs for the ALB impact alarms so we can join them
        // into a single composite alarm for each AZ
        Object.keys(perAZImpactAlarms).forEach((az) => {
            let azLetter = az.substring(az.length - 1);
            let availabilityZoneId = this._azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
            azCompositeAlarms[azLetter] = new aws_cloudwatch_1.CompositeAlarm(this, azLetter + "-alb-impact-composite-alarm", {
                alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...perAZImpactAlarms[az]),
                compositeAlarmName: availabilityZoneId + "-alb-impact-composite-alarm"
            });
        });
        return azCompositeAlarms;
    }
    static isThereAnAZPacketLossImpactNATGW(scope, natgws, availabilityZoneId, availabilityZone, threshold, period, evaluationPeriods, datapointsToAlarm) {
        let keyprefix = MetricsHelper_1.MetricsHelper.nextChar();
        let packetDropCountMetrics = {};
        let packetsInFromSourceMetrics = {};
        let packetsInFromDestinationMetrics = {};
        natgws.forEach((natgw) => {
            packetDropCountMetrics[`${keyprefix}1`] = new aws_cloudwatch_1.Metric({
                metricName: 'PacketsDropCount',
                namespace: 'AWS/NATGateway',
                statistic: aws_cloudwatch_1.Stats.SUM,
                unit: aws_cloudwatch_1.Unit.COUNT,
                label: availabilityZoneId + ' packet drops',
                dimensionsMap: {
                    NatGatewayId: natgw.attrNatGatewayId,
                },
                period: period,
            });
            // Calculate packets in from source
            packetsInFromSourceMetrics[`${keyprefix}2`] = new aws_cloudwatch_1.Metric({
                metricName: 'PacketsInFromSource',
                namespace: 'AWS/NATGateway',
                statistic: aws_cloudwatch_1.Stats.SUM,
                unit: aws_cloudwatch_1.Unit.COUNT,
                label: availabilityZoneId + ' packets in from source',
                dimensionsMap: {
                    NatGatewayId: natgw.attrNatGatewayId,
                },
                period: period,
            });
            // Calculate packets in from destination
            packetsInFromDestinationMetrics[`${keyprefix}3`] = new aws_cloudwatch_1.Metric({
                metricName: 'PacketsInFromDestination',
                namespace: 'AWS/NATGateway',
                statistic: aws_cloudwatch_1.Stats.SUM,
                unit: aws_cloudwatch_1.Unit.COUNT,
                label: availabilityZoneId + ' packets in from destination',
                dimensionsMap: {
                    NatGatewayId: natgw.attrNatGatewayId,
                },
                period: period,
            });
            keyprefix = MetricsHelper_1.MetricsHelper.nextChar(keyprefix);
        });
        let packetDropTotal = new aws_cloudwatch_1.MathExpression({
            expression: Object.keys(packetDropCountMetrics).join("+"),
            usingMetrics: packetDropCountMetrics,
            period: period
        });
        let packetsInFromSourceTotal = new aws_cloudwatch_1.MathExpression({
            expression: Object.keys(packetsInFromSourceMetrics).join("+"),
            usingMetrics: packetsInFromSourceMetrics,
            period: period
        });
        let packetsInFromDestinationTotal = new aws_cloudwatch_1.MathExpression({
            expression: Object.keys(packetsInFromDestinationMetrics).join("+"),
            usingMetrics: packetsInFromDestinationMetrics,
            period: period
        });
        let usingMetrics = {};
        usingMetrics[`${keyprefix}1`] = packetDropTotal;
        usingMetrics[`${keyprefix}2`] = packetsInFromSourceTotal;
        usingMetrics[`${keyprefix}3`] = packetsInFromDestinationTotal;
        // Calculate a percentage of dropped packets for the NAT GW
        let packetDropPercentage = new aws_cloudwatch_1.MathExpression({
            expression: `(${keyprefix}1 / (${keyprefix}2 + ${keyprefix}3))`,
            usingMetrics: usingMetrics,
            label: availabilityZoneId + ' packet drop percentage',
            period: period,
        });
        // Create an alarm for this NAT GW if packet drops exceed the specified threshold
        return new aws_cloudwatch_1.Alarm(scope, availabilityZone.substring(availabilityZone.length - 1) + "-packet-drop-impact-alarm", {
            alarmName: availabilityZoneId +
                '-packet-drop-impact',
            actionsEnabled: false,
            metric: packetDropPercentage,
            threshold: threshold,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: evaluationPeriods,
            datapointsToAlarm: datapointsToAlarm,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE
        });
    }
    static isAZAnOutlierForPacketLossNATGW(scope, natgws, availabilityZoneId, availabilityZone, threshold, period, evaluationPeriods, datapointsToAlarm) {
        let keyprefix = MetricsHelper_1.MetricsHelper.nextChar();
        let azPacketDropCountMetrics = {};
        let azKey = "";
        Object.keys(natgws).forEach((az) => {
            let packetDropCountMetrics = {};
            natgws[az].forEach((natgw, index) => {
                packetDropCountMetrics[`${keyprefix}${index}`] = new aws_cloudwatch_1.Metric({
                    metricName: 'PacketsDropCount',
                    namespace: 'AWS/NATGateway',
                    statistic: aws_cloudwatch_1.Stats.SUM,
                    unit: aws_cloudwatch_1.Unit.COUNT,
                    label: availabilityZoneId + ' packet drops',
                    dimensionsMap: {
                        NatGatewayId: natgw.attrNatGatewayId,
                    },
                    period: period,
                });
                keyprefix = MetricsHelper_1.MetricsHelper.nextChar(keyprefix);
            });
            azPacketDropCountMetrics[`${keyprefix}${natgws[az].length}`] = new aws_cloudwatch_1.MathExpression({
                expression: Object.keys(packetDropCountMetrics).join("+"),
                usingMetrics: packetDropCountMetrics,
                period: period
            });
            if (az == availabilityZone) {
                azKey = `${keyprefix}${natgws[az].length}`;
            }
            keyprefix = MetricsHelper_1.MetricsHelper.nextChar(keyprefix);
        });
        return new aws_cloudwatch_1.Alarm(scope, availabilityZone.substring(availabilityZone.length - 1) + "-packet-loss-outlier", {
            metric: new aws_cloudwatch_1.MathExpression({
                expression: `${azKey} / (${Object.keys(azPacketDropCountMetrics).join("+")})`,
                usingMetrics: azPacketDropCountMetrics,
                period: period
            }),
            threshold: threshold,
            evaluationPeriods: evaluationPeriods,
            datapointsToAlarm: datapointsToAlarm,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE
        });
    }
    createNatGatewayZonalImpactAlarms(props) {
        // Collect alarms for packet drops exceeding a threshold per NAT GW
        let packetLossPerAZAlarms = {};
        // For each AZ, create metrics for each NAT GW
        Object.keys(this.natGateways).forEach((az) => {
            let azLetter = az.substring(az.length - 1);
            let availabilityZoneId = this._azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
            // Is there packet loss impact?
            let packetLossImpact = BasicServiceMultiAZObservability.isThereAnAZPacketLossImpactNATGW(this, this.natGateways[az], availabilityZoneId, az, props.packetLossImpactPercentageThreshold ? props.packetLossImpactPercentageThreshold : 0.01, props.period ? props.period : aws_cdk_lib_1.Duration.minutes(1), props.evaluationPeriods, props.datapointsToAlarm);
            // Is this AZ an outlier for this NATGW?
            let packetLossOutlier = BasicServiceMultiAZObservability.isAZAnOutlierForPacketLossNATGW(this, this.natGateways, availabilityZoneId, az, 0.66, props.period ? props.period : aws_cdk_lib_1.Duration.minutes(1), props.evaluationPeriods, props.datapointsToAlarm);
            packetLossPerAZAlarms[azLetter] = new aws_cloudwatch_1.CompositeAlarm(this, az.substring(az.length - 1) + "-packet-loss-composite-alarm", {
                alarmRule: aws_cloudwatch_1.AlarmRule.allOf(packetLossImpact, packetLossOutlier),
                compositeAlarmName: availabilityZoneId + "-packet-loss-composite-alarm"
            });
        });
        return packetLossPerAZAlarms;
    }
    getTotalPacketDropsPerZone(natgws, period) {
        let dropsPerZone = {};
        let metricsPerAZ = {};
        let keyprefix = MetricsHelper_1.MetricsHelper.nextChar();
        Object.keys(natgws).forEach((availabilityZone) => {
            let azLetter = availabilityZone.substring(availabilityZone.length - 1);
            let availabilityZoneId = this._azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
            if (!(azLetter in metricsPerAZ)) {
                metricsPerAZ[azLetter] = [];
            }
            natgws[availabilityZone].forEach((natgw) => {
                metricsPerAZ[azLetter].push(new aws_cloudwatch_1.Metric({
                    metricName: 'PacketsDropCount',
                    namespace: 'AWS/NATGateway',
                    statistic: aws_cloudwatch_1.Stats.SUM,
                    unit: aws_cloudwatch_1.Unit.COUNT,
                    label: availabilityZoneId + ' packet drops',
                    dimensionsMap: {
                        NatGatewayId: natgw.attrNatGatewayId,
                    },
                    period: period,
                }));
            });
        });
        Object.keys(metricsPerAZ).forEach((availabilityZone) => {
            let azLetter = availabilityZone.substring(availabilityZone.length - 1);
            let availabilityZoneId = this._azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
            let usingMetrics = {};
            metricsPerAZ[azLetter].forEach((metric) => {
                usingMetrics[`${keyprefix}1`] = metric;
                keyprefix = MetricsHelper_1.MetricsHelper.nextChar(keyprefix);
            });
            keyprefix = MetricsHelper_1.MetricsHelper.nextChar(keyprefix);
            dropsPerZone[availabilityZone] = new aws_cloudwatch_1.MathExpression({
                expression: Object.keys(usingMetrics).join("+"),
                usingMetrics: usingMetrics,
                label: availabilityZoneId + " total packet drops",
                period: period
            });
        });
        return dropsPerZone;
    }
}
exports.BasicServiceMultiAZObservability = BasicServiceMultiAZObservability;
_a = JSII_RTTI_SYMBOL_1;
BasicServiceMultiAZObservability[_a] = { fqn: "@cdklabs/multi-az-observability.BasicServiceMultiAZObservability", version: "0.0.1-alpha.14" };
//# sourceMappingURL=data:application/json;base64,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