"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BasicServiceDashboard = void 0;
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
const MetricsHelper_1 = require("../utilities/MetricsHelper");
const ApplicationLoadBalancerMetrics_1 = require("../metrics/ApplicationLoadBalancerMetrics");
class BasicServiceDashboard extends constructs_1.Construct {
    static generateLoadBalancerWidgets(albs, azMapper, period, latencyStatistic) {
        let albWidgets = [];
        albWidgets.push(new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: "Load Balancer Metrics" }));
        let successCountPerZone = ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.getTotalAlbSuccessCountPerZone(albs, period, azMapper);
        let faultCountPerZone = ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.getTotalAlbFaultCountPerZone(albs, period, azMapper);
        let processedBytesPerZone = ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.getTotalAlbProcessedBytesPerZone(albs, period, azMapper);
        let latencyPerZone = ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.getTotalAlbLatencyPerZone(albs, latencyStatistic, period, azMapper);
        let requestsPerZone = ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.getTotalAlbRequestsPerZone(albs, period, azMapper);
        let faultRatePerZone = ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.getTotalAlbFaultRatePerZone(albs, period, azMapper);
        albWidgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 8,
            width: 8,
            title: aws_cdk_lib_1.Fn.sub('${AWS::Region} Zonal Success Count'),
            region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
            left: Object.values(successCountPerZone),
            leftYAxis: {
                min: 0,
                label: 'Sum',
                showUnits: false,
            }
        }));
        albWidgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 8,
            width: 8,
            title: aws_cdk_lib_1.Fn.sub('${AWS::Region} Zonal Fault Count'),
            region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
            left: Object.values(faultCountPerZone),
            leftYAxis: {
                min: 0,
                label: 'Sum',
                showUnits: false,
            }
        }));
        albWidgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 8,
            width: 8,
            title: aws_cdk_lib_1.Fn.sub('${AWS::Region} Zonal Request Count'),
            region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
            left: Object.values(requestsPerZone),
            leftYAxis: {
                min: 0,
                label: 'Sum',
                showUnits: false,
            }
        }));
        albWidgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 8,
            width: 8,
            title: aws_cdk_lib_1.Fn.sub('${AWS::Region} Zonal Fault Rate'),
            region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
            left: Object.values(faultRatePerZone),
            leftYAxis: {
                min: 0,
                label: 'Percent',
                showUnits: false,
            }
        }));
        albWidgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 8,
            width: 8,
            title: aws_cdk_lib_1.Fn.sub('${AWS::Region} Zonal Processed Bytes'),
            region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
            left: Object.values(processedBytesPerZone),
            leftYAxis: {
                min: 0,
                showUnits: false,
                label: 'Bytes'
            }
        }));
        albWidgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 8,
            width: 8,
            title: aws_cdk_lib_1.Fn.sub('${AWS::Region} Zonal Target Response Time (' + latencyStatistic + ')'),
            region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
            left: Object.values(latencyPerZone),
            leftYAxis: {
                min: 0,
                label: "Milliseconds",
                showUnits: false,
            }
        }));
        return albWidgets;
    }
    /*private static createLoadBalancerWidgets(
      alarms: { [key: string]: IAlarm },
      metrics: { [key: string]: IMetric },
      azMapper: AvailabilityZoneMapper,
    ): IWidget[] {
      let widgets: IWidget[] = [];
  
      widgets.push(
        new TextWidget({
          markdown: 'Load Balancer Metrics',
          height: 2,
          width: 24,
        }),
      );
  
      let rowTracker: number = 0;
  
      Object.keys(alarms).forEach((azLetter, index) => {
        let availabilityZoneId: string = azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
  
        widgets.push(
          new GraphWidget({
            height: 6,
            width: 8,
            title: availabilityZoneId + ' Load Balancer Faults',
            region: Fn.sub('${AWS::Region}'),
            left: [metrics[azLetter]],
            statistic: 'Sum',
            leftYAxis: {
              min: 0,
              label: 'Fault Count',
              showUnits: false,
            },
          }),
        );
  
        //We're on the third one for this set, add 3 alarms
        //or if we're at the end, at the necessary amount
        //of alarms, 1, 2, or 3
        if (index % 3 == 2 || index - 1 == Object.keys(alarms).length) {
          for (let k = rowTracker; k <= index; k++) {
            let azId: string = Object.keys(alarms).at(k)!;
            widgets.push(
              new AlarmStatusWidget({
                height: 2,
                width: 8,
                alarms: [alarms[azId]],
              }),
            );
          }
  
          rowTracker += index + 1;
        }
      });
  
      return widgets;
    }*/
    static createNatGatewayWidgets(alarms, metrics, azMapper) {
        let widgets = [];
        widgets.push(new aws_cloudwatch_1.TextWidget({
            markdown: 'NAT Gateway Dropped Packet Metrics',
            height: 2,
            width: 24,
        }));
        let rowTracker = 0;
        Object.keys(alarms).forEach((azLetter, index) => {
            let availabilityZoneId = azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
            widgets.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 8,
                title: availabilityZoneId + ' NAT Gateway Dropped Packets',
                region: aws_cdk_lib_1.Fn.sub('${AWS::Region}'),
                left: [metrics[azLetter]],
                statistic: 'Sum',
                leftYAxis: {
                    min: 0,
                    label: 'Dropped packets',
                    showUnits: false,
                },
            }));
            //We're on the third one for this set, add 3 alarms
            //or if we're at the end, at the necessary amount
            //of alarms, 1, 2, or 3
            if (index % 3 == 2 || index - 1 == Object.keys(alarms).length) {
                for (let k = rowTracker; k <= index; k++) {
                    let azId = Object.keys(alarms).at(k);
                    widgets.push(new aws_cloudwatch_1.AlarmStatusWidget({
                        height: 2,
                        width: 8,
                        alarms: [alarms[azId]],
                    }));
                }
                rowTracker += index + 1;
            }
        });
        return widgets;
    }
    static createTopLevelAlarmWidgets(alarms, azMapper) {
        let widgets = [];
        widgets.push(new aws_cloudwatch_1.TextWidget({
            markdown: 'Availability Zone Isolated Impact Alarms',
            height: 2,
            width: 24,
        }));
        Object.keys(alarms).forEach((azLetter) => {
            let availabilityZoneId = azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
            widgets.push(new aws_cloudwatch_1.AlarmStatusWidget({
                alarms: [alarms[azLetter]],
                height: 2,
                width: 8,
                title: availabilityZoneId + ' Aggregate Isolated Impact',
            }));
        });
        return widgets;
    }
    constructor(scope, id, props) {
        super(scope, id);
        let widgets = [];
        widgets.push(BasicServiceDashboard.createTopLevelAlarmWidgets(props.zonalAggregateIsolatedImpactAlarms, props.azMapper));
        /*if (
          MetricsHelper.isNotEmpty(props.zonalLoadBalancerIsolatedImpactAlarms) &&
          MetricsHelper.isNotEmpty(props.zonalLoadBalancerFaultRateMetrics)
        ) {
          widgets.push(
            BasicServiceDashboard.createLoadBalancerWidgets(
              props.zonalLoadBalancerIsolatedImpactAlarms,
              props.zonalLoadBalancerFaultRateMetrics,
              props.azMapper,
            ),
          );
        }*/
        if (props.albs) {
            widgets.push(BasicServiceDashboard.generateLoadBalancerWidgets(props.albs, props.azMapper, props.period, props.latencyStatistic));
        }
        if (MetricsHelper_1.MetricsHelper.isNotEmpty(props.zonalNatGatewayIsolatedImpactAlarms) &&
            MetricsHelper_1.MetricsHelper.isNotEmpty(props.zonalNatGatewayPacketDropMetrics)) {
            widgets.push(BasicServiceDashboard.createNatGatewayWidgets(props.zonalNatGatewayIsolatedImpactAlarms, props.zonalNatGatewayPacketDropMetrics, props.azMapper));
        }
        this.dashboard = new aws_cloudwatch_1.Dashboard(this, 'BasicServiceDashboard', {
            dashboardName: props.serviceName.toLowerCase() +
                aws_cdk_lib_1.Fn.sub('-per-az-health-${AWS::Region}'),
            defaultInterval: props.interval,
            periodOverride: aws_cloudwatch_1.PeriodOverride.AUTO,
            widgets: widgets,
        });
    }
}
exports.BasicServiceDashboard = BasicServiceDashboard;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiQmFzaWNTZXJ2aWNlRGFzaGJvYXJkLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2Jhc2ljX29ic2VydmFiaWxpdHkvQmFzaWNTZXJ2aWNlRGFzaGJvYXJkLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHFFQUFxRTtBQUNyRSxzQ0FBc0M7QUFDdEMsNkNBQTJDO0FBQzNDLCtEQVNvQztBQUNwQywyQ0FBdUM7QUFHdkMsOERBQTJEO0FBQzNELDhGQUEyRjtBQUczRixNQUFhLHFCQUFzQixTQUFRLHNCQUFTO0lBRTFDLE1BQU0sQ0FBQywyQkFBMkIsQ0FDeEMsSUFBZ0MsRUFDaEMsUUFBZ0MsRUFDaEMsTUFBZ0IsRUFDaEIsZ0JBQXdCO1FBRXhCLElBQUksVUFBVSxHQUFjLEVBQUUsQ0FBQztRQUUvQixVQUFVLENBQUMsSUFBSSxDQUFDLElBQUksMkJBQVUsQ0FBQyxFQUFFLE1BQU0sRUFBRSxDQUFDLEVBQUUsS0FBSyxFQUFFLEVBQUUsRUFBRSxRQUFRLEVBQUUsdUJBQXVCLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFFN0YsSUFBSSxtQkFBbUIsR0FDckIsK0RBQThCLENBQUMsOEJBQThCLENBQUMsSUFBSSxFQUFFLE1BQU0sRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN4RixJQUFJLGlCQUFpQixHQUNuQiwrREFBOEIsQ0FBQyw0QkFBNEIsQ0FBQyxJQUFJLEVBQUUsTUFBTSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3RGLElBQUkscUJBQXFCLEdBQ3ZCLCtEQUE4QixDQUFDLGdDQUFnQyxDQUFDLElBQUksRUFBRSxNQUFNLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDMUYsSUFBSSxjQUFjLEdBQ2hCLCtEQUE4QixDQUFDLHlCQUF5QixDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRSxNQUFNLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDckcsSUFBSSxlQUFlLEdBQ2pCLCtEQUE4QixDQUFDLDBCQUEwQixDQUFDLElBQUksRUFBRSxNQUFNLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDcEYsSUFBSSxnQkFBZ0IsR0FDbEIsK0RBQThCLENBQUMsMkJBQTJCLENBQUMsSUFBSSxFQUFFLE1BQU0sRUFBRSxRQUFRLENBQUMsQ0FBQztRQUVyRixVQUFVLENBQUMsSUFBSSxDQUNiLElBQUksNEJBQVcsQ0FBQztZQUNkLE1BQU0sRUFBRSxDQUFDO1lBQ1QsS0FBSyxFQUFFLENBQUM7WUFDUixLQUFLLEVBQUUsZ0JBQUUsQ0FBQyxHQUFHLENBQUMsb0NBQW9DLENBQUM7WUFDbkQsTUFBTSxFQUFFLGdCQUFFLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDO1lBQ2hDLElBQUksRUFBRSxNQUFNLENBQUMsTUFBTSxDQUFDLG1CQUFtQixDQUFDO1lBQ3hDLFNBQVMsRUFBRTtnQkFDVCxHQUFHLEVBQUUsQ0FBQztnQkFDTixLQUFLLEVBQUUsS0FBSztnQkFDWixTQUFTLEVBQUUsS0FBSzthQUNqQjtTQUNGLENBQUMsQ0FDSCxDQUFDO1FBRUYsVUFBVSxDQUFDLElBQUksQ0FDYixJQUFJLDRCQUFXLENBQUM7WUFDZCxNQUFNLEVBQUUsQ0FBQztZQUNULEtBQUssRUFBRSxDQUFDO1lBQ1IsS0FBSyxFQUFFLGdCQUFFLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxDQUFDO1lBQ2pELE1BQU0sRUFBRSxnQkFBRSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQztZQUNoQyxJQUFJLEVBQUUsTUFBTSxDQUFDLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQztZQUN0QyxTQUFTLEVBQUU7Z0JBQ1QsR0FBRyxFQUFFLENBQUM7Z0JBQ04sS0FBSyxFQUFFLEtBQUs7Z0JBQ1osU0FBUyxFQUFFLEtBQUs7YUFDakI7U0FDRixDQUFDLENBQ0gsQ0FBQztRQUVGLFVBQVUsQ0FBQyxJQUFJLENBQ2IsSUFBSSw0QkFBVyxDQUFDO1lBQ2QsTUFBTSxFQUFFLENBQUM7WUFDVCxLQUFLLEVBQUUsQ0FBQztZQUNSLEtBQUssRUFBRSxnQkFBRSxDQUFDLEdBQUcsQ0FBQyxvQ0FBb0MsQ0FBQztZQUNuRCxNQUFNLEVBQUUsZ0JBQUUsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUM7WUFDaEMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxNQUFNLENBQUMsZUFBZSxDQUFDO1lBQ3BDLFNBQVMsRUFBRTtnQkFDVCxHQUFHLEVBQUUsQ0FBQztnQkFDTixLQUFLLEVBQUUsS0FBSztnQkFDWixTQUFTLEVBQUUsS0FBSzthQUNqQjtTQUNGLENBQUMsQ0FDSCxDQUFDO1FBRUYsVUFBVSxDQUFDLElBQUksQ0FDYixJQUFJLDRCQUFXLENBQUM7WUFDZCxNQUFNLEVBQUUsQ0FBQztZQUNULEtBQUssRUFBRSxDQUFDO1lBQ1IsS0FBSyxFQUFFLGdCQUFFLENBQUMsR0FBRyxDQUFDLGlDQUFpQyxDQUFDO1lBQ2hELE1BQU0sRUFBRSxnQkFBRSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQztZQUNoQyxJQUFJLEVBQUUsTUFBTSxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQztZQUNyQyxTQUFTLEVBQUU7Z0JBQ1QsR0FBRyxFQUFFLENBQUM7Z0JBQ04sS0FBSyxFQUFFLFNBQVM7Z0JBQ2hCLFNBQVMsRUFBRSxLQUFLO2FBQ2pCO1NBQ0YsQ0FBQyxDQUNILENBQUM7UUFFRixVQUFVLENBQUMsSUFBSSxDQUNiLElBQUksNEJBQVcsQ0FBQztZQUNkLE1BQU0sRUFBRSxDQUFDO1lBQ1QsS0FBSyxFQUFFLENBQUM7WUFDUixLQUFLLEVBQUUsZ0JBQUUsQ0FBQyxHQUFHLENBQUMsc0NBQXNDLENBQUM7WUFDckQsTUFBTSxFQUFFLGdCQUFFLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDO1lBQ2hDLElBQUksRUFBRSxNQUFNLENBQUMsTUFBTSxDQUFDLHFCQUFxQixDQUFDO1lBQzFDLFNBQVMsRUFBRTtnQkFDVCxHQUFHLEVBQUUsQ0FBQztnQkFDTixTQUFTLEVBQUUsS0FBSztnQkFDaEIsS0FBSyxFQUFFLE9BQU87YUFDZjtTQUNGLENBQUMsQ0FDSCxDQUFDO1FBRUYsVUFBVSxDQUFDLElBQUksQ0FDYixJQUFJLDRCQUFXLENBQUM7WUFDZCxNQUFNLEVBQUUsQ0FBQztZQUNULEtBQUssRUFBRSxDQUFDO1lBQ1IsS0FBSyxFQUFFLGdCQUFFLENBQUMsR0FBRyxDQUFDLDZDQUE2QyxHQUFHLGdCQUFnQixHQUFHLEdBQUcsQ0FBQztZQUNyRixNQUFNLEVBQUUsZ0JBQUUsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUM7WUFDaEMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxNQUFNLENBQUMsY0FBYyxDQUFDO1lBQ25DLFNBQVMsRUFBRTtnQkFDVCxHQUFHLEVBQUUsQ0FBQztnQkFDTixLQUFLLEVBQUUsY0FBYztnQkFDckIsU0FBUyxFQUFFLEtBQUs7YUFDakI7U0FDRixDQUFDLENBQ0gsQ0FBQztRQUVGLE9BQU8sVUFBVSxDQUFDO0lBQ3BCLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0F3REc7SUFFSyxNQUFNLENBQUMsdUJBQXVCLENBQ3BDLE1BQWlDLEVBQ2pDLE9BQW1DLEVBQ25DLFFBQWdDO1FBRWhDLElBQUksT0FBTyxHQUFjLEVBQUUsQ0FBQztRQUU1QixPQUFPLENBQUMsSUFBSSxDQUNWLElBQUksMkJBQVUsQ0FBQztZQUNiLFFBQVEsRUFBRSxvQ0FBb0M7WUFDOUMsTUFBTSxFQUFFLENBQUM7WUFDVCxLQUFLLEVBQUUsRUFBRTtTQUNWLENBQUMsQ0FDSCxDQUFDO1FBRUYsSUFBSSxVQUFVLEdBQVcsQ0FBQyxDQUFDO1FBRTNCLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsUUFBUSxFQUFFLEtBQUssRUFBRSxFQUFFO1lBRTlDLElBQUksa0JBQWtCLEdBQVcsUUFBUSxDQUFDLDRDQUE0QyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBRWpHLE9BQU8sQ0FBQyxJQUFJLENBQ1YsSUFBSSw0QkFBVyxDQUFDO2dCQUNkLE1BQU0sRUFBRSxDQUFDO2dCQUNULEtBQUssRUFBRSxDQUFDO2dCQUNSLEtBQUssRUFBRSxrQkFBa0IsR0FBRyw4QkFBOEI7Z0JBQzFELE1BQU0sRUFBRSxnQkFBRSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQztnQkFDaEMsSUFBSSxFQUFFLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDO2dCQUN6QixTQUFTLEVBQUUsS0FBSztnQkFDaEIsU0FBUyxFQUFFO29CQUNULEdBQUcsRUFBRSxDQUFDO29CQUNOLEtBQUssRUFBRSxpQkFBaUI7b0JBQ3hCLFNBQVMsRUFBRSxLQUFLO2lCQUNqQjthQUNGLENBQUMsQ0FDSCxDQUFDO1lBRUYsbURBQW1EO1lBQ25ELGlEQUFpRDtZQUNqRCx1QkFBdUI7WUFDdkIsSUFBSSxLQUFLLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLEdBQUcsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsTUFBTSxFQUFFLENBQUM7Z0JBQzlELEtBQUssSUFBSSxDQUFDLEdBQUcsVUFBVSxFQUFFLENBQUMsSUFBSSxLQUFLLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQztvQkFDekMsSUFBSSxJQUFJLEdBQVcsTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFFLENBQUM7b0JBQzlDLE9BQU8sQ0FBQyxJQUFJLENBQ1YsSUFBSSxrQ0FBaUIsQ0FBQzt3QkFDcEIsTUFBTSxFQUFFLENBQUM7d0JBQ1QsS0FBSyxFQUFFLENBQUM7d0JBQ1IsTUFBTSxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDO3FCQUN2QixDQUFDLENBQ0gsQ0FBQztnQkFDSixDQUFDO2dCQUVELFVBQVUsSUFBSSxLQUFLLEdBQUcsQ0FBQyxDQUFDO1lBQzFCLENBQUM7UUFDSCxDQUFDLENBQUMsQ0FBQztRQUVILE9BQU8sT0FBTyxDQUFDO0lBQ2pCLENBQUM7SUFFTyxNQUFNLENBQUMsMEJBQTBCLENBQUMsTUFFekMsRUFDRCxRQUFnQztRQUU5QixJQUFJLE9BQU8sR0FBYyxFQUFFLENBQUM7UUFFNUIsT0FBTyxDQUFDLElBQUksQ0FDVixJQUFJLDJCQUFVLENBQUM7WUFDYixRQUFRLEVBQUUsMENBQTBDO1lBQ3BELE1BQU0sRUFBRSxDQUFDO1lBQ1QsS0FBSyxFQUFFLEVBQUU7U0FDVixDQUFDLENBQ0gsQ0FBQztRQUVGLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsUUFBUSxFQUFFLEVBQUU7WUFDdkMsSUFBSSxrQkFBa0IsR0FBVyxRQUFRLENBQUMsNENBQTRDLENBQUMsUUFBUSxDQUFDLENBQUM7WUFFakcsT0FBTyxDQUFDLElBQUksQ0FDVixJQUFJLGtDQUFpQixDQUFDO2dCQUNwQixNQUFNLEVBQUUsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUM7Z0JBQzFCLE1BQU0sRUFBRSxDQUFDO2dCQUNULEtBQUssRUFBRSxDQUFDO2dCQUNSLEtBQUssRUFBRSxrQkFBa0IsR0FBRyw0QkFBNEI7YUFDekQsQ0FBQyxDQUNILENBQUM7UUFDSixDQUFDLENBQUMsQ0FBQztRQUVILE9BQU8sT0FBTyxDQUFDO0lBQ2pCLENBQUM7SUFJRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWlDO1FBQ3pFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxPQUFPLEdBQWdCLEVBQUUsQ0FBQztRQUU5QixPQUFPLENBQUMsSUFBSSxDQUNWLHFCQUFxQixDQUFDLDBCQUEwQixDQUM5QyxLQUFLLENBQUMsa0NBQWtDLEVBQ3hDLEtBQUssQ0FBQyxRQUFRLENBQ2YsQ0FDRixDQUFDO1FBRUY7Ozs7Ozs7Ozs7O1dBV0c7UUFFSCxJQUFJLEtBQUssQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNmLE9BQU8sQ0FBQyxJQUFJLENBQ1YscUJBQXFCLENBQUMsMkJBQTJCLENBQy9DLEtBQUssQ0FBQyxJQUFJLEVBQ1YsS0FBSyxDQUFDLFFBQVEsRUFDZCxLQUFLLENBQUMsTUFBTSxFQUNaLEtBQUssQ0FBQyxnQkFBZ0IsQ0FDdkIsQ0FDRixDQUFBO1FBQ0gsQ0FBQztRQUVELElBQ0UsNkJBQWEsQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLG1DQUFtQyxDQUFDO1lBQ25FLDZCQUFhLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxnQ0FBZ0MsQ0FBQyxFQUNoRSxDQUFDO1lBQ0QsT0FBTyxDQUFDLElBQUksQ0FDVixxQkFBcUIsQ0FBQyx1QkFBdUIsQ0FDM0MsS0FBSyxDQUFDLG1DQUFtQyxFQUN6QyxLQUFLLENBQUMsZ0NBQWdDLEVBQ3RDLEtBQUssQ0FBQyxRQUFRLENBQ2YsQ0FDRixDQUFDO1FBQ0osQ0FBQztRQUVELElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSwwQkFBUyxDQUFDLElBQUksRUFBRSx1QkFBdUIsRUFBRTtZQUM1RCxhQUFhLEVBQ1gsS0FBSyxDQUFDLFdBQVcsQ0FBQyxXQUFXLEVBQUU7Z0JBQy9CLGdCQUFFLENBQUMsR0FBRyxDQUFDLCtCQUErQixDQUFDO1lBQ3pDLGVBQWUsRUFBRSxLQUFLLENBQUMsUUFBUTtZQUMvQixjQUFjLEVBQUUsK0JBQWMsQ0FBQyxJQUFJO1lBQ25DLE9BQU8sRUFBRSxPQUFPO1NBQ2pCLENBQUMsQ0FBQztJQUNMLENBQUM7Q0FDRjtBQXRVRCxzREFzVUMiLCJzb3VyY2VzQ29udGVudCI6WyIvLyBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbi8vIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG5pbXBvcnQgeyBEdXJhdGlvbiwgRm4gfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQge1xuICBBbGFybVN0YXR1c1dpZGdldCxcbiAgRGFzaGJvYXJkLFxuICBHcmFwaFdpZGdldCxcbiAgSUFsYXJtLFxuICBJTWV0cmljLFxuICBJV2lkZ2V0LFxuICBQZXJpb2RPdmVycmlkZSxcbiAgVGV4dFdpZGdldCxcbn0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWNsb3Vkd2F0Y2gnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBCYXNpY1NlcnZpY2VEYXNoYm9hcmRQcm9wcyB9IGZyb20gJy4vcHJvcHMvQmFzaWNTZXJ2aWNlRGFzaGJvYXJkUHJvcHMnO1xuaW1wb3J0IHsgQXZhaWxhYmlsaXR5Wm9uZU1hcHBlciB9IGZyb20gJy4uL2F6bWFwcGVyL0F2YWlsYWJpbGl0eVpvbmVNYXBwZXInO1xuaW1wb3J0IHsgTWV0cmljc0hlbHBlciB9IGZyb20gJy4uL3V0aWxpdGllcy9NZXRyaWNzSGVscGVyJztcbmltcG9ydCB7IEFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyTWV0cmljcyB9IGZyb20gJy4uL21ldHJpY3MvQXBwbGljYXRpb25Mb2FkQmFsYW5jZXJNZXRyaWNzJztcbmltcG9ydCB7IElBcHBsaWNhdGlvbkxvYWRCYWxhbmNlciB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1lbGFzdGljbG9hZGJhbGFuY2luZ3YyJztcblxuZXhwb3J0IGNsYXNzIEJhc2ljU2VydmljZURhc2hib2FyZCBleHRlbmRzIENvbnN0cnVjdCB7XG5cbiAgcHJpdmF0ZSBzdGF0aWMgZ2VuZXJhdGVMb2FkQmFsYW5jZXJXaWRnZXRzKFxuICAgIGFsYnM6IElBcHBsaWNhdGlvbkxvYWRCYWxhbmNlcltdLFxuICAgIGF6TWFwcGVyOiBBdmFpbGFiaWxpdHlab25lTWFwcGVyLFxuICAgIHBlcmlvZDogRHVyYXRpb24sXG4gICAgbGF0ZW5jeVN0YXRpc3RpYzogc3RyaW5nXG4gICk6IElXaWRnZXRbXSB7XG4gICAgbGV0IGFsYldpZGdldHM6IElXaWRnZXRbXSA9IFtdO1xuXG4gICAgYWxiV2lkZ2V0cy5wdXNoKG5ldyBUZXh0V2lkZ2V0KHsgaGVpZ2h0OiAyLCB3aWR0aDogMjQsIG1hcmtkb3duOiBcIkxvYWQgQmFsYW5jZXIgTWV0cmljc1wiIH0pKTtcblxuICAgIGxldCBzdWNjZXNzQ291bnRQZXJab25lOiB7W2tleTogc3RyaW5nXTogSU1ldHJpY30gPSBcbiAgICAgIEFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyTWV0cmljcy5nZXRUb3RhbEFsYlN1Y2Nlc3NDb3VudFBlclpvbmUoYWxicywgcGVyaW9kLCBhek1hcHBlcik7XG4gICAgbGV0IGZhdWx0Q291bnRQZXJab25lOiB7W2tleTogc3RyaW5nXTogSU1ldHJpY30gPSBcbiAgICAgIEFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyTWV0cmljcy5nZXRUb3RhbEFsYkZhdWx0Q291bnRQZXJab25lKGFsYnMsIHBlcmlvZCwgYXpNYXBwZXIpO1xuICAgIGxldCBwcm9jZXNzZWRCeXRlc1BlclpvbmU6IHtba2V5OiBzdHJpbmddOiBJTWV0cmljfSA9IFxuICAgICAgQXBwbGljYXRpb25Mb2FkQmFsYW5jZXJNZXRyaWNzLmdldFRvdGFsQWxiUHJvY2Vzc2VkQnl0ZXNQZXJab25lKGFsYnMsIHBlcmlvZCwgYXpNYXBwZXIpO1xuICAgIGxldCBsYXRlbmN5UGVyWm9uZToge1trZXk6IHN0cmluZ106IElNZXRyaWN9ID0gXG4gICAgICBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlck1ldHJpY3MuZ2V0VG90YWxBbGJMYXRlbmN5UGVyWm9uZShhbGJzLCBsYXRlbmN5U3RhdGlzdGljLCBwZXJpb2QsIGF6TWFwcGVyKTtcbiAgICBsZXQgcmVxdWVzdHNQZXJab25lOiB7W2tleTogc3RyaW5nXTogSU1ldHJpY30gPSBcbiAgICAgIEFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyTWV0cmljcy5nZXRUb3RhbEFsYlJlcXVlc3RzUGVyWm9uZShhbGJzLCBwZXJpb2QsIGF6TWFwcGVyKTsgIFxuICAgIGxldCBmYXVsdFJhdGVQZXJab25lOiB7W2tleTogc3RyaW5nXTogSU1ldHJpY30gPVxuICAgICAgQXBwbGljYXRpb25Mb2FkQmFsYW5jZXJNZXRyaWNzLmdldFRvdGFsQWxiRmF1bHRSYXRlUGVyWm9uZShhbGJzLCBwZXJpb2QsIGF6TWFwcGVyKTsgIFxuXG4gICAgYWxiV2lkZ2V0cy5wdXNoKFxuICAgICAgbmV3IEdyYXBoV2lkZ2V0KHtcbiAgICAgICAgaGVpZ2h0OiA4LFxuICAgICAgICB3aWR0aDogOCxcbiAgICAgICAgdGl0bGU6IEZuLnN1YignJHtBV1M6OlJlZ2lvbn0gWm9uYWwgU3VjY2VzcyBDb3VudCcpLFxuICAgICAgICByZWdpb246IEZuLnN1YignJHtBV1M6OlJlZ2lvbn0nKSxcbiAgICAgICAgbGVmdDogT2JqZWN0LnZhbHVlcyhzdWNjZXNzQ291bnRQZXJab25lKSxcbiAgICAgICAgbGVmdFlBeGlzOiB7XG4gICAgICAgICAgbWluOiAwLFxuICAgICAgICAgIGxhYmVsOiAnU3VtJyxcbiAgICAgICAgICBzaG93VW5pdHM6IGZhbHNlLFxuICAgICAgICB9XG4gICAgICB9KVxuICAgICk7XG5cbiAgICBhbGJXaWRnZXRzLnB1c2goXG4gICAgICBuZXcgR3JhcGhXaWRnZXQoe1xuICAgICAgICBoZWlnaHQ6IDgsXG4gICAgICAgIHdpZHRoOiA4LFxuICAgICAgICB0aXRsZTogRm4uc3ViKCcke0FXUzo6UmVnaW9ufSBab25hbCBGYXVsdCBDb3VudCcpLFxuICAgICAgICByZWdpb246IEZuLnN1YignJHtBV1M6OlJlZ2lvbn0nKSxcbiAgICAgICAgbGVmdDogT2JqZWN0LnZhbHVlcyhmYXVsdENvdW50UGVyWm9uZSksXG4gICAgICAgIGxlZnRZQXhpczoge1xuICAgICAgICAgIG1pbjogMCxcbiAgICAgICAgICBsYWJlbDogJ1N1bScsXG4gICAgICAgICAgc2hvd1VuaXRzOiBmYWxzZSxcbiAgICAgICAgfVxuICAgICAgfSlcbiAgICApO1xuXG4gICAgYWxiV2lkZ2V0cy5wdXNoKFxuICAgICAgbmV3IEdyYXBoV2lkZ2V0KHtcbiAgICAgICAgaGVpZ2h0OiA4LFxuICAgICAgICB3aWR0aDogOCxcbiAgICAgICAgdGl0bGU6IEZuLnN1YignJHtBV1M6OlJlZ2lvbn0gWm9uYWwgUmVxdWVzdCBDb3VudCcpLFxuICAgICAgICByZWdpb246IEZuLnN1YignJHtBV1M6OlJlZ2lvbn0nKSxcbiAgICAgICAgbGVmdDogT2JqZWN0LnZhbHVlcyhyZXF1ZXN0c1BlclpvbmUpLFxuICAgICAgICBsZWZ0WUF4aXM6IHtcbiAgICAgICAgICBtaW46IDAsXG4gICAgICAgICAgbGFiZWw6ICdTdW0nLFxuICAgICAgICAgIHNob3dVbml0czogZmFsc2UsXG4gICAgICAgIH1cbiAgICAgIH0pXG4gICAgKTtcblxuICAgIGFsYldpZGdldHMucHVzaChcbiAgICAgIG5ldyBHcmFwaFdpZGdldCh7XG4gICAgICAgIGhlaWdodDogOCxcbiAgICAgICAgd2lkdGg6IDgsXG4gICAgICAgIHRpdGxlOiBGbi5zdWIoJyR7QVdTOjpSZWdpb259IFpvbmFsIEZhdWx0IFJhdGUnKSxcbiAgICAgICAgcmVnaW9uOiBGbi5zdWIoJyR7QVdTOjpSZWdpb259JyksXG4gICAgICAgIGxlZnQ6IE9iamVjdC52YWx1ZXMoZmF1bHRSYXRlUGVyWm9uZSksXG4gICAgICAgIGxlZnRZQXhpczoge1xuICAgICAgICAgIG1pbjogMCxcbiAgICAgICAgICBsYWJlbDogJ1BlcmNlbnQnLFxuICAgICAgICAgIHNob3dVbml0czogZmFsc2UsXG4gICAgICAgIH1cbiAgICAgIH0pXG4gICAgKTtcblxuICAgIGFsYldpZGdldHMucHVzaChcbiAgICAgIG5ldyBHcmFwaFdpZGdldCh7XG4gICAgICAgIGhlaWdodDogOCxcbiAgICAgICAgd2lkdGg6IDgsXG4gICAgICAgIHRpdGxlOiBGbi5zdWIoJyR7QVdTOjpSZWdpb259IFpvbmFsIFByb2Nlc3NlZCBCeXRlcycpLFxuICAgICAgICByZWdpb246IEZuLnN1YignJHtBV1M6OlJlZ2lvbn0nKSxcbiAgICAgICAgbGVmdDogT2JqZWN0LnZhbHVlcyhwcm9jZXNzZWRCeXRlc1BlclpvbmUpLFxuICAgICAgICBsZWZ0WUF4aXM6IHtcbiAgICAgICAgICBtaW46IDAsXG4gICAgICAgICAgc2hvd1VuaXRzOiBmYWxzZSxcbiAgICAgICAgICBsYWJlbDogJ0J5dGVzJ1xuICAgICAgICB9XG4gICAgICB9KVxuICAgICk7XG5cbiAgICBhbGJXaWRnZXRzLnB1c2goXG4gICAgICBuZXcgR3JhcGhXaWRnZXQoe1xuICAgICAgICBoZWlnaHQ6IDgsXG4gICAgICAgIHdpZHRoOiA4LFxuICAgICAgICB0aXRsZTogRm4uc3ViKCcke0FXUzo6UmVnaW9ufSBab25hbCBUYXJnZXQgUmVzcG9uc2UgVGltZSAoJyArIGxhdGVuY3lTdGF0aXN0aWMgKyAnKScpLFxuICAgICAgICByZWdpb246IEZuLnN1YignJHtBV1M6OlJlZ2lvbn0nKSxcbiAgICAgICAgbGVmdDogT2JqZWN0LnZhbHVlcyhsYXRlbmN5UGVyWm9uZSksXG4gICAgICAgIGxlZnRZQXhpczoge1xuICAgICAgICAgIG1pbjogMCxcbiAgICAgICAgICBsYWJlbDogXCJNaWxsaXNlY29uZHNcIixcbiAgICAgICAgICBzaG93VW5pdHM6IGZhbHNlLFxuICAgICAgICB9XG4gICAgICB9KVxuICAgICk7XG5cbiAgICByZXR1cm4gYWxiV2lkZ2V0cztcbiAgfVxuXG4gIC8qcHJpdmF0ZSBzdGF0aWMgY3JlYXRlTG9hZEJhbGFuY2VyV2lkZ2V0cyhcbiAgICBhbGFybXM6IHsgW2tleTogc3RyaW5nXTogSUFsYXJtIH0sXG4gICAgbWV0cmljczogeyBba2V5OiBzdHJpbmddOiBJTWV0cmljIH0sXG4gICAgYXpNYXBwZXI6IEF2YWlsYWJpbGl0eVpvbmVNYXBwZXIsXG4gICk6IElXaWRnZXRbXSB7XG4gICAgbGV0IHdpZGdldHM6IElXaWRnZXRbXSA9IFtdO1xuXG4gICAgd2lkZ2V0cy5wdXNoKFxuICAgICAgbmV3IFRleHRXaWRnZXQoe1xuICAgICAgICBtYXJrZG93bjogJ0xvYWQgQmFsYW5jZXIgTWV0cmljcycsXG4gICAgICAgIGhlaWdodDogMixcbiAgICAgICAgd2lkdGg6IDI0LFxuICAgICAgfSksXG4gICAgKTtcblxuICAgIGxldCByb3dUcmFja2VyOiBudW1iZXIgPSAwO1xuXG4gICAgT2JqZWN0LmtleXMoYWxhcm1zKS5mb3JFYWNoKChhekxldHRlciwgaW5kZXgpID0+IHtcbiAgICAgIGxldCBhdmFpbGFiaWxpdHlab25lSWQ6IHN0cmluZyA9IGF6TWFwcGVyLmF2YWlsYWJpbGl0eVpvbmVJZEZyb21BdmFpbGFiaWxpdHlab25lTGV0dGVyKGF6TGV0dGVyKTtcblxuICAgICAgd2lkZ2V0cy5wdXNoKFxuICAgICAgICBuZXcgR3JhcGhXaWRnZXQoe1xuICAgICAgICAgIGhlaWdodDogNixcbiAgICAgICAgICB3aWR0aDogOCxcbiAgICAgICAgICB0aXRsZTogYXZhaWxhYmlsaXR5Wm9uZUlkICsgJyBMb2FkIEJhbGFuY2VyIEZhdWx0cycsXG4gICAgICAgICAgcmVnaW9uOiBGbi5zdWIoJyR7QVdTOjpSZWdpb259JyksXG4gICAgICAgICAgbGVmdDogW21ldHJpY3NbYXpMZXR0ZXJdXSxcbiAgICAgICAgICBzdGF0aXN0aWM6ICdTdW0nLFxuICAgICAgICAgIGxlZnRZQXhpczoge1xuICAgICAgICAgICAgbWluOiAwLFxuICAgICAgICAgICAgbGFiZWw6ICdGYXVsdCBDb3VudCcsXG4gICAgICAgICAgICBzaG93VW5pdHM6IGZhbHNlLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pLFxuICAgICAgKTtcblxuICAgICAgLy9XZSdyZSBvbiB0aGUgdGhpcmQgb25lIGZvciB0aGlzIHNldCwgYWRkIDMgYWxhcm1zXG4gICAgICAvL29yIGlmIHdlJ3JlIGF0IHRoZSBlbmQsIGF0IHRoZSBuZWNlc3NhcnkgYW1vdW50XG4gICAgICAvL29mIGFsYXJtcywgMSwgMiwgb3IgM1xuICAgICAgaWYgKGluZGV4ICUgMyA9PSAyIHx8IGluZGV4IC0gMSA9PSBPYmplY3Qua2V5cyhhbGFybXMpLmxlbmd0aCkge1xuICAgICAgICBmb3IgKGxldCBrID0gcm93VHJhY2tlcjsgayA8PSBpbmRleDsgaysrKSB7XG4gICAgICAgICAgbGV0IGF6SWQ6IHN0cmluZyA9IE9iamVjdC5rZXlzKGFsYXJtcykuYXQoaykhO1xuICAgICAgICAgIHdpZGdldHMucHVzaChcbiAgICAgICAgICAgIG5ldyBBbGFybVN0YXR1c1dpZGdldCh7XG4gICAgICAgICAgICAgIGhlaWdodDogMixcbiAgICAgICAgICAgICAgd2lkdGg6IDgsXG4gICAgICAgICAgICAgIGFsYXJtczogW2FsYXJtc1theklkXV0sXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICApO1xuICAgICAgICB9XG5cbiAgICAgICAgcm93VHJhY2tlciArPSBpbmRleCArIDE7XG4gICAgICB9XG4gICAgfSk7XG5cbiAgICByZXR1cm4gd2lkZ2V0cztcbiAgfSovXG5cbiAgcHJpdmF0ZSBzdGF0aWMgY3JlYXRlTmF0R2F0ZXdheVdpZGdldHMoXG4gICAgYWxhcm1zOiB7IFtrZXk6IHN0cmluZ106IElBbGFybSB9LFxuICAgIG1ldHJpY3M6IHsgW2tleTogc3RyaW5nXTogSU1ldHJpYyB9LFxuICAgIGF6TWFwcGVyOiBBdmFpbGFiaWxpdHlab25lTWFwcGVyLFxuICApOiBJV2lkZ2V0W10ge1xuICAgIGxldCB3aWRnZXRzOiBJV2lkZ2V0W10gPSBbXTtcblxuICAgIHdpZGdldHMucHVzaChcbiAgICAgIG5ldyBUZXh0V2lkZ2V0KHtcbiAgICAgICAgbWFya2Rvd246ICdOQVQgR2F0ZXdheSBEcm9wcGVkIFBhY2tldCBNZXRyaWNzJyxcbiAgICAgICAgaGVpZ2h0OiAyLFxuICAgICAgICB3aWR0aDogMjQsXG4gICAgICB9KSxcbiAgICApO1xuXG4gICAgbGV0IHJvd1RyYWNrZXI6IG51bWJlciA9IDA7XG5cbiAgICBPYmplY3Qua2V5cyhhbGFybXMpLmZvckVhY2goKGF6TGV0dGVyLCBpbmRleCkgPT4ge1xuXG4gICAgICBsZXQgYXZhaWxhYmlsaXR5Wm9uZUlkOiBzdHJpbmcgPSBhek1hcHBlci5hdmFpbGFiaWxpdHlab25lSWRGcm9tQXZhaWxhYmlsaXR5Wm9uZUxldHRlcihhekxldHRlcik7XG5cbiAgICAgIHdpZGdldHMucHVzaChcbiAgICAgICAgbmV3IEdyYXBoV2lkZ2V0KHtcbiAgICAgICAgICBoZWlnaHQ6IDYsXG4gICAgICAgICAgd2lkdGg6IDgsXG4gICAgICAgICAgdGl0bGU6IGF2YWlsYWJpbGl0eVpvbmVJZCArICcgTkFUIEdhdGV3YXkgRHJvcHBlZCBQYWNrZXRzJyxcbiAgICAgICAgICByZWdpb246IEZuLnN1YignJHtBV1M6OlJlZ2lvbn0nKSxcbiAgICAgICAgICBsZWZ0OiBbbWV0cmljc1thekxldHRlcl1dLFxuICAgICAgICAgIHN0YXRpc3RpYzogJ1N1bScsXG4gICAgICAgICAgbGVmdFlBeGlzOiB7XG4gICAgICAgICAgICBtaW46IDAsXG4gICAgICAgICAgICBsYWJlbDogJ0Ryb3BwZWQgcGFja2V0cycsXG4gICAgICAgICAgICBzaG93VW5pdHM6IGZhbHNlLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pLFxuICAgICAgKTtcblxuICAgICAgLy9XZSdyZSBvbiB0aGUgdGhpcmQgb25lIGZvciB0aGlzIHNldCwgYWRkIDMgYWxhcm1zXG4gICAgICAvL29yIGlmIHdlJ3JlIGF0IHRoZSBlbmQsIGF0IHRoZSBuZWNlc3NhcnkgYW1vdW50XG4gICAgICAvL29mIGFsYXJtcywgMSwgMiwgb3IgM1xuICAgICAgaWYgKGluZGV4ICUgMyA9PSAyIHx8IGluZGV4IC0gMSA9PSBPYmplY3Qua2V5cyhhbGFybXMpLmxlbmd0aCkge1xuICAgICAgICBmb3IgKGxldCBrID0gcm93VHJhY2tlcjsgayA8PSBpbmRleDsgaysrKSB7XG4gICAgICAgICAgbGV0IGF6SWQ6IHN0cmluZyA9IE9iamVjdC5rZXlzKGFsYXJtcykuYXQoaykhO1xuICAgICAgICAgIHdpZGdldHMucHVzaChcbiAgICAgICAgICAgIG5ldyBBbGFybVN0YXR1c1dpZGdldCh7XG4gICAgICAgICAgICAgIGhlaWdodDogMixcbiAgICAgICAgICAgICAgd2lkdGg6IDgsXG4gICAgICAgICAgICAgIGFsYXJtczogW2FsYXJtc1theklkXV0sXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICApO1xuICAgICAgICB9XG5cbiAgICAgICAgcm93VHJhY2tlciArPSBpbmRleCArIDE7XG4gICAgICB9XG4gICAgfSk7XG5cbiAgICByZXR1cm4gd2lkZ2V0cztcbiAgfVxuXG4gIHByaXZhdGUgc3RhdGljIGNyZWF0ZVRvcExldmVsQWxhcm1XaWRnZXRzKGFsYXJtczoge1xuICAgIFtrZXk6IHN0cmluZ106IElBbGFybTtcbiAgfSxcbiAgYXpNYXBwZXI6IEF2YWlsYWJpbGl0eVpvbmVNYXBwZXIsXG4gICk6IElXaWRnZXRbXSB7XG4gICAgbGV0IHdpZGdldHM6IElXaWRnZXRbXSA9IFtdO1xuXG4gICAgd2lkZ2V0cy5wdXNoKFxuICAgICAgbmV3IFRleHRXaWRnZXQoe1xuICAgICAgICBtYXJrZG93bjogJ0F2YWlsYWJpbGl0eSBab25lIElzb2xhdGVkIEltcGFjdCBBbGFybXMnLFxuICAgICAgICBoZWlnaHQ6IDIsXG4gICAgICAgIHdpZHRoOiAyNCxcbiAgICAgIH0pLFxuICAgICk7XG5cbiAgICBPYmplY3Qua2V5cyhhbGFybXMpLmZvckVhY2goKGF6TGV0dGVyKSA9PiB7XG4gICAgICBsZXQgYXZhaWxhYmlsaXR5Wm9uZUlkOiBzdHJpbmcgPSBhek1hcHBlci5hdmFpbGFiaWxpdHlab25lSWRGcm9tQXZhaWxhYmlsaXR5Wm9uZUxldHRlcihhekxldHRlcik7XG5cbiAgICAgIHdpZGdldHMucHVzaChcbiAgICAgICAgbmV3IEFsYXJtU3RhdHVzV2lkZ2V0KHtcbiAgICAgICAgICBhbGFybXM6IFthbGFybXNbYXpMZXR0ZXJdXSxcbiAgICAgICAgICBoZWlnaHQ6IDIsXG4gICAgICAgICAgd2lkdGg6IDgsXG4gICAgICAgICAgdGl0bGU6IGF2YWlsYWJpbGl0eVpvbmVJZCArICcgQWdncmVnYXRlIElzb2xhdGVkIEltcGFjdCcsXG4gICAgICAgIH0pLFxuICAgICAgKTtcbiAgICB9KTtcblxuICAgIHJldHVybiB3aWRnZXRzO1xuICB9XG5cbiAgZGFzaGJvYXJkOiBEYXNoYm9hcmQ7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEJhc2ljU2VydmljZURhc2hib2FyZFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGxldCB3aWRnZXRzOiBJV2lkZ2V0W11bXSA9IFtdO1xuXG4gICAgd2lkZ2V0cy5wdXNoKFxuICAgICAgQmFzaWNTZXJ2aWNlRGFzaGJvYXJkLmNyZWF0ZVRvcExldmVsQWxhcm1XaWRnZXRzKFxuICAgICAgICBwcm9wcy56b25hbEFnZ3JlZ2F0ZUlzb2xhdGVkSW1wYWN0QWxhcm1zLFxuICAgICAgICBwcm9wcy5hek1hcHBlcixcbiAgICAgICksXG4gICAgKTtcblxuICAgIC8qaWYgKFxuICAgICAgTWV0cmljc0hlbHBlci5pc05vdEVtcHR5KHByb3BzLnpvbmFsTG9hZEJhbGFuY2VySXNvbGF0ZWRJbXBhY3RBbGFybXMpICYmXG4gICAgICBNZXRyaWNzSGVscGVyLmlzTm90RW1wdHkocHJvcHMuem9uYWxMb2FkQmFsYW5jZXJGYXVsdFJhdGVNZXRyaWNzKVxuICAgICkge1xuICAgICAgd2lkZ2V0cy5wdXNoKFxuICAgICAgICBCYXNpY1NlcnZpY2VEYXNoYm9hcmQuY3JlYXRlTG9hZEJhbGFuY2VyV2lkZ2V0cyhcbiAgICAgICAgICBwcm9wcy56b25hbExvYWRCYWxhbmNlcklzb2xhdGVkSW1wYWN0QWxhcm1zLFxuICAgICAgICAgIHByb3BzLnpvbmFsTG9hZEJhbGFuY2VyRmF1bHRSYXRlTWV0cmljcyxcbiAgICAgICAgICBwcm9wcy5hek1hcHBlcixcbiAgICAgICAgKSxcbiAgICAgICk7XG4gICAgfSovXG5cbiAgICBpZiAocHJvcHMuYWxicykge1xuICAgICAgd2lkZ2V0cy5wdXNoKFxuICAgICAgICBCYXNpY1NlcnZpY2VEYXNoYm9hcmQuZ2VuZXJhdGVMb2FkQmFsYW5jZXJXaWRnZXRzKFxuICAgICAgICAgIHByb3BzLmFsYnMsXG4gICAgICAgICAgcHJvcHMuYXpNYXBwZXIsXG4gICAgICAgICAgcHJvcHMucGVyaW9kLFxuICAgICAgICAgIHByb3BzLmxhdGVuY3lTdGF0aXN0aWNcbiAgICAgICAgKVxuICAgICAgKVxuICAgIH1cblxuICAgIGlmIChcbiAgICAgIE1ldHJpY3NIZWxwZXIuaXNOb3RFbXB0eShwcm9wcy56b25hbE5hdEdhdGV3YXlJc29sYXRlZEltcGFjdEFsYXJtcykgJiZcbiAgICAgIE1ldHJpY3NIZWxwZXIuaXNOb3RFbXB0eShwcm9wcy56b25hbE5hdEdhdGV3YXlQYWNrZXREcm9wTWV0cmljcylcbiAgICApIHtcbiAgICAgIHdpZGdldHMucHVzaChcbiAgICAgICAgQmFzaWNTZXJ2aWNlRGFzaGJvYXJkLmNyZWF0ZU5hdEdhdGV3YXlXaWRnZXRzKFxuICAgICAgICAgIHByb3BzLnpvbmFsTmF0R2F0ZXdheUlzb2xhdGVkSW1wYWN0QWxhcm1zLFxuICAgICAgICAgIHByb3BzLnpvbmFsTmF0R2F0ZXdheVBhY2tldERyb3BNZXRyaWNzLFxuICAgICAgICAgIHByb3BzLmF6TWFwcGVyLFxuICAgICAgICApLFxuICAgICAgKTtcbiAgICB9XG5cbiAgICB0aGlzLmRhc2hib2FyZCA9IG5ldyBEYXNoYm9hcmQodGhpcywgJ0Jhc2ljU2VydmljZURhc2hib2FyZCcsIHtcbiAgICAgIGRhc2hib2FyZE5hbWU6XG4gICAgICAgIHByb3BzLnNlcnZpY2VOYW1lLnRvTG93ZXJDYXNlKCkgK1xuICAgICAgICBGbi5zdWIoJy1wZXItYXotaGVhbHRoLSR7QVdTOjpSZWdpb259JyksXG4gICAgICBkZWZhdWx0SW50ZXJ2YWw6IHByb3BzLmludGVydmFsLFxuICAgICAgcGVyaW9kT3ZlcnJpZGU6IFBlcmlvZE92ZXJyaWRlLkFVVE8sXG4gICAgICB3aWRnZXRzOiB3aWRnZXRzLFxuICAgIH0pO1xuICB9XG59XG4iXX0=