"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AvailabilityZoneMapper = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const fs = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
const MetricsHelper_1 = require("../utilities/MetricsHelper");
/**
 * A construct that allows you to map AZ names to ids and back
 */
class AvailabilityZoneMapper extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /*
        let currentNode: Construct | undefined = this.node.scope;
    
        while (currentNode !== undefined && !(currentNode instanceof Stack)) {
          currentNode = currentNode.node.scope;
        }
    
        if (currentNode !== undefined) {
          (currentNode as Stack).addTransform('AWS::LanguageExtensions');
        }
        */
        let xrayManagedPolicy = new aws_iam_1.ManagedPolicy(this, 'XrayManagedPolicy', {
            path: '/azmapper/',
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: [
                        'xray:PutTraceSegments',
                        'xray:PutTelemetryRecords',
                        'xray:GetSamplingRules',
                        'xray:GetSamplingTargets',
                        'xray:GetSamplingStatisticSummaries',
                    ],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: ['*'],
                }),
            ],
        });
        let ec2ManagedPolicy = new aws_iam_1.ManagedPolicy(this, 'EC2ManagedPolicy', {
            path: '/azmapper/',
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: ['ec2:DescribeAvailabilityZones'],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: ['*'],
                }),
            ],
        });
        let executionRole = new aws_iam_1.Role(this, 'executionRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            path: '/azmapper/',
            managedPolicies: [xrayManagedPolicy, ec2ManagedPolicy],
        });
        const file = fs.readFileSync(path.resolve(__dirname, './../azmapper/src/index.py'), 'utf-8');
        this.function = new aws_lambda_1.Function(this, 'AvailabilityZoneMapperFunction', {
            runtime: MetricsHelper_1.MetricsHelper.PythonRuntime,
            code: aws_lambda_1.Code.fromInline(file),
            handler: 'index.handler',
            role: executionRole,
            architecture: aws_lambda_1.Architecture.ARM_64,
            tracing: aws_lambda_1.Tracing.ACTIVE,
            timeout: aws_cdk_lib_1.Duration.seconds(20),
            memorySize: 512,
            environment: {
                REGION: aws_cdk_lib_1.Fn.ref('AWS::Region'),
                PARTITION: aws_cdk_lib_1.Fn.ref('AWS::Partition'),
            },
        });
        this.logGroup = new aws_logs_1.LogGroup(this, 'LogGroup', {
            logGroupName: `/aws/lambda/${this.function.functionName}`,
            retention: aws_logs_1.RetentionDays.ONE_DAY,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        new aws_iam_1.ManagedPolicy(this, 'CloudWatchManagedPolicy', {
            path: '/azmapper/',
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: ['cloudwatch:PutMetricData'],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: ['*'],
                }),
                new aws_iam_1.PolicyStatement({
                    actions: ['logs:CreateLogStream', 'logs:PutLogEvents'],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: [
                        aws_cdk_lib_1.Fn.sub('arn:${AWS::Partition}:logs:${AWS::Region}:${AWS::AccountId}:log-group:') +
                            this.logGroup.logGroupName +
                            ':*',
                    ],
                }),
            ],
            roles: [executionRole],
        });
        this.mapper = new aws_cdk_lib_1.CustomResource(this, 'AvailabilityZoneMapper', {
            serviceToken: this.function.functionArn,
            properties: props?.availabilityZoneNames !== undefined
                ? { AvailabilityZones: props.availabilityZoneNames }
                : {},
        });
    }
    /**
     * Gets the Availability Zone Id for the given Availability Zone Name in this account
     * @param availabilityZoneName
     * @returns
     */
    availabilityZoneId(availabilityZoneName) {
        return this.mapper.getAttString(availabilityZoneName);
    }
    /**
     * Gets the Availability Zone Name for the given Availability Zone Id in this account
     * @param availabilityZoneId
     * @returns
     */
    availabilityZoneName(availabilityZoneId) {
        return this.mapper.getAttString(availabilityZoneId);
    }
    /**
     * Gets the prefix for the region used with Availability Zone Ids, for example
     * in us-east-1, this returns "use1"
     * @returns
     */
    regionPrefixForAvailabilityZoneIds() {
        return this.mapper.getAttString(aws_cdk_lib_1.Fn.ref('AWS::Region'));
    }
    /**
     * Returns an array for Availability Zone Ids for the supplied Availability Zone names,
     * they are returned in the same order the names were provided
     * @param availabilityZoneNames
     * @returns
     */
    availabilityZoneIdsAsArray(availabilityZoneNames) {
        let ids = [];
        for (let i = 0; i < availabilityZoneNames.length; i++) {
            ids.push(this.availabilityZoneId(availabilityZoneNames[i]));
        }
        return ids;
    }
    /**
     * Returns a comma delimited list of Availability Zone Ids for the supplied
     * Availability Zone names. You can use this string with Fn.Select(x, Fn.Split(",", azs)) to
     * get a specific Availability Zone Id
     * @param availabilityZoneNames
     * @returns
     */
    availabilityZoneIdsAsCommaDelimitedList(availabilityZoneNames) {
        let ids = [];
        for (let i = 0; i < availabilityZoneNames.length; i++) {
            ids.push(this.availabilityZoneId(availabilityZoneNames[i]));
        }
        return ids.join(',');
    }
    /**
     * Returns a comma delimited list of Availability Zone Ids for the supplied
     * Availability Zone names. You can use this string with Fn.Select(x, Fn.Split(",", azs)) to
     * get a specific Availability Zone Id
     * @returns
     */
    allAvailabilityZoneIdsAsCommaDelimitedList() {
        return this.mapper.getAttString('AllAvailabilityZoneIds');
    }
    /**
     * Returns a reference that can be cast to a string array with all of the
     * Availability Zone Ids
     * @returns
     */
    allAvailabilityZoneIdsAsArray() {
        return this.mapper.getAtt('AllAvailabilityZoneIdsArray');
    }
    /**
     * Given a letter like "f" or "a", returns the Availability Zone Id for that
     * Availability Zone name in this account
     * @param letter
     * @returns
     */
    availabilityZoneIdFromAvailabilityZoneLetter(letter) {
        return this.mapper.getAttString(letter);
    }
    /**
     * Gets all of the Availability Zone names in this Region as a comma delimited
     * list. You can use this string with Fn.Select(x, Fn.Split(",", azs)) to
     * get a specific Availability Zone Name
     * @returns
     */
    allAvailabilityZoneNamesAsCommaDelimitedList() {
        return this.mapper.getAttString('AllAvailabilityZoneNames');
    }
}
exports.AvailabilityZoneMapper = AvailabilityZoneMapper;
_a = JSII_RTTI_SYMBOL_1;
AvailabilityZoneMapper[_a] = { fqn: "@cdklabs/multi-az-observability.AvailabilityZoneMapper", version: "0.0.1-alpha.14" };
//# sourceMappingURL=data:application/json;base64,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