"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceAlarmsAndRules = void 0;
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
const RegionalAvailabilityMetrics_1 = require("../metrics/RegionalAvailabilityMetrics");
const AvailabilityMetricType_1 = require("../utilities/AvailabilityMetricType");
const MetricsHelper_1 = require("../utilities/MetricsHelper");
/**
 * Service level alarms and rules using critical operations
 */
class ServiceAlarmsAndRules extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.service = props.service;
        let criticalOperations = props.service.operations
            .filter((x) => x.critical == true)
            .map((x) => x.operationName);
        let counter = 1;
        this.zonalAggregateIsolatedImpactAlarms = [];
        this.zonalServerSideIsolatedImpactAlarms = [];
        let availabilityZoneIds = props.service.availabilityZoneNames.map((x) => {
            return props.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(x.substring(x.length - 1));
        });
        for (let i = 0; i < availabilityZoneIds.length; i++) {
            let availabilityZonedId = availabilityZoneIds[i];
            this.zonalAggregateIsolatedImpactAlarms.push(new aws_cloudwatch_1.CompositeAlarm(this, 'AZ' + counter + 'ServiceAggregateIsolatedImpactAlarm', {
                compositeAlarmName: availabilityZonedId +
                    '-' +
                    props.service.serviceName.toLowerCase() +
                    '-isolated-impact-aggregate-alarm',
                alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...Object.values(Object.entries(props.perOperationAlarmsAndRules).reduce((filtered, [key, value]) => {
                    if (criticalOperations.indexOf(key) > -1) {
                        filtered[key] = value;
                    }
                    return filtered;
                }, {})).map((x) => x.aggregateZonalAlarms[i])),
            }));
            this.zonalServerSideIsolatedImpactAlarms.push(new aws_cloudwatch_1.CompositeAlarm(this, 'AZ' + counter + 'ServiceServerSideIsolatedImpactAlarm', {
                compositeAlarmName: availabilityZonedId +
                    '-' +
                    props.service.serviceName.toLowerCase() +
                    '-isolated-impact-server-side-alarm',
                alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...Object.values(Object.entries(props.perOperationAlarmsAndRules).reduce((filtered, [key, value]) => {
                    if (criticalOperations.indexOf(key) > -1) {
                        filtered[key] = value;
                    }
                    return filtered;
                }, {})).map((x) => x.serverSideZonalAlarmsMap[availabilityZonedId])),
            }));
            counter++;
        }
        let keyPrefix = '';
        let regionalOperationFaultCountMetrics = {};
        props.service.operations
            .filter((x) => x.critical == true)
            .forEach((x) => {
            keyPrefix = MetricsHelper_1.MetricsHelper.nextChar(keyPrefix);
            regionalOperationFaultCountMetrics[keyPrefix] =
                RegionalAvailabilityMetrics_1.RegionalAvailabilityMetrics.createRegionalAvailabilityMetric({
                    label: x.operationName + ' fault count',
                    metricDetails: x.serverSideAvailabilityMetricDetails,
                    metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
                });
        });
        let regionalFaultCount = new aws_cloudwatch_1.MathExpression({
            usingMetrics: regionalOperationFaultCountMetrics,
            expression: Object.keys(regionalOperationFaultCountMetrics).join('+'),
            label: props.service.serviceName + ' fault count',
            period: props.service.period,
        });
        this.regionalFaultCountServerSideAlarm = new aws_cloudwatch_1.Alarm(this, 'RegionalFaultCount', {
            alarmName: aws_cdk_lib_1.Fn.ref('AWS::Region') +
                '-' +
                props.service.serviceName.toLowerCase() +
                '-fault-count',
            datapointsToAlarm: 3,
            evaluationPeriods: 5,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            threshold: props.service.faultCountThreshold,
            alarmDescription: 'Counts faults from all critical operation in the service',
            metric: regionalFaultCount,
        });
        let canaryAlarms = Object.values(Object.entries(props.perOperationAlarmsAndRules).reduce((filtered, [key, value]) => {
            if (criticalOperations.indexOf(key) > -1) {
                filtered[key] = value;
            }
            return filtered;
        }, {}))
            .reduce((filtered, value) => {
            if (value.canaryRegionalAlarmsAndRules !== undefined) {
                filtered.push(value.canaryRegionalAlarmsAndRules);
            }
            return filtered;
        }, [])
            .map((x) => x.availabilityOrLatencyAlarm);
        if (canaryAlarms !== undefined &&
            canaryAlarms !== null &&
            canaryAlarms.length > 0) {
            this.regionalAvailabilityOrLatencyCanaryAlarm = new aws_cloudwatch_1.CompositeAlarm(this, 'ServiceCanaryAvailabilityOrLatencyAggregateAlarm', {
                compositeAlarmName: aws_cdk_lib_1.Fn.ref('AWS::Region') +
                    '-' +
                    props.service.serviceName.toLowerCase() +
                    '-canary-availability-or-latency-aggregate-alarm',
                alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...canaryAlarms),
            });
        }
        let canaryAvailabilityAlarms = Object.values(Object.entries(props.perOperationAlarmsAndRules).reduce((filtered, [key, value]) => {
            if (criticalOperations.indexOf(key) > -1) {
                filtered[key] = value;
            }
            return filtered;
        }, {}))
            .reduce((filtered, value) => {
            if (value.canaryRegionalAlarmsAndRules !== undefined) {
                filtered.push(value.canaryRegionalAlarmsAndRules);
            }
            return filtered;
        }, [])
            .map((x) => x.availabilityAlarm);
        if (canaryAvailabilityAlarms !== undefined &&
            canaryAvailabilityAlarms !== null &&
            canaryAvailabilityAlarms.length > 0) {
            this.regionalAvailabilityCanaryAlarm = new aws_cloudwatch_1.CompositeAlarm(this, 'ServiceCanaryAvailabilityAggregateAlarm', {
                compositeAlarmName: aws_cdk_lib_1.Fn.ref('AWS::Region') +
                    '-' +
                    props.service.serviceName.toLowerCase() +
                    '-canary-availability-aggregate-alarm',
                alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...canaryAvailabilityAlarms),
            });
        }
        this.regionalAvailabilityOrLatencyServerSideAlarm = new aws_cloudwatch_1.CompositeAlarm(this, 'ServiceServerSideAggregateIsolatedImpactAlarm', {
            compositeAlarmName: aws_cdk_lib_1.Fn.ref('AWS::Region') +
                '-' +
                props.service.serviceName.toLowerCase() +
                '-server-side-aggregate-alarm',
            alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...Object.values(Object.entries(props.perOperationAlarmsAndRules).reduce((filtered, [key, value]) => {
                if (criticalOperations.indexOf(key) > -1) {
                    filtered[key] = value;
                }
                return filtered;
            }, {}))
                .map((x) => x.serverSideRegionalAlarmsAndRules)
                .map((x) => x.availabilityOrLatencyAlarm)),
        });
        this.regionalAvailabilityServerSideAlarm = new aws_cloudwatch_1.CompositeAlarm(this, 'ServiceServerSideAvailabilityAlarm', {
            compositeAlarmName: aws_cdk_lib_1.Fn.ref('AWS::Region') +
                '-' +
                props.service.serviceName.toLowerCase() +
                '-server-side-availability-alarm',
            alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...Object.values(Object.entries(props.perOperationAlarmsAndRules).reduce((filtered, [key, value]) => {
                if (criticalOperations.indexOf(key) > -1) {
                    filtered[key] = value;
                }
                return filtered;
            }, {}))
                .map((x) => x.serverSideRegionalAlarmsAndRules)
                .map((x) => x.availabilityAlarm)),
        });
    }
}
exports.ServiceAlarmsAndRules = ServiceAlarmsAndRules;
//# sourceMappingURL=data:application/json;base64,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