"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerSideOperationZonalAlarmsAndRules = void 0;
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const AvailabilityAndLatencyAlarmsAndRules_1 = require("./AvailabilityAndLatencyAlarmsAndRules");
const BaseOperationZonalAlarmsAndRules_1 = require("./BaseOperationZonalAlarmsAndRules");
const OutlierDetectionAlgorithm_1 = require("../utilities/OutlierDetectionAlgorithm");
/**
 * The server side alarms and rules for an operation in an Availability Zone
 */
class ServerSideOperationZonalAlarmsAndRules extends BaseOperationZonalAlarmsAndRules_1.BaseOperationZonalAlarmsAndRules {
    constructor(scope, id, props) {
        super(scope, id, props);
        if (props.outlierDetectionAlgorithm == OutlierDetectionAlgorithm_1.OutlierDetectionAlgorithm.STATIC) {
            this.availabilityZoneIsOutlierForFaults =
                AvailabilityAndLatencyAlarmsAndRules_1.AvailabilityAndLatencyAlarmsAndRules.createZonalFaultRateStaticOutlierAlarm(this, props.availabilityMetricDetails, props.availabilityZoneId, props.counter, props.outlierThreshold, props.nameSuffix);
            this.availabilityZoneIsOutlierForLatency =
                AvailabilityAndLatencyAlarmsAndRules_1.AvailabilityAndLatencyAlarmsAndRules.createZonalHighLatencyStaticOutlierAlarm(this, props.latencyMetricDetails, props.availabilityZoneId, props.counter, props.outlierThreshold, props.nameSuffix);
        }
        else {
            this.availabilityZoneIsOutlierForFaults =
                AvailabilityAndLatencyAlarmsAndRules_1.AvailabilityAndLatencyAlarmsAndRules.createZonalFaultRateOutlierAlarm(this, props.availabilityMetricDetails, props.availabilityZoneId, props.operation.service.availabilityZoneNames.map((az) => {
                    return props.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(az.substring(az.length - 1));
                }), props.outlierThreshold, props.outlierDetectionFunction, props.outlierDetectionAlgorithm, props.counter, props.nameSuffix);
            this.availabilityZoneIsOutlierForLatency =
                AvailabilityAndLatencyAlarmsAndRules_1.AvailabilityAndLatencyAlarmsAndRules.createZonalHighLatencyOutlierAlarm(this, props.latencyMetricDetails, props.availabilityZoneId, props.operation.service.availabilityZoneNames.map((az) => {
                    return props.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(az.substring(az.length - 1));
                }), props.outlierThreshold, props.outlierDetectionFunction, props.outlierDetectionAlgorithm, props.counter, props.nameSuffix);
        }
        if (props.contributorInsightRuleDetails !== undefined &&
            props.contributorInsightRuleDetails != null) {
            this.instancesHandlingRequestsInThisAZ =
                AvailabilityAndLatencyAlarmsAndRules_1.AvailabilityAndLatencyAlarmsAndRules.createServerSideInstancesHandlingRequestsInThisAZRule(this, props.availabilityMetricDetails.operationName, props.availabilityZoneId, props.contributorInsightRuleDetails, props.counter, props.nameSuffix);
            this.instanceContributorsToFaultsInThisAZ =
                AvailabilityAndLatencyAlarmsAndRules_1.AvailabilityAndLatencyAlarmsAndRules.createServerSideInstanceFaultContributorsInThisAZRule(this, props.availabilityMetricDetails.operationName, props.availabilityZoneId, props.contributorInsightRuleDetails, props.counter, props.nameSuffix);
            this.multipleInstancesProducingFaultsInThisAvailabilityZone =
                AvailabilityAndLatencyAlarmsAndRules_1.AvailabilityAndLatencyAlarmsAndRules.createServerSideZonalMoreThanOneInstanceProducingFaultsAlarm(this, props.availabilityMetricDetails, props.availabilityZoneId, props.counter, props.outlierThreshold, this.instanceContributorsToFaultsInThisAZ, this.instancesHandlingRequestsInThisAZ, props.nameSuffix);
            this.instanceContributorsToHighLatencyInThisAZ =
                AvailabilityAndLatencyAlarmsAndRules_1.AvailabilityAndLatencyAlarmsAndRules.createServerSideInstanceHighLatencyContributorsInThisAZRule(this, props.latencyMetricDetails, props.availabilityZoneId, props.contributorInsightRuleDetails, props.counter, props.nameSuffix);
            this.multipleInstancesProducingHighLatencyInThisAZ =
                AvailabilityAndLatencyAlarmsAndRules_1.AvailabilityAndLatencyAlarmsAndRules.createServerSideZonalMoreThanOneInstanceProducingHighLatencyAlarm(this, props.latencyMetricDetails, props.availabilityZoneId, props.counter, props.outlierThreshold, this.instanceContributorsToHighLatencyInThisAZ, this.instancesHandlingRequestsInThisAZ, props.nameSuffix);
            this.isolatedImpactAlarm =
                AvailabilityAndLatencyAlarmsAndRules_1.AvailabilityAndLatencyAlarmsAndRules.createServerSideIsolatedAZImpactAlarm(this, props.availabilityMetricDetails.operationName, props.availabilityZoneId, props.counter, this.availabilityZoneIsOutlierForFaults, this.availabilityAlarm, this.multipleInstancesProducingFaultsInThisAvailabilityZone, this.availabilityZoneIsOutlierForLatency, this.latencyAlarm, this.multipleInstancesProducingHighLatencyInThisAZ, props.nameSuffix);
        }
        else {
            this.isolatedImpactAlarm = new aws_cloudwatch_1.CompositeAlarm(scope, props.operation.operationName +
                'AZ' +
                props.counter +
                'IsolatedImpactAlarm' +
                props.nameSuffix, {
                compositeAlarmName: props.availabilityZoneId +
                    `-${props.operation.operationName.toLowerCase()}-isolated-impact-alarm` +
                    props.nameSuffix,
                alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(aws_cloudwatch_1.AlarmRule.allOf(this.availabilityZoneIsOutlierForFaults, this.availabilityAlarm), aws_cloudwatch_1.AlarmRule.allOf(this.availabilityZoneIsOutlierForLatency, this.latencyAlarm)),
                actionsEnabled: false,
            });
        }
    }
}
exports.ServerSideOperationZonalAlarmsAndRules = ServerSideOperationZonalAlarmsAndRules;
//# sourceMappingURL=data:application/json;base64,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