"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OperationAlarmsAndRules = void 0;
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
const CanaryOperationRegionalAlarmsAndRules_1 = require("./CanaryOperationRegionalAlarmsAndRules");
const CanaryOperationZonalAlarmsAndRules_1 = require("./CanaryOperationZonalAlarmsAndRules");
const ServerSideOperationRegionalAlarmsAndRules_1 = require("./ServerSideOperationRegionalAlarmsAndRules");
const ServerSideOperationZonalAlarmsAndRules_1 = require("./ServerSideOperationZonalAlarmsAndRules");
/**
 * Creates alarms and rules for an operation for both regional and zonal metrics
 */
class OperationAlarmsAndRules extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.serverSideZonalAlarmsAndRules = [];
        this.canaryZonalAlarmsAndRules = [];
        this.aggregateZonalAlarms = [];
        this.operation = props.operation;
        this.aggregateZonalAlarmsMap = {};
        this.serverSideZonalAlarmsMap = {};
        let availabilityZoneIds = props.operation.service.availabilityZoneNames.map((x) => {
            return props.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(x.substring(x.length - 1));
        });
        let loadBalancerArn = '';
        if (props.loadBalancer !== undefined) {
            loadBalancerArn = props.loadBalancer
                .loadBalancerArn;
        }
        this.serverSideRegionalAlarmsAndRules =
            new ServerSideOperationRegionalAlarmsAndRules_1.ServerSideOperationRegionalAlarmsAndRules(this, props.operation.operationName + 'ServerSideRegionalAlarms', {
                availabilityMetricDetails: props.operation.serverSideAvailabilityMetricDetails,
                latencyMetricDetails: props.operation.serverSideLatencyMetricDetails,
                contributorInsightRuleDetails: props.operation
                    .serverSideContributorInsightRuleDetails
                    ? props.operation.serverSideContributorInsightRuleDetails
                    : props.operation.service.defaultContributorInsightRuleDetails,
                nameSuffix: '-server',
            });
        if (props.operation.canaryMetricDetails !== undefined &&
            props.operation.canaryMetricDetails != null) {
            this.canaryRegionalAlarmsAndRules =
                new CanaryOperationRegionalAlarmsAndRules_1.CanaryOperationRegionalAlarmsAndRules(this, props.operation.operationName + 'CanaryRegionalAlarms', {
                    availabilityMetricDetails: props.operation.canaryMetricDetails
                        .canaryAvailabilityMetricDetails,
                    latencyMetricDetails: props.operation.canaryMetricDetails.canaryLatencyMetricDetails,
                    nameSuffix: '-canary',
                });
            this.aggregateRegionalAlarm = new aws_cloudwatch_1.CompositeAlarm(this, props.operation.operationName + 'AggregateRegionalAlarm', {
                actionsEnabled: false,
                compositeAlarmName: aws_cdk_lib_1.Fn.ref('AWS::Region') +
                    '-' +
                    props.operation.operationName.toLowerCase() +
                    '-' +
                    'aggregate-alarm',
                alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(this.serverSideRegionalAlarmsAndRules.availabilityOrLatencyAlarm, this.canaryRegionalAlarmsAndRules.availabilityOrLatencyAlarm),
            });
        }
        else {
            this.aggregateRegionalAlarm =
                this.serverSideRegionalAlarmsAndRules.availabilityOrLatencyAlarm;
        }
        let counter = 1;
        for (let i = 0; i < availabilityZoneIds.length; i++) {
            let availabilityZoneId = availabilityZoneIds[i];
            this.serverSideZonalAlarmsAndRules.push(new ServerSideOperationZonalAlarmsAndRules_1.ServerSideOperationZonalAlarmsAndRules(this, props.operation.operationName +
                'AZ' +
                counter +
                'ServerSideZonalAlarmsAndRules', {
                availabilityZoneId: availabilityZoneId,
                availabilityMetricDetails: props.operation.serverSideAvailabilityMetricDetails,
                latencyMetricDetails: props.operation.serverSideLatencyMetricDetails,
                contributorInsightRuleDetails: props.operation
                    .serverSideContributorInsightRuleDetails
                    ? props.operation.serverSideContributorInsightRuleDetails
                    : props.operation.service.defaultContributorInsightRuleDetails,
                counter: counter,
                outlierThreshold: props.outlierThreshold,
                outlierDetectionAlgorithm: props.outlierDetectionAlgorithm,
                nameSuffix: '-server',
                operation: props.operation,
                azMapper: props.azMapper,
                outlierDetectionFunction: props.outlierDetectionFunction,
            }));
            this.serverSideZonalAlarmsMap[availabilityZoneId] =
                this.serverSideZonalAlarmsAndRules[i].isolatedImpactAlarm;
            if (props.operation.canaryMetricDetails !== undefined &&
                props.operation.canaryMetricDetails != null) {
                this.canaryZonalAlarmsAndRules.push(new CanaryOperationZonalAlarmsAndRules_1.CanaryOperationZonalAlarmsAndRules(this, props.operation.operationName +
                    'AZ' +
                    counter +
                    'CanaryZonalAlarmsAndRules', {
                    availabilityZoneId: availabilityZoneId,
                    availabilityMetricDetails: props.operation.canaryMetricDetails
                        .canaryAvailabilityMetricDetails,
                    latencyMetricDetails: props.operation.canaryMetricDetails.canaryLatencyMetricDetails,
                    counter: counter,
                    outlierThreshold: props.outlierThreshold,
                    outlierDetectionAlgorithm: props.outlierDetectionAlgorithm,
                    nameSuffix: '-canary',
                    operation: props.operation,
                    azMapper: props.azMapper,
                    outlierDetectionFunction: props.outlierDetectionFunction,
                }));
                this.aggregateZonalAlarms.push(new aws_cloudwatch_1.CompositeAlarm(this, props.operation.operationName +
                    'AZ' +
                    counter +
                    'AggregateZonalIsolatedImpactAlarm', {
                    compositeAlarmName: availabilityZoneId +
                        '-' +
                        props.operation.operationName.toLowerCase() +
                        '-aggregate-isolated-az-impact',
                    alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(this.canaryZonalAlarmsAndRules[i].isolatedImpactAlarm, this.serverSideZonalAlarmsAndRules[i].isolatedImpactAlarm),
                    actionsEnabled: false,
                    alarmDescription: '{"loadBalancer":"' +
                        loadBalancerArn +
                        '","az-id":"' +
                        availabilityZoneId +
                        '"}',
                }));
            }
            else {
                this.aggregateZonalAlarms.push(this.serverSideZonalAlarmsAndRules[i].isolatedImpactAlarm);
            }
            this.aggregateZonalAlarmsMap[availabilityZoneId] =
                this.aggregateZonalAlarms[-1];
            counter++;
        }
    }
}
exports.OperationAlarmsAndRules = OperationAlarmsAndRules;
//# sourceMappingURL=data:application/json;base64,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