"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AvailabilityAndLatencyAlarmsAndRules = void 0;
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const InsightRuleBody_1 = require("./InsightRuleBody");
const RegionalAvailabilityMetrics_1 = require("../metrics/RegionalAvailabilityMetrics");
const RegionalLatencyMetrics_1 = require("../metrics/RegionalLatencyMetrics");
const ZonalAvailabilityMetrics_1 = require("../metrics/ZonalAvailabilityMetrics");
const ZonalLatencyMetrics_1 = require("../metrics/ZonalLatencyMetrics");
const AvailabilityMetricType_1 = require("../utilities/AvailabilityMetricType");
const LatencyMetricType_1 = require("../utilities/LatencyMetricType");
const MetricsHelper_1 = require("../utilities/MetricsHelper");
/**
 * Class used to create availability and latency alarms and Contributor Insight rules
 */
class AvailabilityAndLatencyAlarmsAndRules {
    /**
     * Creates a zonal availability alarm
     * @param scope
     * @param metricDetails
     * @param availabilityZoneId
     * @param nameSuffix
     * @param counter
     * @returns
     */
    static createZonalAvailabilityAlarm(scope, metricDetails, availabilityZoneId, counter, nameSuffix) {
        return new aws_cloudwatch_1.Alarm(scope, metricDetails.operationName + 'AZ' + counter + 'AvailabilityAlarm', {
            alarmName: availabilityZoneId +
                '-' +
                metricDetails.operationName.toLowerCase() +
                '-success-rate' +
                nameSuffix,
            evaluationPeriods: metricDetails.evaluationPeriods,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            threshold: metricDetails.successAlarmThreshold,
            actionsEnabled: false,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE,
            metric: ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalAvailabilityMetric({
                availabilityZoneId: availabilityZoneId,
                label: availabilityZoneId + ' availability',
                metricDetails: metricDetails,
                metricType: AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE,
            }),
        });
    }
    /**
     * Creates a zonal latency alarm
     * @param scope
     * @param metricDetails
     * @param availabilityZoneId
     * @param nameSuffix
     * @param counter
     * @returns
     */
    static createZonalLatencyAlarm(scope, metricDetails, availabilityZoneId, counter, nameSuffix) {
        return new aws_cloudwatch_1.Alarm(scope, metricDetails.operationName + 'AZ' + counter + 'LatencyAlarm', {
            alarmName: availabilityZoneId +
                '-' +
                metricDetails.operationName.toLowerCase() +
                '-success-latency' +
                nameSuffix,
            evaluationPeriods: metricDetails.evaluationPeriods,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            threshold: metricDetails.successAlarmThreshold,
            actionsEnabled: false,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE,
            metric: ZonalLatencyMetrics_1.ZonalLatencyMetrics.createZonalAverageLatencyMetric({
                availabilityZoneId: availabilityZoneId,
                label: availabilityZoneId +
                    ' ' +
                    metricDetails.alarmStatistic +
                    ' latency',
                metricDetails: metricDetails,
                metricType: LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY,
                statistic: metricDetails.alarmStatistic,
            }),
        });
    }
    /**
     * Creates a composite alarm when either latency or availability is breached in the Availabiltiy Zone
     * @param scope
     * @param operation
     * @param availabilityZoneId
     * @param nameSuffix
     * @param counter
     * @param zonalAvailabilityAlarm
     * @param zonalLatencyAlarm
     * @returns
     */
    static createZonalAvailabilityOrLatencyCompositeAlarm(scope, operationName, availabilityZoneId, counter, zonalAvailabilityAlarm, zonalLatencyAlarm, nameSuffix) {
        return new aws_cloudwatch_1.CompositeAlarm(scope, 'AZ' + counter + 'ZonalImpactAlarm', {
            actionsEnabled: false,
            alarmDescription: availabilityZoneId +
                ' has latency or availability impact. This does not indicate it is an outlier and shows isolated impact.',
            compositeAlarmName: availabilityZoneId +
                `-${operationName.toLowerCase()}-impact-aggregate-alarm` +
                nameSuffix,
            alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(zonalAvailabilityAlarm, zonalLatencyAlarm),
        });
    }
    /**
     * An alarm that compares error rate in this AZ to the overall region error based only on metric data
     * @param scope
     * @param metricDetails
     * @param availabilityZoneId
     * @param nameSuffix
     * @param counter
     * @param outlierThreshold
     * @returns
     */
    static createZonalFaultRateStaticOutlierAlarm(scope, metricDetails, availabilityZoneId, counter, outlierThreshold, nameSuffix) {
        let zonalFaults = ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalAvailabilityMetric({
            availabilityZoneId: availabilityZoneId,
            metricDetails: metricDetails,
            metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
            keyPrefix: 'a',
        });
        // For server-side, this metric is ok, because the AZ specific metrics are dual-reported
        // with dimensions for both the AZ and Region. But for the canary, it produces 1 metric
        // for testing the regional endpoint and a separate metric for each AZ. So in that case
        // we don't want to divide by the regional metric, use per-AZ
        let regionalFaults = RegionalAvailabilityMetrics_1.RegionalAvailabilityMetrics.createRegionalAvailabilityMetric({
            metricDetails: metricDetails,
            metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
            keyPrefix: 'b',
        });
        return new aws_cloudwatch_1.Alarm(scope, 'AZ' + counter + 'IsolatedImpactAlarmStatic', {
            alarmName: availabilityZoneId +
                `-${metricDetails.operationName.toLowerCase()}-static-majority-errors-impact` +
                nameSuffix,
            metric: new aws_cloudwatch_1.MathExpression({
                expression: 'IF(m2 > 0, (m1 / m2), 0)',
                usingMetrics: {
                    m1: zonalFaults,
                    m2: regionalFaults,
                },
                period: metricDetails.period,
                label: availabilityZoneId + ' percent faults',
            }),
            threshold: outlierThreshold,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            evaluationPeriods: metricDetails.evaluationPeriods,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
        });
    }
    /**
   * An alarm that compares error rate in this AZ to the overall region error based only on metric data.
   * This is different for canaries because the metrics they test at the regional level are different
   * requests than the ones sent to the zonal endpoints. So you have to add all of the zonal requests together
   * to compare one AZ to the others (you can't compare a zone to the regional metrics).
   * @param scope
   * @param metricDetails
   * @param availabilityZoneId
   * @param nameSuffix
   * @param counter
   * @param outlierThreshold
   * @returns
   */
    static createZonalFaultRateStaticOutlierAlarmForCanaries(scope, metricDetails, availabilityZoneId, availabilityZones, counter, outlierThreshold, nameSuffix) {
        let zonalFaults = ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalAvailabilityMetric({
            availabilityZoneId: availabilityZoneId,
            metricDetails: metricDetails,
            metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
            keyPrefix: 'a',
        });
        let prefix = 'b';
        let usingMetrics = {};
        availabilityZones.forEach((az) => {
            prefix = MetricsHelper_1.MetricsHelper.nextChar(prefix);
            let azFaults = ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalAvailabilityMetric({
                availabilityZoneId: az,
                metricDetails: metricDetails,
                metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
                keyPrefix: prefix
            });
            prefix = MetricsHelper_1.MetricsHelper.nextChar(prefix);
            usingMetrics[`${prefix}1`] = azFaults;
        });
        prefix = MetricsHelper_1.MetricsHelper.nextChar(prefix);
        let regionalFaultCount = new aws_cloudwatch_1.MathExpression({
            expression: Object.keys(usingMetrics).join("+"),
            usingMetrics: usingMetrics,
            period: metricDetails.period
        });
        return new aws_cloudwatch_1.Alarm(scope, 'AZ' + counter + 'IsolatedImpactAlarmStatic', {
            alarmName: availabilityZoneId +
                `-${metricDetails.operationName.toLowerCase()}-static-majority-errors-impact` +
                nameSuffix,
            metric: new aws_cloudwatch_1.MathExpression({
                expression: 'IF(m2 > 0, (m1 / m2), 0)',
                usingMetrics: {
                    m1: zonalFaults,
                    m2: regionalFaultCount,
                },
                period: metricDetails.period,
                label: availabilityZoneId + ' percent faults',
            }),
            threshold: outlierThreshold,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            evaluationPeriods: metricDetails.evaluationPeriods,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
        });
    }
    static createZonalFaultRateOutlierAlarm(scope, metricDetails, availabilityZoneId, allAvailabilityZoneIds, outlierThreshold, outlierDetectionFunction, outlierDetectionAlgorithm, counter, nameSuffix) {
        let metricDimensions = {};
        allAvailabilityZoneIds.forEach((azId) => {
            metricDimensions[azId] = [
                metricDetails.metricDimensions.zonalDimensions(azId, aws_cdk_lib_1.Fn.ref('AWS::Region')),
            ];
        });
        let str = JSON.stringify(metricDimensions)
            .replace(/[\\]/g, '\\\\')
            .replace(/[\"]/g, '\\"')
            .replace(/[\/]/g, '\\/')
            .replace(/[\b]/g, '\\b')
            .replace(/[\f]/g, '\\f')
            .replace(/[\n]/g, '\\n')
            .replace(/[\r]/g, '\\r')
            .replace(/[\t]/g, '\\t');
        let outlierMetrics = new aws_cloudwatch_1.MathExpression({
            expression: `MAX(LAMBDA("${outlierDetectionFunction.functionName}",` +
                `"${outlierDetectionAlgorithm.toString()}",` +
                `"${outlierThreshold}",` +
                `"${availabilityZoneId}",` +
                `"${str}",` +
                `"${metricDetails.metricNamespace}",` +
                `"${metricDetails.faultMetricNames.join(':')}",` +
                '"Sum",' +
                '"Count"' +
                '))',
            period: metricDetails.period
        });
        return new aws_cloudwatch_1.Alarm(scope, 'AZ' + counter + 'FaultIsolatedImpactAlarmOutlier', {
            alarmName: availabilityZoneId +
                `-${metricDetails.operationName.toLowerCase()}-majority-errors-impact` +
                nameSuffix,
            metric: outlierMetrics,
            threshold: 1,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            evaluationPeriods: metricDetails.evaluationPeriods,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
        });
    }
    static createZonalFaultRateOutlierAlarmForAlb(scope, loadBalancers, availabilityZoneId, outlierThreshold, outlierDetectionFunction, outlierDetectionAlgorithm, azMapper, counter, evaluationPeriods, datapointsToAlarm, period, nameSuffix) {
        let metricDimensions = {};
        loadBalancers.forEach((x) => {
            x.vpc?.availabilityZones.forEach((az) => {
                let azId = azMapper.availabilityZoneIdFromAvailabilityZoneLetter(az.substring(az.length - 1));
                if (!(azId in metricDimensions)) {
                    metricDimensions[azId] = [];
                }
                metricDimensions[azId].push({
                    AvailabilityZone: az,
                    LoadBalancer: x
                        .loadBalancerFullName,
                });
            });
        });
        let str = JSON.stringify(metricDimensions)
            .replace(/[\\]/g, '\\\\')
            .replace(/[\"]/g, '\\"')
            .replace(/[\/]/g, '\\/')
            .replace(/[\b]/g, '\\b')
            .replace(/[\f]/g, '\\f')
            .replace(/[\n]/g, '\\n')
            .replace(/[\r]/g, '\\r')
            .replace(/[\t]/g, '\\t');
        let outlierMetrics = new aws_cloudwatch_1.MathExpression({
            expression: `MAX(LAMBDA("${outlierDetectionFunction.functionName}",` +
                `"${outlierDetectionAlgorithm.toString()}",` +
                `"${outlierThreshold}",` +
                `"${availabilityZoneId}",` +
                `"${str}",` +
                '"AWS/ApplicationELB",' +
                '"HTTPCode_ELB_5XX_Count:HTTPCode_Target_5XX_Count",' +
                '"Sum",' +
                '"Count"' +
                '))',
            period: period
        });
        return new aws_cloudwatch_1.Alarm(scope, 'AZ' + counter + 'AlbIsolatedImpactAlarmOutlier', {
            alarmName: availabilityZoneId + '-alb-majority-errors-impact' + nameSuffix,
            metric: outlierMetrics,
            threshold: 1,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            evaluationPeriods: evaluationPeriods,
            datapointsToAlarm: datapointsToAlarm,
        });
    }
    static createZonalFaultRateOutlierAlarmForNatGW(scope, natGateways, availabilityZoneId, outlierThreshold, outlierDetectionFunction, outlierDetectionAlgorithm, azMapper, counter, evaluationPeriods, datapointsToAlarm, period, nameSuffix) {
        let metricDimensions = {};
        Object.keys(natGateways).forEach((az) => {
            let azId = azMapper.availabilityZoneIdFromAvailabilityZoneLetter(az.substring(az.length - 1));
            if (!(azId in metricDimensions)) {
                metricDimensions[azId] = [];
            }
            natGateways[az].forEach((natgw) => {
                metricDimensions[azId].push({
                    NatGatewayId: natgw.attrNatGatewayId,
                });
            });
        });
        let str = JSON.stringify(metricDimensions)
            .replace(/[\\]/g, '\\\\')
            .replace(/[\"]/g, '\\"')
            .replace(/[\/]/g, '\\/')
            .replace(/[\b]/g, '\\b')
            .replace(/[\f]/g, '\\f')
            .replace(/[\n]/g, '\\n')
            .replace(/[\r]/g, '\\r')
            .replace(/[\t]/g, '\\t');
        let outlierMetrics = new aws_cloudwatch_1.MathExpression({
            expression: `MAX(LAMBDA("${outlierDetectionFunction.functionName}",` +
                `"${outlierDetectionAlgorithm.toString()}",` +
                `"${outlierThreshold}",` +
                `"${availabilityZoneId}",` +
                `"${str}",` +
                '"AWS/NATGateway",' +
                '"PacketsDropCount",' +
                '"Sum",' +
                '"Count"' +
                '))',
            period: period
        });
        return new aws_cloudwatch_1.Alarm(scope, 'AZ' + counter + 'NatGWIsolatedImpactAlarmOutlier', {
            alarmName: availabilityZoneId + '-nat-gw-majority-errors-impact' + nameSuffix,
            metric: outlierMetrics,
            threshold: 1,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            evaluationPeriods: evaluationPeriods,
            datapointsToAlarm: datapointsToAlarm,
        });
    }
    static createZonalHighLatencyOutlierAlarm(scope, metricDetails, availabilityZoneId, allAvailabilityZoneIds, outlierThreshold, outlierDetectionFunction, outlierDetectionAlgorithm, counter, nameSuffix) {
        let metricDimensions = {};
        allAvailabilityZoneIds.forEach((azId) => {
            metricDimensions[azId] = [
                metricDetails.metricDimensions.zonalDimensions(azId, aws_cdk_lib_1.Fn.ref('AWS::Region')),
            ];
        });
        let str = JSON.stringify(metricDimensions)
            .replace(/[\\]/g, '\\\\')
            .replace(/[\"]/g, '\\"')
            .replace(/[\/]/g, '\\/')
            .replace(/[\b]/g, '\\b')
            .replace(/[\f]/g, '\\f')
            .replace(/[\n]/g, '\\n')
            .replace(/[\r]/g, '\\r')
            .replace(/[\t]/g, '\\t');
        let outlierMetrics = new aws_cloudwatch_1.MathExpression({
            expression: `MAX(LAMBDA("${outlierDetectionFunction.functionName}",` +
                `"${outlierDetectionAlgorithm.toString()}",` +
                `"${outlierThreshold}",` +
                `"${availabilityZoneId}",` +
                `"${str}",` +
                `"${metricDetails.metricNamespace}",` +
                `"${metricDetails.successMetricNames.join(':')}",` +
                `"TC(${metricDetails.successAlarmThreshold}:)",` +
                '"Milliseconds"' +
                '))',
            period: metricDetails.period
        });
        return new aws_cloudwatch_1.Alarm(scope, metricDetails.operationName +
            'AZ' +
            counter +
            'LatencyIsolatedImpactAlarmOutlier', {
            alarmName: availabilityZoneId +
                `-${metricDetails.operationName.toLowerCase()}-majority-high-latency-impact` +
                nameSuffix,
            metric: outlierMetrics,
            threshold: 1,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            evaluationPeriods: metricDetails.evaluationPeriods,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
        });
    }
    static createZonalHighLatencyStaticOutlierAlarm(scope, metricDetails, availabilityZoneId, counter, outlierThreshold, nameSuffix) {
        let zonalLatency = ZonalLatencyMetrics_1.ZonalLatencyMetrics.createZonalCountLatencyMetric({
            availabilityZoneId: availabilityZoneId,
            label: availabilityZoneId +
                '-' +
                metricDetails.operationName +
                '-high-latency-requests',
            metricDetails: metricDetails,
            metricType: LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY,
            statistic: `TC(${metricDetails.successAlarmThreshold}:)`,
            keyPrefix: 'a',
        });
        let regionalLatency = RegionalLatencyMetrics_1.RegionalLatencyMetrics.createRegionalLatencyCountMetric({
            label: aws_cdk_lib_1.Fn.ref('AWS::Region') +
                '-' +
                metricDetails.operationName +
                '-high-latency-requests',
            metricDetails: metricDetails,
            metricType: LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY,
            statistic: `TC(${metricDetails.successAlarmThreshold}:)`,
            keyPrefix: 'b',
        });
        return new aws_cloudwatch_1.Alarm(scope, metricDetails.operationName +
            'AZ' +
            counter +
            'IsolatedImpactAlarmStatic', {
            alarmName: availabilityZoneId +
                `-${metricDetails.operationName.toLowerCase()}-static-majority-high-latency-impact` +
                nameSuffix,
            metric: new aws_cloudwatch_1.MathExpression({
                expression: 'IF(m2 > 0, (m1 / m2), 0)',
                usingMetrics: {
                    m1: zonalLatency,
                    m2: regionalLatency,
                },
                period: metricDetails.period,
                label: availabilityZoneId + ' percent high latency requests',
            }),
            threshold: outlierThreshold,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            evaluationPeriods: metricDetails.evaluationPeriods,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
        });
    }
    static createZonalHighLatencyStaticOutlierAlarmForCanaries(scope, metricDetails, availabilityZoneId, availabilityZones, counter, outlierThreshold, nameSuffix) {
        let zonalLatency = ZonalLatencyMetrics_1.ZonalLatencyMetrics.createZonalCountLatencyMetric({
            availabilityZoneId: availabilityZoneId,
            label: availabilityZoneId +
                '-' +
                metricDetails.operationName +
                '-high-latency-requests',
            metricDetails: metricDetails,
            metricType: LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY,
            statistic: `TC(${metricDetails.successAlarmThreshold}:)`,
            keyPrefix: 'a',
        });
        let prefix = 'b';
        let usingMetrics = {};
        availabilityZones.forEach((az) => {
            prefix = MetricsHelper_1.MetricsHelper.nextChar(prefix);
            let azLatencyMetrics = ZonalLatencyMetrics_1.ZonalLatencyMetrics.createZonalLatencyMetrics({
                availabilityZoneId: az,
                metricDetails: metricDetails,
                metricType: LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY,
                keyPrefix: prefix,
                statistic: `TC(${metricDetails.successAlarmThreshold}:)`
            });
            prefix = MetricsHelper_1.MetricsHelper.nextChar(prefix);
            let innerUsingMetrics = {};
            azLatencyMetrics.forEach((metric, index) => {
                innerUsingMetrics[`${prefix}${index}`] = metric;
            });
            let azLatencyCount = new aws_cloudwatch_1.MathExpression({
                expression: Object.keys(innerUsingMetrics).join("+"),
                usingMetrics: innerUsingMetrics,
                period: metricDetails.period
            });
            prefix = MetricsHelper_1.MetricsHelper.nextChar(prefix);
            usingMetrics[`${prefix}1`] = azLatencyCount;
        });
        prefix = MetricsHelper_1.MetricsHelper.nextChar(prefix);
        let regionalLatency = new aws_cloudwatch_1.MathExpression({
            expression: Object.keys(usingMetrics).join("+"),
            usingMetrics: usingMetrics,
            period: metricDetails.period
        });
        return new aws_cloudwatch_1.Alarm(scope, metricDetails.operationName +
            'AZ' +
            counter +
            'IsolatedImpactAlarmStatic', {
            alarmName: availabilityZoneId +
                `-${metricDetails.operationName.toLowerCase()}-static-majority-high-latency-impact` +
                nameSuffix,
            metric: new aws_cloudwatch_1.MathExpression({
                expression: 'IF(m2 > 0, (m1 / m2), 0)',
                usingMetrics: {
                    m1: zonalLatency,
                    m2: regionalLatency,
                },
                period: metricDetails.period,
                label: availabilityZoneId + ' percent high latency requests',
            }),
            threshold: outlierThreshold,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            evaluationPeriods: metricDetails.evaluationPeriods,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
        });
    }
    /**
     * An insight rule that calculates how many instances are responding to requests in
     * the specified AZ. Only useful for server-side metrics since the canary doesn't record instance id metrics.
     * @param scope
     * @param metricDetails
     * @param availabilityZoneId
     * @param logGroups
     * @param nameSuffix
     * @param counter
     * @param instanceIdPath
     * @param operationNamePath
     * @param availabilityZoneIdPath
     * @returns
     */
    static createServerSideInstancesHandlingRequestsInThisAZRule(scope, operationName, availabilityZoneId, ruleDetails, counter, nameSuffix) {
        let ruleBody = new InsightRuleBody_1.InsightRuleBody();
        ruleBody.logGroupNames = ruleDetails.logGroups.map((x) => x.logGroupName);
        ruleBody.aggregateOn = 'Count';
        ruleBody.logFormat = 'JSON';
        ruleBody.contribution = {
            keys: [ruleDetails.instanceIdJsonPath],
            filters: [
                {
                    Match: ruleDetails.availabilityZoneIdJsonPath,
                    In: [availabilityZoneId],
                },
                {
                    Match: ruleDetails.operationNameJsonPath,
                    In: [operationName],
                },
            ],
        };
        return new aws_cloudwatch_1.CfnInsightRule(scope, 'AZ' + counter + 'InstancesInTheAZRule', {
            ruleName: availabilityZoneId +
                `-${operationName.toLowerCase()}-instances-in-the-az` +
                nameSuffix,
            ruleState: 'ENABLED',
            ruleBody: ruleBody.toJson(),
        });
    }
    /**
     * An insight rule that calculates the instances contributing to errors
     * in this AZ. Only useful for server-side metrics since the canary doesn't record instance id metrics.
     * @param scope
     * @param operation
     * @param availabilityZoneId
     * @param logGroups
     * @param nameSuffix
     * @param counter
     * @param instanceIdPath
     * @param operationNamePath
     * @param availabilityZoneIdPath
     * @param errorMetricPath
     * @returns
     */
    static createServerSideInstanceFaultContributorsInThisAZRule(scope, operationName, availabilityZoneId, ruleDetails, counter, nameSuffix) {
        let ruleBody = new InsightRuleBody_1.InsightRuleBody();
        ruleBody.logGroupNames = ruleDetails.logGroups.map((x) => x.logGroupName);
        ruleBody.aggregateOn = 'Count';
        ruleBody.logFormat = 'JSON';
        ruleBody.contribution = {
            keys: [ruleDetails.instanceIdJsonPath],
            filters: [
                {
                    Match: ruleDetails.availabilityZoneIdJsonPath,
                    In: [availabilityZoneId],
                },
                {
                    Match: ruleDetails.operationNameJsonPath,
                    In: [operationName],
                },
                {
                    Match: ruleDetails.faultMetricJsonPath,
                    GreaterThan: 0,
                },
            ],
        };
        return new aws_cloudwatch_1.CfnInsightRule(scope, 'AZ' + counter + 'InstanceErrorContributionRule', {
            ruleName: availabilityZoneId +
                `-${operationName.toLowerCase()}-per-instance-faults` +
                nameSuffix,
            ruleState: 'ENABLED',
            ruleBody: ruleBody.toJson(),
        });
    }
    /**
     * An insight rule that calculates instances contributing to high latency in this AZ. Only
     * useful for server-side metrics since the canary doesn't record instance id metrics.
     * @param scope
     * @param metricDetails
     * @param availabilityZoneId
     * @param logGroups
     * @param nameSuffix
     * @param counter
     * @returns
     */
    static createServerSideInstanceHighLatencyContributorsInThisAZRule(scope, metricDetails, availabilityZoneId, ruleDetails, counter, nameSuffix) {
        let ruleBody = new InsightRuleBody_1.InsightRuleBody();
        ruleBody.logGroupNames = ruleDetails.logGroups.map((x) => x.logGroupName);
        ruleBody.aggregateOn = 'Count';
        ruleBody.logFormat = 'JSON';
        ruleBody.contribution = {
            keys: [ruleDetails.instanceIdJsonPath],
            filters: [
                {
                    Match: ruleDetails.availabilityZoneIdJsonPath,
                    In: [availabilityZoneId],
                },
                {
                    Match: ruleDetails.operationNameJsonPath,
                    In: [metricDetails.operationName],
                },
                {
                    Match: ruleDetails.successLatencyMetricJsonPath,
                    GreaterThan: metricDetails.successAlarmThreshold,
                },
            ],
        };
        return new aws_cloudwatch_1.CfnInsightRule(scope, 'AZ' + counter + 'LatencyContributorsRule', {
            ruleName: availabilityZoneId +
                `-${metricDetails.operationName.toLowerCase()}-per-instance-high-latency` +
                nameSuffix,
            ruleState: 'ENABLED',
            ruleBody: ruleBody.toJson(),
        });
    }
    /**
     * An alarm that indicates some percentage of the instances in this AZ are producing errors. Only
     * useful for server-side metrics since the canary doesn't record instance id metrics.
     * @param scope
     * @param metricDetails
     * @param availabilityZoneId
     * @param nameSuffix
     * @param counter
     * @param outlierThreshold
     * @param instanceFaultRateContributorsInThisAZ
     * @param instancesHandlingRequestsInThisAZ
     * @returns
     */
    static createServerSideZonalMoreThanOneInstanceProducingFaultsAlarm(scope, metricDetails, availabilityZoneId, counter, outlierThreshold, instanceFaultRateContributorsInThisAZ, instancesHandlingRequestsInThisAZ, nameSuffix) {
        return new aws_cloudwatch_1.Alarm(scope, 'AZ' + counter + 'MoreThanOneAlarmForErrors', {
            alarmName: availabilityZoneId +
                `-${metricDetails.operationName.toLowerCase()}-multiple-instances-faults` +
                nameSuffix,
            metric: new aws_cloudwatch_1.MathExpression({
                expression: `INSIGHT_RULE_METRIC(\"${instanceFaultRateContributorsInThisAZ.attrRuleName}\", \"UniqueContributors\") / INSIGHT_RULE_METRIC(\"${instancesHandlingRequestsInThisAZ.attrRuleName}\", \"UniqueContributors\")`,
                period: metricDetails.period,
            }),
            evaluationPeriods: metricDetails.evaluationPeriods,
            threshold: outlierThreshold,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
            actionsEnabled: false,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE,
        });
    }
    /**
     * An alarm indicating more than some percentage of instances in this AZ
     * are contributing to high latency. Only useful for server-side metrics since
     * the canary doesn't record instance id metrics.
     * @param scope
     * @param metricDetails
     * @param availabilityZoneId
     * @param nameSuffix
     * @param counter
     * @param outlierThreshold
     * @param instanceHighLatencyContributorsInThisAZ
     * @param instancesHandlingRequestsInThisAZ
     * @returns
     */
    static createServerSideZonalMoreThanOneInstanceProducingHighLatencyAlarm(scope, metricDetails, availabilityZoneId, counter, outlierThreshold, instanceHighLatencyContributorsInThisAZ, instancesHandlingRequestsInThisAZ, nameSuffix) {
        return new aws_cloudwatch_1.Alarm(scope, 'AZ' + counter + 'MoreThanOneAlarmForHighLatency', {
            alarmName: availabilityZoneId +
                `-${metricDetails.operationName.toLowerCase()}-multiple-instances-high-latency` +
                nameSuffix,
            metric: new aws_cloudwatch_1.MathExpression({
                expression: `INSIGHT_RULE_METRIC(\"${instanceHighLatencyContributorsInThisAZ.attrRuleName}\", \"UniqueContributors\") / INSIGHT_RULE_METRIC(\"${instancesHandlingRequestsInThisAZ.attrRuleName}\", \"UniqueContributors\")`,
                period: metricDetails.period,
            }),
            evaluationPeriods: metricDetails.evaluationPeriods,
            threshold: outlierThreshold,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
            actionsEnabled: false,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE,
        });
    }
    /**
     * An alarm that indicates this AZ as an outlier
     * for availability or latency. This does not ensure that the errors
     * or latency originate from more than one instance.
     * @param scope
     * @param operation
     * @param availabilityZoneId
     * @param logGroups
     * @param nameSuffix
     * @param counter
     * @param azIsOutlierForFaultsAlarm
     * @param availabilityImpactAlarm
     * @param azIsOutlierForLatencyAlarm
     * @param latencyImpactAlarm
     * @returns
     */
    static createCanaryIsolatedAZImpactAlarm(scope, operationName, availabilityZoneId, counter, azIsOutlierForFaultsAlarm, availabilityImpactAlarm, azIsOutlierForLatencyAlarm, latencyImpactAlarm, nameSuffix) {
        return new aws_cloudwatch_1.CompositeAlarm(scope, operationName + 'AZ' + counter + 'IsolatedImpactAlarm' + nameSuffix, {
            compositeAlarmName: availabilityZoneId +
                `-${operationName.toLowerCase()}-isolated-impact-alarm` +
                nameSuffix,
            alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(aws_cloudwatch_1.AlarmRule.allOf(azIsOutlierForFaultsAlarm, availabilityImpactAlarm), aws_cloudwatch_1.AlarmRule.allOf(azIsOutlierForLatencyAlarm, latencyImpactAlarm)),
            actionsEnabled: false,
        });
    }
    /**
     * Creates the server side alarm to identify isolated single AZ
     * impact meaning that this one AZ is affected and the others aren't
     * @param scope
     * @param operation
     * @param availabilityZoneId
     * @param nameSuffix
     * @param counter
     * @param azIsOutlierForFaultsAlarm
     * @param availabilityImpactAlarm
     * @param moreThanOneInstanceContributingToFaults
     * @param azIsOutlierForLatencyAlarm
     * @param latencyImpactAlarm
     * @param moreThanOneInstanceContributingToLatency
     * @returns
     */
    static createServerSideIsolatedAZImpactAlarm(scope, operationName, availabilityZoneId, counter, azIsOutlierForFaultsAlarm, availabilityImpactAlarm, moreThanOneInstanceContributingToFaults, azIsOutlierForLatencyAlarm, latencyImpactAlarm, moreThanOneInstanceContributingToLatency, nameSuffix) {
        return new aws_cloudwatch_1.CompositeAlarm(scope, operationName + 'AZ' + counter + 'IsolatedImpactAlarm' + nameSuffix, {
            compositeAlarmName: availabilityZoneId +
                `-${operationName.toLowerCase()}-isolated-impact-alarm` +
                nameSuffix,
            alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(moreThanOneInstanceContributingToFaults === undefined ||
                moreThanOneInstanceContributingToFaults == null
                ? aws_cloudwatch_1.AlarmRule.allOf(azIsOutlierForFaultsAlarm, availabilityImpactAlarm)
                : aws_cloudwatch_1.AlarmRule.allOf(azIsOutlierForFaultsAlarm, availabilityImpactAlarm, moreThanOneInstanceContributingToFaults), moreThanOneInstanceContributingToLatency === undefined ||
                moreThanOneInstanceContributingToLatency == null
                ? aws_cloudwatch_1.AlarmRule.allOf(azIsOutlierForLatencyAlarm, latencyImpactAlarm)
                : aws_cloudwatch_1.AlarmRule.allOf(azIsOutlierForLatencyAlarm, latencyImpactAlarm, moreThanOneInstanceContributingToLatency)),
            actionsEnabled: false,
        });
    }
    /**
     * Creates an alarm that fires if either the canary or the
     * server side detect single AZ isolated impact
     * @param scope
     * @param operation
     * @param availabilityZoneId
     * @param counter
     * @param serverSideAlarm
     * @param canaryAlarm
     * @returns
     */
    static createAggregateIsolatedAZImpactAlarm(scope, operation, availabilityZoneId, counter, serverSideAlarm, canaryAlarm) {
        return new aws_cloudwatch_1.CompositeAlarm(scope, operation.operationName + 'AZ' + counter + 'AggregateIsolatedImpactAlarm', {
            compositeAlarmName: availabilityZoneId +
                `-${operation.operationName.toLowerCase()}-aggregate-isolated-impact-alarm`,
            alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(serverSideAlarm, canaryAlarm),
            actionsEnabled: false,
        });
    }
    /**
     * Creates a regional availability alarm for the operation
     * @param scope
     * @param metricDetails
     * @param nameSuffix
     * @param counter
     * @returns
     */
    static createRegionalAvailabilityAlarm(scope, metricDetails, nameSuffix) {
        return new aws_cloudwatch_1.Alarm(scope, metricDetails.operationName + 'RegionalAvailabilityAlarm', {
            alarmName: aws_cdk_lib_1.Fn.ref('AWS::Region') +
                '-' +
                metricDetails.operationName.toLowerCase() +
                '-success-rate' +
                nameSuffix,
            evaluationPeriods: metricDetails.evaluationPeriods,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            threshold: metricDetails.successAlarmThreshold,
            actionsEnabled: false,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE,
            metric: RegionalAvailabilityMetrics_1.RegionalAvailabilityMetrics.createRegionalAvailabilityMetric({
                label: aws_cdk_lib_1.Fn.ref('AWS::Region') + ' availability',
                metricDetails: metricDetails,
                metricType: AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE,
            }),
        });
    }
    /**
     * Creates a regional latency alarm for the operation
     * @param scope
     * @param metricDetails
     * @param nameSuffix
     * @param counter
     * @returns
     */
    static createRegionalLatencyAlarm(scope, metricDetails, nameSuffix) {
        return new aws_cloudwatch_1.Alarm(scope, metricDetails.operationName + 'RegionalLatencyAlarm', {
            alarmName: aws_cdk_lib_1.Fn.ref('AWS::Region') +
                '-' +
                metricDetails.operationName.toLowerCase() +
                '-success-latency' +
                nameSuffix,
            evaluationPeriods: metricDetails.evaluationPeriods,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            threshold: metricDetails.successAlarmThreshold,
            actionsEnabled: false,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE,
            metric: RegionalLatencyMetrics_1.RegionalLatencyMetrics.createRegionalAverageLatencyMetric({
                label: aws_cdk_lib_1.Fn.ref('AWS::Region') +
                    ' ' +
                    metricDetails.alarmStatistic +
                    ' latency',
                metricDetails: metricDetails,
                metricType: LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY,
                statistic: metricDetails.alarmStatistic,
            }),
        });
    }
    /**
     * A composite alarm combining latency and availability alarms for this operation in the region
     * as measured from either the server side or canary
     * @param scope
     * @param operation
     * @param nameSuffix
     * @param regionalAvailabilityAlarm
     * @param regionalLatencyAlarm
     * @returns
     */
    static createRegionalCustomerExperienceAlarm(scope, operationName, nameSuffix, regionalAvailabilityAlarm, regionalLatencyAlarm) {
        return new aws_cloudwatch_1.CompositeAlarm(scope, operationName + 'RegionalCustomerExperienceAlarm', {
            compositeAlarmName: aws_cdk_lib_1.Fn.ref('AWS::Region') +
                '-' +
                operationName.toLowerCase() +
                '-customer-experience-imact' +
                nameSuffix,
            alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(regionalAvailabilityAlarm, regionalLatencyAlarm),
        });
    }
    static createRegionalInstanceContributorsToHighLatency(scope, metricDetails, ruleDetails) {
        let ruleBody = new InsightRuleBody_1.InsightRuleBody();
        ruleBody.logGroupNames = ruleDetails.logGroups.map((x) => x.logGroupName);
        ruleBody.aggregateOn = 'Count';
        ruleBody.logFormat = 'JSON';
        ruleBody.contribution = {
            keys: [ruleDetails.instanceIdJsonPath],
            filters: [
                {
                    Match: ruleDetails.successLatencyMetricJsonPath,
                    GreaterThan: metricDetails.successAlarmThreshold,
                },
                {
                    Match: ruleDetails.operationNameJsonPath,
                    In: [metricDetails.operationName],
                },
            ],
        };
        return new aws_cloudwatch_1.CfnInsightRule(scope, 'RegionPerInstanceHighLatencyRule', {
            ruleName: aws_cdk_lib_1.Fn.ref('AWS::Region') +
                `-${metricDetails.operationName.toLowerCase()}-per-instance-high-latency-server`,
            ruleState: 'ENABLED',
            ruleBody: ruleBody.toJson(),
        });
    }
    static createRegionalInstanceContributorsToFaults(scope, metricDetails, ruleDetails) {
        let ruleBody = new InsightRuleBody_1.InsightRuleBody();
        ruleBody.logGroupNames = ruleDetails.logGroups.map((x) => x.logGroupName);
        ruleBody.aggregateOn = 'Count';
        ruleBody.logFormat = 'JSON';
        ruleBody.contribution = {
            keys: [ruleDetails.instanceIdJsonPath],
            filters: [
                {
                    Match: ruleDetails.faultMetricJsonPath,
                    GreaterThan: 0,
                },
                {
                    Match: ruleDetails.operationNameJsonPath,
                    In: [metricDetails.operationName],
                },
            ],
        };
        return new aws_cloudwatch_1.CfnInsightRule(scope, 'RegionPerInstanceErrorRule', {
            ruleName: aws_cdk_lib_1.Fn.ref('AWS::Region') +
                `-${metricDetails.operationName.toLowerCase()}-per-instance-faults-server`,
            ruleState: 'ENABLED',
            ruleBody: ruleBody.toJson(),
        });
    }
}
exports.AvailabilityAndLatencyAlarmsAndRules = AvailabilityAndLatencyAlarmsAndRules;
//# sourceMappingURL=data:application/json;base64,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