"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToSqs = void 0;
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
const cdk = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayToSqs class.
 */
class ApiGatewayToSqs extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayToSqs class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ApiGatewayToSqsProps} props - user provided props for the construct.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the dead letter queue, if applicable
        if (!props.deployDeadLetterQueue || props.deployDeadLetterQueue === true) {
            const [dlq] = defaults.buildQueue(this, 'deadLetterQueue', {
                queueProps: props.queueProps
            });
            this.deadLetterQueue = defaults.buildDeadLetterQueue({
                deadLetterQueue: dlq,
                maxReceiveCount: (props.maxReceiveCount) ? props.maxReceiveCount : 3
            });
        }
        // Setup the queue
        [this.sqsQueue] = defaults.buildQueue(this, 'queue', {
            queueProps: props.queueProps,
            deadLetterQueue: this.deadLetterQueue
        });
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, props.apiGatewayProps);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        // Setup the API Gateway resource
        const apiGatewayResource = this.apiGateway.root.addResource('message');
        // Setup API Gateway methods
        // Create
        if (props.allowCreateOperation && props.allowCreateOperation === true && props.createRequestTemplate) {
            const createRequestTemplate = "Action=SendMessage&MessageBody=$util.urlEncode(\"$input.body\")";
            this.addActionToPolicy("sqs:SendMessage");
            defaults.addProxyMethodToApiResource({
                service: "sqs",
                path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "POST",
                apiResource: this.apiGateway.root,
                requestTemplate: createRequestTemplate,
                contentType: "'application/x-www-form-urlencoded'"
            });
        }
        // Read
        if (!props.allowReadOperation || props.allowReadOperation === true) {
            const getRequestTemplate = "Action=ReceiveMessage";
            this.addActionToPolicy("sqs:ReceiveMessage");
            defaults.addProxyMethodToApiResource({
                service: "sqs",
                path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "GET",
                apiResource: this.apiGateway.root,
                requestTemplate: getRequestTemplate,
                contentType: "'application/x-www-form-urlencoded'"
            });
        }
        // Delete
        if (props.allowDeleteOperation && props.allowDeleteOperation === true) {
            const deleteRequestTemplate = "Action=DeleteMessage&ReceiptHandle=$util.urlEncode($input.params('receiptHandle'))";
            this.addActionToPolicy("sqs:DeleteMessage");
            defaults.addProxyMethodToApiResource({
                service: "sqs",
                path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "DELETE",
                apiResource: apiGatewayResource,
                requestTemplate: deleteRequestTemplate,
                contentType: "'application/x-www-form-urlencoded'"
            });
        }
    }
    addActionToPolicy(action) {
        this.apiGatewayRole.addToPolicy(new iam.PolicyStatement({
            resources: [
                this.sqsQueue.queueArn
            ],
            actions: [`${action}`]
        }));
    }
}
exports.ApiGatewayToSqs = ApiGatewayToSqs;
//# sourceMappingURL=data:application/json;base64,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