from __future__ import annotations

import dataclasses
import re
from collections import defaultdict
from typing import List, Any, Optional, Type, TypeVar, Dict, DefaultDict

header = [
    "# Autogenerated by solidity-typestub",
    "from typing import List, Tuple, overload",
    "",
    "AddressType = str",
    "",
]


@dataclasses.dataclass(frozen=True)
class AddressType:
    address: str

    __name__ = "AddressType"


T = TypeVar("T", bound="Parsable")


@dataclasses.dataclass(frozen=True)
class Parsable:
    @classmethod
    def parse(cls: Type[T], data: Dict[str, Any]) -> Optional[T]:
        raise NotImplementedError()


@dataclasses.dataclass(frozen=True)
class SolidityType:
    name: str

    def python_type(self) -> type:
        raise NotImplementedError()

    def python_type_repr(self) -> str:
        return self.python_type().__name__

    @classmethod
    def parse(cls, data: Dict[str, Any]) -> SolidityType:
        typ: str = data["type"]
        array_pat = re.compile(r"(?P<typ>.*)\[(?P<size>\d*)]$")
        # Once mypy supports match we can upgrade this and use walrus operator
        if re.match(array_pat, typ):
            match = re.match(array_pat, typ)
            assert match
            data = data.copy()
            data["type"] = re.sub(array_pat, r"\g<typ>", data["type"])
            size = match.groupdict().get("size")

            return Array(
                typ, SolidityType.parse(data), size=int(size) if size else None
            )
        elif typ.startswith("uint") or typ.startswith("int"):
            return Integer(typ)
        elif typ.startswith("bytes"):
            return Bytes(typ)
        elif typ == "string":
            return String(typ)
        elif typ == "address":
            return Address(typ)
        elif typ == "bool":
            return Bool(typ)
        else:
            print("UNRECOGNIZED", typ)
            assert False


@dataclasses.dataclass(frozen=True)
class String(SolidityType):
    def python_type(self) -> type:
        return str


@dataclasses.dataclass(frozen=True)
class Bool(SolidityType):
    def python_type(self) -> type:
        return bool


@dataclasses.dataclass(frozen=True)
class Address(SolidityType):
    def python_type(self) -> type:
        return AddressType


@dataclasses.dataclass(frozen=True)
class Array(SolidityType):
    internal_type: SolidityType
    size: Optional[int] = None

    def python_type(self) -> type:
        # return List[self.internal_type.python_type()]
        return List[Any]

    def python_type_repr(self) -> str:
        return f"List[{self.internal_type.python_type_repr()}]"


@dataclasses.dataclass(frozen=True)
class Integer(SolidityType):
    size: int = dataclasses.field(init=False)

    def __post_init__(self) -> None:
        size = self.name.replace("uint", "").replace("int", "")
        size_i = int(size) if size else 256
        object.__setattr__(self, "size", size_i)

    def python_type(self) -> type:
        return int


@dataclasses.dataclass(frozen=True)
class Bytes(SolidityType):
    size: int = dataclasses.field(init=False)

    def __post_init__(self) -> None:
        size = int(self.name.replace("bytes", ""))
        assert size <= 32, "Bytes must be <= 32"
        object.__setattr__(self, "size", size)

    def python_type(self) -> type:
        return bytes


@dataclasses.dataclass(frozen=True)
class AbiType(Parsable):
    name: str
    internal_type: Optional[str]
    typ: SolidityType
    components: List[Any] = dataclasses.field(default_factory=list)

    @classmethod
    def parse(cls, data: Dict[str, Any]) -> AbiType:
        return cls(
            data["name"],
            data.get("internalType"),
            SolidityType.parse(data),
        )

    @property
    def printable_name(self) -> str:
        return self.name or "_"


@dataclasses.dataclass(frozen=True)
class SolidityMethod(Parsable):
    @classmethod
    def parse(cls, data: Dict[str, Any]) -> Optional[SolidityMethod]:
        data_type = data["type"]
        if data_type == "function":
            return Function.parse(data)
        if data_type == "constructor":
            return Constructor.parse(data)
        else:
            return None
            # assert False, f"Unsupported method type: {data_type}"

    def stub_source(self) -> str:
        raise NotImplementedError()


@dataclasses.dataclass(frozen=True)
class Function(SolidityMethod):
    name: str
    inputs: List[AbiType] = dataclasses.field(default_factory=list)
    outputs: List[AbiType] = dataclasses.field(default_factory=list)
    state_mutability: Optional[str] = None

    @classmethod
    def parse(cls, data: Dict[str, Any]) -> Function:
        return cls(
            name=data["name"],
            inputs=[AbiType.parse(i) for i in data["inputs"]],
            outputs=[AbiType.parse(i) for i in data["outputs"]],
        )

    def stub_source(self) -> str:
        input_data: List[str] = ["self"] + [
            f"{i.printable_name}: {i.typ.python_type_repr()}"
            for i in self.inputs
            if i.typ
        ]
        inputs = ", ".join(input_data)
        output_data = [
            i.typ.python_type_repr() for i in self.outputs if i.typ is not None
        ]
        if len(output_data) == 1:
            outputs = output_data[0]
        elif output_data:
            outputs = "Tuple[" + ", ".join(output_data) + "]"
        else:
            outputs = "None"
        return f"\tdef call({inputs}) -> {outputs}: ..."

    @property
    def class_function_name(self) -> str:
        return f"Function{self.name}"


@dataclasses.dataclass(frozen=True)
class Constructor(Function):
    @classmethod
    def parse(cls, data: Dict[str, Any]) -> Constructor:
        return cls(
            "constructor",
            inputs=[AbiType.parse(i) for i in data["inputs"]],
        )


@dataclasses.dataclass(frozen=True)
class Contract:
    name: str
    raw_abi: List[Dict[str, Any]]
    function_data: List[Function] = dataclasses.field(init=False)

    def __post_init__(self) -> None:
        data = [SolidityMethod.parse(method_abi) for method_abi in self.raw_abi]
        object.__setattr__(
            self,
            "function_data",
            [m for m in data if m],
        )

    @classmethod
    def from_json(cls, json: Dict[str, Any]) -> Contract:
        return cls(json["contractName"], json["abi"])

    def stub_source(self) -> List[str]:

        function_data: DefaultDict[str, List[str]] = defaultdict(list)

        functions: Dict[str, Function] = {}

        for func in self.function_data:
            function_data[func.name].append(func.stub_source())
            functions[func.name] = func

        data = []

        for fname, methods in function_data.items():
            function_contract: List[str] = [
                f"class {functions[fname].class_function_name}:"
            ]

            for method in methods:
                if len(methods) > 1:
                    function_contract.append("\t@overload")
                function_contract.append(f"{method}")
            data.extend(function_contract)

        data.append(f"class {self.proxy_name}:")
        for fname in function_data:
            data.append(
                f"\tdef {fname}(self) -> {functions[fname].class_function_name}: ..."
            )

        data.extend(
            [
                f"class {self.name}:",
                f"\tfunctions : {self.proxy_name}",
            ]
        )

        return data

    @property
    def proxy_name(self) -> str:
        return f"Proxy{self.name}"
