"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeBuildAction = exports.CodeBuildActionType = void 0;
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const __1 = require("..");
const action_1 = require("../action");
const source_action_1 = require("../codecommit/source-action");
/**
 * The type of the CodeBuild action that determines its CodePipeline Category - Build, or Test.
 *
 * The default is Build.
 *
 * @stability stable
 */
var CodeBuildActionType;
(function (CodeBuildActionType) {
    CodeBuildActionType[CodeBuildActionType["BUILD"] = 0] = "BUILD";
    CodeBuildActionType[CodeBuildActionType["TEST"] = 1] = "TEST";
})(CodeBuildActionType = exports.CodeBuildActionType || (exports.CodeBuildActionType = {}));
/**
 * CodePipeline build action that uses AWS CodeBuild.
 *
 * @stability stable
 */
class CodeBuildAction extends action_1.Action {
    /**
     * @stability stable
     */
    constructor(props) {
        super({
            ...props,
            category: props.type === CodeBuildActionType.TEST
                ? codepipeline.ActionCategory.TEST
                : codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 1, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [props.input, ...props.extraInputs || []],
            resource: props.project,
        });
        this.props = props;
    }
    /**
     * Reference a CodePipeline variable defined by the CodeBuild project this action points to.
     *
     * Variables in CodeBuild actions are defined using the 'exported-variables' subsection of the 'env'
     * section of the buildspec.
     *
     * @param variableName the name of the variable to reference.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec-ref-syntax
     * @stability stable
     */
    variable(variableName) {
        return this.variableExpression(variableName);
    }
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     *
     * @stability stable
     */
    bound(scope, _stage, options) {
        var _a;
        // check for a cross-account action if there are any outputs
        if ((this.actionProperties.outputs || []).length > 0) {
            const pipelineStack = cdk.Stack.of(scope);
            const projectStack = cdk.Stack.of(this.props.project);
            if (pipelineStack.account !== projectStack.account) {
                throw new Error('A cross-account CodeBuild action cannot have outputs. ' +
                    'This is a known CodeBuild limitation. ' +
                    'See https://github.com/aws/aws-cdk/issues/4169 for details');
            }
        }
        // grant the Pipeline role the required permissions to this Project
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.props.project.projectArn],
            actions: [
                'codebuild:BatchGetBuilds',
                `codebuild:${this.props.executeBatchBuild ? 'StartBuildBatch' : 'StartBuild'}`,
                `codebuild:${this.props.executeBatchBuild ? 'StopBuildBatch' : 'StopBuild'}`,
            ],
        }));
        // allow the Project access to the Pipeline's artifact Bucket
        // but only if the project is not imported
        // (ie., has a role) - otherwise, the IAM library throws an error
        if (this.props.project.role) {
            if ((this.actionProperties.outputs || []).length > 0) {
                options.bucket.grantReadWrite(this.props.project);
            }
            else {
                options.bucket.grantRead(this.props.project);
            }
        }
        if (this.props.project instanceof codebuild.Project) {
            this.props.project.bindToCodePipeline(scope, {
                artifactBucket: options.bucket,
            });
        }
        for (const inputArtifact of this.actionProperties.inputs || []) {
            // if any of the inputs come from the BitBucketSourceAction
            // with codeBuildCloneOutput=true,
            // grant the Project's Role to use the connection
            const connectionArn = inputArtifact.getMetadata(__1.BitBucketSourceAction._CONNECTION_ARN_PROPERTY);
            if (connectionArn) {
                this.props.project.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['codestar-connections:UseConnection'],
                    resources: [connectionArn],
                }));
            }
            // if any of the inputs come from the CodeCommitSourceAction
            // with codeBuildCloneOutput=true,
            // grant the Project's Role git pull access to the repository
            const codecommitRepositoryArn = inputArtifact.getMetadata(source_action_1.CodeCommitSourceAction._FULL_CLONE_ARN_PROPERTY);
            if (codecommitRepositoryArn) {
                this.props.project.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['codecommit:GitPull'],
                    resources: [codecommitRepositoryArn],
                }));
            }
        }
        const configuration = {
            ProjectName: this.props.project.projectName,
            EnvironmentVariables: this.props.environmentVariables &&
                cdk.Stack.of(scope).toJsonString(codebuild.Project.serializeEnvVariables(this.props.environmentVariables, (_a = this.props.checkSecretsInPlainTextEnvVariables) !== null && _a !== void 0 ? _a : true, this.props.project)),
        };
        if ((this.actionProperties.inputs || []).length > 1) {
            // lazy, because the Artifact name might be generated lazily
            configuration.PrimarySource = cdk.Lazy.string({ produce: () => this.props.input.artifactName });
        }
        if (this.props.executeBatchBuild) {
            configuration.BatchEnabled = 'true';
            this.props.project.enableBatchBuilds();
        }
        return {
            configuration,
        };
    }
}
exports.CodeBuildAction = CodeBuildAction;
//# sourceMappingURL=data:application/json;base64,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