#!/usr/bin/env python

# Always prefer setuptools over distutils
from __future__ import print_function
import collections, os, imp, sys
from setuptools import setup, find_packages
import inspect
import re

import distutils.cmd

__doc__ = """

To install as system package:

  python setup.py install

To install as local package, just run:

  mkdir /tmp/builds/
  python setup.py install --root=/tmp/builds
  /tmp/builds/usr/bin/$DS -? -v4

To tune some options:

  RU=/opt/control
  python setup.py egg_info --egg-base=tmp install --root=$RU/files --no-compile \
    --install-lib=lib/python/site-packages --install-scripts=ds

-------------------------------------------------------------------------------
"""
try:
    # python3
    from builtins import str
except:
    pass

release = open("fandango/VERSION").read()

scripts = [
    "./fandango/scripts/csv2tango",
    "./fandango/scripts/tango2csv",
    "./fandango/scripts/tango2json",
    "./fandango/scripts/sardanact",
    "./fandango/scripts/tango_cleanup",
    "./fandango/scripts/folder-gui",
    "./fandango/scripts/fandango",
    "./fandango/scripts/DynamicDS",
   "./fandango/scripts/WorkerDS",
   "./fandango/scripts/CopyCatDS",
    "./fandango/scripts/FolderDS",
    "./fandango/scripts/tango_servers",
    "./fandango/scripts/tango_host",
    "./fandango/scripts/tango_property",
    "./fandango/scripts/tango_monitor",
]

entry_points = {
    "console_scripts": [
        #'CopyCatDS = fandango.interface.CopyCatDS:main',
        #'WorkerDS = fandango.device.WorkerDS:main',
    ],
}


class GenerateTestsCommand(distutils.cmd.Command):
    description = "Command to generate fandango tests based on their definitions"
    user_options = [
        # long option, short option, description
        (
            "test-folder-path=",
            None,
            "Automatic tests will be generated here. The script will also look here for definitions.py "
            "which have to be a python module containing tests definitions in format described in "
            "fandango/tests/definitions.py",
        ),
        (
            "test-files-suffix=",
            None,
            "All autogenerated tests files will have this suffix.",
        ),
    ]

    def initialize_options(self):
        self.test_folder_path = os.path.dirname(os.path.abspath(__file__)) + "/tests"
        self.test_files_suffix = "auto_gen"
        self.definitions_file_path = self.test_folder_path + "/definitions.py"

    def finalize_options(self):
        if (
            os.path.isdir(self.test_folder_path) is False
            or os.path.isfile(self.definitions_file_path) is False
        ):
            raise Exception("File definitions.py not found.")

    def run(self):
        """This will generate tests for all definitions from definitions.py.
        Existing tests with name test_* in test folder will be reloaded."""
        # load test definitions
        sys.path.append(self.test_folder_path)
        import definitions

        def_tests_dict = collections.OrderedDict(sorted(definitions.tests.items()))
        skip_tests = definitions.skip_tests
        tests = self.prepare_tests(def_tests_dict, skip_tests)

        self.generate_tests_files(tests)
        print("Tests generated. Please verify output.")

    def prepare_tests(self, defined_tests, skip_tests):
        tests = {}
        # tests = { 'test_functional': { 'test_floor': { 'docs': 'Here's my doc', 'params': ((...), (...)) } }
        for tested_function, test_defs in list(defined_tests.items()):
            if tested_function in skip_tests:
                continue
            test_filename, test_name = self.parse_test_name(tested_function)
            if test_filename not in tests:
                tests[test_filename] = collections.OrderedDict()
            for test_suffix, test_def in list(test_defs.items()):
                if test_def.get("method"):
                    continue
                test_def["function"] = tested_function
                if test_suffix:
                    tests[test_filename]["_".join([test_name, test_suffix])] = test_def
                else:
                    tests[test_filename][test_name] = test_def
        return tests


    def parse_test_name(self, s):
        """Takes fully qualified name of tested method/function name from definitions.py and returns test filename and test name."""
        splitted = re.sub("^fandango.", "", s).split(".")
        file = "_".join(["test", splitted[0]]) + ".py"
        test = "_".join(["test"] + splitted[1:])
        return file, test

    def generate_tests_files(self, tests):
        if sys.version_info[0] >= 3 and sys.version_info[1] >= 6:
            import black

        for filename, test_defs in list(tests.items()):
            filename = filename.split(".")[:1]
            filename = "_".join(filename + [self.test_files_suffix]) + ".py"
            filepath = "/".join([self.test_folder_path, filename])
            with open(filepath, "w") as f:
                content = self.generate_tests_definitions(test_defs)
                if sys.version_info[0] >= 3 and sys.version_info[1] >= 6:
                    content = black.format_file_contents(
                        content, fast=True, mode=black.FileMode()
                    )
                f.write(content)

    def generate_tests_definitions(self, test_defs):
        """Generates tests function definitions based on provided definition.
        Parameters
        ----------
        test_defs : Tests definitions. Eg. { test_name_with_suffix_1: {'docs': ..., 'params': ...} ... }
        """
        tests = ""
        header = "import fandango\n"
        for test_name, test_definition in list(test_defs.items()):
            test = (
                "\ndef {}():\n"
                '    """{}"""\n'
                "    params = {}\n"
                "    for args, kwargs, init_args, init_kwargs, result in params:\n"
                "        assert {}(*args, **kwargs) == result\n"
            ).format(
                test_name,
                test_definition["docs"],
                test_definition["params"],
                test_definition["function"],
            )
            tests += test
        return header + tests


setup(
    name="fandango",
    version=str(release).strip(),
    packages=find_packages(),
    description="Simplify the configuration of big Tango control systems",
    long_description="Fandango is a Python module created to simplify the "
    "configuration of big control systems; implementing the behavior of Jive "
    "(configuration) and/or Astor (deployment) tools in methods that could "
    "be called from scripts using regexp and wildcards. Fandango provides "
    "functional methods, classes and utilities to develop high-level device "
    "servers and APIs for Tango control system.",
    author="Sergi Rubio",
    author_email="srubio@cells.es",
    classifiers=[
        "Development Status :: 5 - Production/Stable",
        "Environment :: Other Environment",
        "Intended Audience :: Developers",
        "License :: OSI Approved :: "
        "GNU Lesser General Public License v3 or later (LGPLv3+)",
        "Natural Language :: English",
        "Operating System :: Microsoft :: Windows",
        "Operating System :: POSIX",
        "Operating System :: POSIX :: Linux",
        "Operating System :: Unix",
        "Programming Language :: Python",
        "Topic :: Scientific/Engineering",
        "Topic :: Software Development :: Libraries",
    ],
    extras_require={
        "tests": [
            "pytest",
            "pytest-freezegun",
            "pytz",
            "freezegun",
            "future",
            #'black;python_version>="3.6"',
        ]
    },
    platforms=["Linux,Windows XP/Vista/7/8"],
    install_requires=['future'],
    scripts=scripts,
    entry_points=entry_points,
    include_package_data=True,
    zip_safe=False,
    python_requires=">3.6",
    cmdclass={
        "generate_tests": GenerateTestsCommand,
    },
)
