"""Simple Wiki - это вики движок, отображающий файловую структуру в веб-энциклопедию; поддерживает статьи в форматах .html, .md. Simple Wiki написана на языке python, поддерживает гибкое расширение и стилизацию."""
from __future__ import annotations

import json
import mimetypes
import random
import re
from dataclasses import dataclass
from pathlib import Path
from typing import Optional, Tuple, List, Callable

import chardet
import jinja2
import typer
from attributedict.collections import AttributeDict
from markdown import markdown

from webserver import serve, default_response_callback

__version__ = '0.1.2'
templates = jinja2.Environment(loader=jinja2.FileSystemLoader('templates'), autoescape=['html'])
mimetypes.add_type('text/markdown', '.md', strict=False)
mimetypes.add_type('font/woff', '.woff', strict=False)
mimetypes.add_type('font/woff2', '.woff2', strict=False)
config = AttributeDict(json.loads((Path.cwd() / 'config.json').read_text()))
parsers = {
	'.md': lambda text: markdown(text, extensions=['extra', 'mdx_math', 'admonition', 'toc', 'internallinks'] + config['markdown_extensions'], extension_configs={
		'extra'        : {
			'footnotes'  : {
				'UNIQUE_IDS': True
			},
			'fenced_code': {
				'lang_prefix': 'lang-'
			}
		},
		'mdx_math'     : {
			'enable_dollar_delimiter': True,
			'add_preview'            : True
		},
		'internallinks': {
			'pattern'  : r'\[\[([\w0-9_ -/#]+)\]\]',
			'end_url'  : '',
			'build_url': make_internal_link
		}
	}),
}

# region default images

default_icon = 'data:image/png;base64,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'

default_logo = 'data:image/png;base64,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'


# endregion


@dataclass
class WikiLink:
	name: str
	url: str
	page: bool

	@staticmethod
	def from_path(wiki_link_path: Path, wiki_root_path: Path) -> WikiLink:
		"""
		Generate link from file system path.
		"""
		if not wiki_link_path.is_relative_to(wiki_root_path):
			raise ValueError(f'Wiki path {wiki_link_path} must be relative to root {wiki_root_path}.')
		if not is_page(wiki_link_path) and not is_section(wiki_link_path):
			raise ValueError(f'Wiki path must point to page or section not to resource.')
		page = not is_section(wiki_link_path)
		if page:
			wiki_link_path = wiki_link_path.parent / wiki_link_path.with_suffix('')
		return WikiLink(name=wiki_link_path.name, url='/' + str(wiki_link_path.relative_to(wiki_root_path)), page=page)

	def __repr__(self):
		return (self.page and 'Page' or 'Section') + f': {self.name} ({self.url})'

	def __str__(self):
		return repr(self)


def validate_port(port: int) -> int:
	"""
	Check whether port index is valid.
	"""
	if not (0 < port <= 65535):
		raise typer.BadParameter('Port value must be in range from 1 to 65535.')
	return port


def show_version(value: bool):
	"""
	Print version info and exit.
	"""
	if value:
		typer.echo(f"Simple Wiki {__version__}")
		raise typer.Exit()


def render_template(name: str, **rendering_arguments) -> str:
	return templates.get_template(name).render(config=config, **rendering_arguments)


def same_paths(*paths: Path) -> bool:
	"""
	Whether the given paths point to same location.
	"""
	if len(paths) < 2:
		raise ValueError('At least two paths are reqired for comparison.')
	pattern = paths[0].absolute()
	return all([x.absolute() == pattern for x in paths[1:]])


def make_wiki_path(path: Path, wiki_root_path: Path) -> str:
	"""
	Convert path to wiki page url.
	"""
	return str(path.relative_to(wiki_root_path).with_suffix(''))


def make_internal_link(label, prefix, postfix):
	wiki_path = Path.cwd() / 'wiki'
	label = re.sub(r'\s*/\s*', '/', label)
	match = re.search(r'\s*#.*', label)
	if match:
		anchor = re.sub(r'\s*#\s*', '#', match.group())
		label = label[:match.start(0)]
	else:
		anchor = ''
	found = list(wiki_path.rglob(f'{label}*'))
	if len(found) != 0:
		name = found[0].name.split('.')[0]
		label = str(found[0].relative_to(wiki_path))
	else:
		name = label.split('/')[-1]
	# if len(anchor) != 0:
	# 	name = anchor[1:]
	return f'{prefix}{label}{anchor}{postfix}', name


def find_paths(path: Path, pattern: str, filter: Callable[[Path], bool]) -> List[Path]:
	"""
	Glob pattern relatively to path and filter results by predicate.
	"""
	return [x for x in sorted(path.glob(pattern), key=str) if filter(x)]


def generate_side_block(links: List[WikiLink], label: str = '', *, sort: bool = True) -> str:
	"""
	Generate block of links markup for side bar.
	:param label: label of block.
	:param links: list of label, url pairs.
	:param sort: whether to sort links by name (label).
	"""
	if sort:
		links = sorted(links, key=lambda link: link.name)
	return render_template('side_block.html', label=label, links=links)


def is_section(wiki_path: Path) -> bool:
	return wiki_path.exists() and wiki_path.is_dir()


def is_page(wiki_path: Path) -> bool:
	page_types = ['text', 'pdf']
	mime = mimetypes.guess_type(wiki_path, strict=False)[0]
	return wiki_path.exists() and wiki_path.is_file() and mime is not None and any([x in mime for x in page_types])


def is_resource(path: Path) -> bool:
	return path.is_relative_to(Path.cwd() / 'resources')


def is_wiki_page(path: Path) -> bool:
	return path.is_relative_to(Path.cwd() / 'wiki')


def is_search_query(path: Path) -> bool:
	cwd = Path.cwd()
	search_path = cwd / 'search'
	# noinspection PyUnboundLocalVariable
	return not same_paths(path, search_path) and path.is_relative_to(search_path) and len(parents := path.relative_to(search_path).parents) == 1 and same_paths(search_path, search_path / parents[0])


def get_query(search_query: Path) -> Optional[str]:
	if is_search_query(search_query):
		return search_query.name


def generate_sidebar(current_wiki_path: Path) -> str:
	"""
	Generate wiki side bar menu.
	"""
	wiki_path = Path.cwd() / 'wiki'
	main_links = [WikiLink(name='Заглавная страница', url='/', page=False)]
	if not same_paths(current_wiki_path, wiki_path):
		# current section
		if current_wiki_path.parent.is_relative_to(wiki_path) and not same_paths(current_wiki_path.parent, wiki_path):
			main_links.append(WikiLink.from_path(current_wiki_path.parent, wiki_path))
		# supsection
		if not is_section(current_wiki_path) and current_wiki_path.parent.parent.is_relative_to(wiki_path) and not same_paths(current_wiki_path.parent.parent, wiki_path):
			main_links.append(WikiLink.from_path(current_wiki_path.parent.parent, wiki_path))
	main_block = generate_side_block(main_links, sort=False)
	search_section = current_wiki_path
	if not is_section(current_wiki_path):
		search_section = search_section.parent
	siblings = [WikiLink.from_path(x, wiki_path) for x in find_paths(search_section, '*.*', lambda x: is_page(x) and not same_paths(x, current_wiki_path))]
	if len(siblings) == 0:
		siblings_block = ''
	else:
		if len(siblings) > 10:
			siblings = random.sample(siblings, 10)
		siblings_block = generate_side_block(siblings, 'Статьи в разделе')
	subsections = [WikiLink.from_path(x, wiki_path) for x in find_paths(search_section, '*', is_section)]
	if len(subsections) == 0:
		subsections_block = ''
	else:
		if len(subsections) > 10:
			subsections = random.sample(subsections, 10)
		subsections_block = generate_side_block(subsections, 'Подразделы')
	return main_block + siblings_block + subsections_block


def generate_page(content: str, current_wiki_path: Path) -> str:
	"""
	Generate page markup for content markup part.
	"""
	resources_root = Path('resources')
	icon_path = resources_root / config['icon']
	if icon_path.exists() and icon_path.is_file():
		icon_path = f'/{config["icon"]}'
	else:
		icon_path = default_icon
	logo_path = resources_root / config['logo']
	if logo_path.exists() and logo_path.is_file():
		logo = logo_path.read_text(encoding='utf-8')
	else:
		logo = f'<a href="/" title="Перейти на заглавную страницу" style="background-position:center center;background-repeat:no-repeat;display:block;width:10em;height:160px;text-decoration:none;background-image:url({default_logo});background-size:contain"></a>'
	return render_template('page.html', logo=logo, content=content, sidebar=generate_sidebar(current_wiki_path), icon=icon_path)


def generate_section_content(path: Path) -> str:
	"""
	Generate markup for whole section.
	"""
	if not is_section(path):
		raise ValueError(f'{path} is not a section.')
	wiki_path = Path.cwd() / 'wiki'
	links = [WikiLink.from_path(x, wiki_path) for x in path.glob('*') if is_page(x) or is_section(x)]
	return render_template('section.html', section=path.name, subsections=[x for x in links if not x.page], pages=[x for x in links if x.page])


def match(query, text) -> bool:
	"""
	Match test with query.
	"""
	if len(query) < 3 or len(text) == 0:
		return False
	return query.lower() in text.lower()


def generate_search_results(query) -> str:
	"""
	Generate search results content markup.
	"""
	query = query.strip()
	wiki_path = Path.cwd() / 'wiki'
	results = []
	for path in wiki_path.rglob('*'):
		if not is_page(path) and not is_section(path):
			continue
		if match(query, str(path.relative_to(wiki_path))):
			results.append(path)
		elif is_page(path):
			mime = mimetypes.guess_type(path, strict=False)[0]
			if mime is not None and 'text' in mime:
				content = path.read_bytes()
				try:
					content = content.decode(chardet.detect(content)['encoding'])
					if match(query, content):
						results.append(path)
				except UnicodeDecodeError:
					pass
	return render_template('search_results.html', query=query, found=[WikiLink.from_path(x, wiki_path) for x in results])


def route(requested_path: Path) -> Optional[Path]:
	cwd = Path.cwd()
	if is_search_query(requested_path):
		return requested_path
	if len(requested_path.suffixes) == 0:
		path = cwd / 'wiki' / requested_path.relative_to(cwd)
		if not same_paths(path, cwd / 'wiki'):
			for file in path.parent.glob(f'{path.name}.*'):
				return file
		if path.exists() and path.is_dir():
			return path
		return None
	else:
		path = cwd / 'wiki' / requested_path.relative_to(cwd)
		if path.exists() and path.is_file():
			return path
		path = cwd / 'resources' / requested_path.relative_to(cwd)
		if path.exists() and path.is_file():
			return path
	return None


def response(requested_path: Path) -> Optional[Tuple[bytes, str]]:
	query = get_query(requested_path)
	if query is not None:
		return generate_page(generate_search_results(query), requested_path).encode('utf-8'), 'text/html'
	elif is_resource(requested_path):
		return default_response_callback(requested_path)
	elif is_wiki_page(requested_path):
		if is_section(requested_path):
			content = generate_section_content(requested_path)
		elif requested_path.exists() and requested_path.is_file():
			mime = mimetypes.guess_type(requested_path, strict=False)[0]
			if mime is None or 'text' not in mime:
				return default_response_callback(requested_path)
			content = requested_path.read_bytes()
			try:
				content = content.decode(chardet.detect(content)['encoding'])
			except UnicodeDecodeError:
				return default_response_callback(requested_path)
			content_type = requested_path.suffix
			if content_type in parsers:
				content = parsers[content_type](content)
		else:
			return None
		return generate_page(content, requested_path).encode('utf-8'), 'text/html'
	else:
		return None


def cli(interface: str = typer.Option(config['interface'], '--interface', '-i', help='Interface IP v4 address or resolvable name (like "127.0.0.1" or "localhost") on which to serve wiki. Default value is "0.0.0.0" for all connected interfaces. Overwrites config.json.'), port: int = typer.Option(config['port'], '--port', '-p', help='Port on which to serve wiki. Default value is 80 for all connected interfaces. Overwrites config.json.', callback=validate_port), version: Optional[bool] = typer.Option(None, "--version", callback=show_version, is_eager=True, help=show_version.__doc__)):
	"""
	Run wiki server.
	"""
	try:
		serve(interface=interface, port=port, routing_callback=route, response_callback=response)
	except Exception as error:
		typer.secho(str(error), fg=typer.colors.RED, err=True)
		typer.Exit(-1)


def main():
	typer.run(cli)


if __name__ == "__main__":
	main()
