'''
Modified WikiLinks Extension for Python-Markdown
======================================
Added ability to set custom pattern of markdown inline syntax via "pattern" option. Also label can be changed via second return value.
======================================

Converts [[WikiLinks]] to relative links.

See <https://Python-Markdown.github.io/extensions/wikilinks>
for documentation.

Original code Copyright [Waylan Limberg](http://achinghead.com/).

All changes Copyright The Python Markdown Project

License: [BSD](https://opensource.org/licenses/bsd-license.php)

'''

from markdown.extensions import Extension
from markdown.inlinepatterns import InlineProcessor
from markdown.extensions.wikilinks import build_url
import xml.etree.ElementTree as etree


class InternalLinkExtension(Extension):

	def __init__(self, **kwargs):
		self.config = {
			'base_url'  : ['/', 'String to append to beginning or URL.'],
			'end_url'   : ['/', 'String to append to end of URL.'],
			'html_class': ['wikilink', 'CSS hook. Leave blank for none.'],
			'build_url' : [build_url, 'Callable formats URL and label.'],
			'pattern'   : [r'\[\[([\w0-9_ -]+)\]\]', 'Regular expression pattern for markdown inline syntax.'],
		}

		super().__init__(**kwargs)

	def extendMarkdown(self, md):
		self.md = md

		# append to end of inline patterns
		INTERNALLINK_RE = self.config['pattern'][0]
		internallinkPattern = InternalLinksInlineProcessor(INTERNALLINK_RE, self.getConfigs())
		internallinkPattern.md = md
		md.inlinePatterns.register(internallinkPattern, 'internallink', 75)


class InternalLinksInlineProcessor(InlineProcessor):
	def __init__(self, pattern, config):
		super().__init__(pattern)
		self.config = config

	def handleMatch(self, m, data):
		if m.group(1).strip():
			base_url, end_url, html_class = self._getMeta()
			label = m.group(1).strip()
			url, label = self.config['build_url'](label, base_url, end_url)
			a = etree.Element('a')
			a.text = label
			a.set('href', url)
			if html_class:
				a.set('class', html_class)
		else:
			a = ''
		return a, m.start(0), m.end(0)

	def _getMeta(self):
		""" Return meta data or config data. """
		base_url = self.config['base_url']
		end_url = self.config['end_url']
		html_class = self.config['html_class']
		if hasattr(self.md, 'Meta'):
			if 'wiki_base_url' in self.md.Meta:
				base_url = self.md.Meta['wiki_base_url'][0]
			if 'wiki_end_url' in self.md.Meta:
				end_url = self.md.Meta['wiki_end_url'][0]
			if 'wiki_html_class' in self.md.Meta:
				html_class = self.md.Meta['wiki_html_class'][0]
		return base_url, end_url, html_class


def makeExtension(**kwargs):  # pragma: no cover
	return InternalLinkExtension(**kwargs)
