''' This module supports commandline functionality '''
from __future__ import print_function

import argparse
import sys
import os
from psiturk.version import version_number
from psiturk.psiturk_org_services import ExperimentExchangeServices


def process():
    ''' Figure out how we were invoked '''
    invoked_as = os.path.basename(sys.argv[0])

    if invoked_as == "psiturk":
        launch_shell()
    elif invoked_as == "psiturk-server":
        launch_server()
    elif invoked_as == "psiturk-shell":
        launch_shell()
    elif invoked_as == "psiturk-setup-example":
        setup_example()
    elif invoked_as == "psiturk-install":
        install_from_exchange()


def install_from_exchange():
    ''' Install from experiment exchange. '''
    parser = argparse.ArgumentParser(
        description='Download experiment from the psiturk.org experiment\
        exchange (http://psiturk.org/ee).'
    )
    parser.add_argument(
        'exp_id', metavar='exp_id', type=str, help='the id number of the\
        experiment in the exchange'
    )
    args = parser.parse_args()
    exp_exch = ExperimentExchangeServices()
    exp_exch.download_experiment(args.exp_id)


def setup_example():
    ''' Add commands for testing, etc. '''
    parser = argparse.ArgumentParser(
        description='Creates a simple default project (stroop) in the current\
        directory with the necessary psiTurk files.'
    )

    # Optional flags
    parser.add_argument(
        '-v', '--version', help='Print version number.', action="store_true"
    )
    args = parser.parse_args()

    # If requested version just print and quite
    if args.version:
        print(version_number)
    else:
        import psiturk.setup_example as se
        se.setup_example()


def launch_server():
    ''' Add commands for testing, etc.. '''
    parser = argparse.ArgumentParser(
        description='Launch psiTurk experiment webserver process on the\
        host/port defined in config.txt.'
    )

    # Optional flags
    parser.add_argument(
        '-v', '--version', help='Print version number.', action="store_true"
    )
    args = parser.parse_args()

    # If requested version just print and quite
    if args.version:
        print(version_number)
    else:
        import psiturk.experiment_server as es
        es.launch()


def launch_shell():
    ''' Add commands for testing, etc.. '''
    parser = argparse.ArgumentParser(
        description='Launch the psiTurk interactive shell.'
    )

    # Optional flags
    parser.add_argument(
        '-v', '--version', help='Print version number.', action="store_true"
    )
    parser.add_argument(
        '-c', '--cabinmode', help='Launch psiturk in cabin (offline) mode',
        action="store_true"
    )
    script_group = parser.add_mutually_exclusive_group()
    script_group.add_argument(
        '-s', '--script', help='Run commands from a script file'
    )
    script_group.add_argument(
        '-e', '--execute', help='Execute one command specified on the command line'
    )
    script_group.add_argument(
        '-t', '--test', help='Run cmd2 unittest using provided file'
    )
    args, unknownargs = parser.parse_known_args()
    # If requested version just print and quit
    if args.version:
        print(version_number)
    else:
        import psiturk.psiturk_shell as ps
        if args.script:
            ps.run(cabinmode=args.cabinmode, script=args.script, quiet=True)
        elif args.test:
            ps.run(testfile=args.test, quiet=True)
        elif args.execute or unknownargs:
            if unknownargs:
                try:
                    from shlex import quote as cmd_quote
                except ImportError:
                    from pipes import quote as cmd_quote
                execute = ' '.join([cmd_quote(e) for e in unknownargs])
            else:
                execute = args.execute
            ps.run(cabinmode=args.cabinmode, execute=execute, quiet=True)
        else:
            ps.run(cabinmode=args.cabinmode)
