#! /usr/bin/env python
# -*- coding: utf-8 -*-

"""
Module that contains base callbackManager class
"""

from __future__ import print_function, division, absolute_import

import logging

from tpDcc import dcc
from tpDcc.core import dcc as core_dcc
from tpDcc.abstract import callback
from tpDcc.dcc import callback as dcc_callback
from tpDcc.libs.python import decorators

logger = logging.getLogger('tpDcc-core')


@decorators.add_metaclass(decorators.Singleton)
class CallbacksManager(object):
    """
    Static class used to manage all callbacks instances
    """

    _initialized = False
    _callbacks = dict()

    @classmethod
    def initialize(cls):
        """
        Initializes all module callbacks
        """

        if cls._initialized:
            return

        default_callbacks = {
            'Tick': callback.PythonTickCallback
        }

        try:
            shutdown_type = getattr(dcc_callback.Callback(), 'ShutdownCallback')
        except AttributeError:
            shutdown_type = None

        for callback_name in core_dcc.callbacks():

            # Get callback type from tpDcc.DccCallbacks
            n_type = getattr(core_dcc.DccCallbacks, callback_name)[1]['type']
            if n_type == 'simple':
                callback_type = callback.SimpleCallback
            elif n_type == 'filter':
                callback_type = callback.FilterCallback
            else:
                logger.warning('Callback Type "{}" is not valid! Using Simplecallback instead ...'.format(n_type))
                callback_type = callback.SimpleCallback

            # We extract callback types from the specific registered callbacks module
            if not dcc_callback.Callback():
                logger.warning('DCC {} has no callbacks registered!'.format(dcc.get_name()))
                return

            callback_class = getattr(dcc_callback.Callback(), '{}Callback'.format(callback_name), None)
            if not callback_class:
                callback_class = default_callbacks.get(callback_name, callback.ICallback)
                logger.warning(
                    'Dcc {} does not provides an ICallback for {}Callback. Using {} instead'.format(
                        dcc.get_name(), callback_name, callback_class.__name__))

            new_callback = CallbacksManager._callbacks.get(callback_name, None)
            if new_callback:
                new_callback.cleanup()

            CallbacksManager._callbacks[callback_name] = callback_type(callback_class, shutdown_type)

            logger.debug('Creating Callback "{}" of type "{}" ...'.format(callback_name, callback_class))

        cls._initialized = True

    @classmethod
    def register(cls, callback_type, fn, owner=None):
        """
        Registers, is callback exists, a new callback
        :param callback_type: str, type of callback
        :param fn: Python function to be called when callback is emitted
        :param owner, class
        """

        if type(callback_type) in [list, tuple]:
            callback_type = callback_type[0]

        if callback_type in CallbacksManager._callbacks:
            CallbacksManager._callbacks[callback_type].register(fn, owner)

    @classmethod
    def unregister(cls, callback_type, fn):
        """
        Unregisters, is callback exists, a new callback
        :param callback_type: str, type of callback
        :param fn: Python function we want to unregister
        """

        if type(callback_type) in [list, tuple]:
            callback_type = callback_type[0]

        if callback_type in CallbacksManager._callbacks:
            CallbacksManager._callbacks[callback_type].unregister(fn)

    @classmethod
    def unregister_owner_callbacks(cls, owner):
        """
        Unregister all the callbacks from all registered callbacks that belongs to a specific owner
        :param owner: class
        """

        if not cls._initialized:
            return

        for callback_name, register_callback in CallbacksManager._callbacks.items():
            register_callback.unregister_owner_callbacks(owner=owner)

    @classmethod
    def cleanup(cls):
        """
        Cleanup all module callbacks
        :return:
        """

        # if not cls._initialized:
        #     return

        callbacks_to_clean = list()
        for callback_name, register_callback in CallbacksManager._callbacks.items():
            register_callback.cleanup()
            callbacks_to_clean.append(callback_name)
        for callback_name in callbacks_to_clean:
            CallbacksManager._callbacks.pop(callback_name)

        cls._initialized = False
