"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPool = exports.AccountRecovery = exports.Mfa = exports.VerificationEmailStyle = exports.UserPoolOperation = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const punycode_1 = require("punycode/");
const cognito_generated_1 = require("./cognito.generated");
const attr_names_1 = require("./private/attr-names");
const user_pool_client_1 = require("./user-pool-client");
const user_pool_domain_1 = require("./user-pool-domain");
const user_pool_resource_server_1 = require("./user-pool-resource-server");
/**
 * User pool operations to which lambda triggers can be attached.
 */
class UserPoolOperation {
    constructor(operationName) {
        this.operationName = operationName;
    }
    /** A custom user pool operation */
    static of(name) {
        const lowerCamelCase = name.charAt(0).toLowerCase() + name.slice(1);
        return new UserPoolOperation(lowerCamelCase);
    }
}
exports.UserPoolOperation = UserPoolOperation;
_a = JSII_RTTI_SYMBOL_1;
UserPoolOperation[_a] = { fqn: "@aws-cdk/aws-cognito.UserPoolOperation", version: "1.152.0" };
/**
 * Creates a challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
 */
UserPoolOperation.CREATE_AUTH_CHALLENGE = new UserPoolOperation('createAuthChallenge');
/**
 * Advanced customization and localization of messages
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
 */
UserPoolOperation.CUSTOM_MESSAGE = new UserPoolOperation('customMessage');
/**
 * Determines the next challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
 */
UserPoolOperation.DEFINE_AUTH_CHALLENGE = new UserPoolOperation('defineAuthChallenge');
/**
 * Event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
 */
UserPoolOperation.POST_AUTHENTICATION = new UserPoolOperation('postAuthentication');
/**
 * Custom welcome messages or event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
 */
UserPoolOperation.POST_CONFIRMATION = new UserPoolOperation('postConfirmation');
/**
 * Custom validation to accept or deny the sign-in request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
 */
UserPoolOperation.PRE_AUTHENTICATION = new UserPoolOperation('preAuthentication');
/**
 * Custom validation to accept or deny the sign-up request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
 */
UserPoolOperation.PRE_SIGN_UP = new UserPoolOperation('preSignUp');
/**
 * Add or remove attributes in Id tokens
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html
 */
UserPoolOperation.PRE_TOKEN_GENERATION = new UserPoolOperation('preTokenGeneration');
/**
 * Migrate a user from an existing user directory to user pools
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html
 */
UserPoolOperation.USER_MIGRATION = new UserPoolOperation('userMigration');
/**
 * Determines if a response is correct in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
 */
UserPoolOperation.VERIFY_AUTH_CHALLENGE_RESPONSE = new UserPoolOperation('verifyAuthChallengeResponse');
/**
 * Amazon Cognito invokes this trigger to send email notifications to users.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-email-sender.html
 */
UserPoolOperation.CUSTOM_EMAIL_SENDER = new UserPoolOperation('customEmailSender');
/**
 * Amazon Cognito invokes this trigger to send email notifications to users.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-sms-sender.html
 */
UserPoolOperation.CUSTOM_SMS_SENDER = new UserPoolOperation('customSmsSender');
/**
 * The email verification style
 */
var VerificationEmailStyle;
(function (VerificationEmailStyle) {
    /** Verify email via code */
    VerificationEmailStyle["CODE"] = "CONFIRM_WITH_CODE";
    /** Verify email via link */
    VerificationEmailStyle["LINK"] = "CONFIRM_WITH_LINK";
})(VerificationEmailStyle = exports.VerificationEmailStyle || (exports.VerificationEmailStyle = {}));
/**
 * The different ways in which a user pool's MFA enforcement can be configured.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html
 */
var Mfa;
(function (Mfa) {
    /** Users are not required to use MFA for sign in, and cannot configure one. */
    Mfa["OFF"] = "OFF";
    /** Users are not required to use MFA for sign in, but can configure one if they so choose to. */
    Mfa["OPTIONAL"] = "OPTIONAL";
    /** Users are required to configure an MFA, and have to use it to sign in. */
    Mfa["REQUIRED"] = "ON";
})(Mfa = exports.Mfa || (exports.Mfa = {}));
/**
 * How will a user be able to recover their account?
 *
 * When a user forgets their password, they can have a code sent to their verified email or verified phone to recover their account.
 * You can choose the preferred way to send codes below.
 * We recommend not allowing phone to be used for both password resets and multi-factor authentication (MFA).
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-recover-a-user-account.html
 */
var AccountRecovery;
(function (AccountRecovery) {
    /**
     * Email if available, otherwise phone, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["EMAIL_AND_PHONE_WITHOUT_MFA"] = 0] = "EMAIL_AND_PHONE_WITHOUT_MFA";
    /**
     * Phone if available, otherwise email, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["PHONE_WITHOUT_MFA_AND_EMAIL"] = 1] = "PHONE_WITHOUT_MFA_AND_EMAIL";
    /**
     * Email only
     */
    AccountRecovery[AccountRecovery["EMAIL_ONLY"] = 2] = "EMAIL_ONLY";
    /**
     * Phone only, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["PHONE_ONLY_WITHOUT_MFA"] = 3] = "PHONE_ONLY_WITHOUT_MFA";
    /**
     * (Not Recommended) Phone if available, otherwise email, and do allow a user to reset their password via phone if they are also using it for MFA.
     */
    AccountRecovery[AccountRecovery["PHONE_AND_EMAIL"] = 4] = "PHONE_AND_EMAIL";
    /**
     * None – users will have to contact an administrator to reset their passwords
     */
    AccountRecovery[AccountRecovery["NONE"] = 5] = "NONE";
})(AccountRecovery = exports.AccountRecovery || (exports.AccountRecovery = {}));
class UserPoolBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.identityProviders = [];
    }
    addClient(id, options) {
        return new user_pool_client_1.UserPoolClient(this, id, {
            userPool: this,
            ...options,
        });
    }
    addDomain(id, options) {
        return new user_pool_domain_1.UserPoolDomain(this, id, {
            userPool: this,
            ...options,
        });
    }
    addResourceServer(id, options) {
        return new user_pool_resource_server_1.UserPoolResourceServer(this, id, {
            userPool: this,
            ...options,
        });
    }
    registerIdentityProvider(provider) {
        this.identityProviders.push(provider);
    }
}
/**
 * Define a Cognito User Pool
 */
class UserPool extends UserPoolBase {
    constructor(scope, id, props = {}) {
        var _c, _d, _e, _f, _g, _h;
        super(scope, id);
        this.triggers = {};
        jsiiDeprecationWarnings._aws_cdk_aws_cognito_UserPoolProps(props);
        const signIn = this.signInConfiguration(props);
        if (props.customSenderKmsKey) {
            const kmsKey = props.customSenderKmsKey;
            this.triggers.kmsKeyId = kmsKey.keyArn;
        }
        if (props.lambdaTriggers) {
            for (const t of Object.keys(props.lambdaTriggers)) {
                let trigger;
                switch (t) {
                    case 'customSmsSender':
                    case 'customEmailSender':
                        if (!this.triggers.kmsKeyId) {
                            throw new Error('you must specify a KMS key if you are using customSmsSender or customEmailSender.');
                        }
                        trigger = props.lambdaTriggers[t];
                        const version = 'V1_0';
                        if (trigger !== undefined) {
                            this.addLambdaPermission(trigger, t);
                            this.triggers[t] = {
                                lambdaArn: trigger.functionArn,
                                lambdaVersion: version,
                            };
                        }
                        break;
                    default:
                        trigger = props.lambdaTriggers[t];
                        if (trigger !== undefined) {
                            this.addLambdaPermission(trigger, t);
                            this.triggers[t] = trigger.functionArn;
                        }
                        break;
                }
            }
        }
        const verificationMessageTemplate = this.verificationMessageConfiguration(props);
        let emailVerificationMessage;
        let emailVerificationSubject;
        if (verificationMessageTemplate.defaultEmailOption === VerificationEmailStyle.CODE) {
            emailVerificationMessage = verificationMessageTemplate.emailMessage;
            emailVerificationSubject = verificationMessageTemplate.emailSubject;
        }
        const smsVerificationMessage = verificationMessageTemplate.smsMessage;
        const inviteMessageTemplate = {
            emailMessage: (_c = props.userInvitation) === null || _c === void 0 ? void 0 : _c.emailBody,
            emailSubject: (_d = props.userInvitation) === null || _d === void 0 ? void 0 : _d.emailSubject,
            smsMessage: (_e = props.userInvitation) === null || _e === void 0 ? void 0 : _e.smsMessage,
        };
        const selfSignUpEnabled = (_f = props.selfSignUpEnabled) !== null && _f !== void 0 ? _f : false;
        const adminCreateUserConfig = {
            allowAdminCreateUserOnly: !selfSignUpEnabled,
            inviteMessageTemplate: props.userInvitation !== undefined ? inviteMessageTemplate : undefined,
        };
        const passwordPolicy = this.configurePasswordPolicy(props);
        if (props.email && props.emailSettings) {
            throw new Error('you must either provide "email" or "emailSettings", but not both');
        }
        const emailConfiguration = props.email ? props.email._bind(this) : undefinedIfNoKeys({
            from: encodePuny((_g = props.emailSettings) === null || _g === void 0 ? void 0 : _g.from),
            replyToEmailAddress: encodePuny((_h = props.emailSettings) === null || _h === void 0 ? void 0 : _h.replyTo),
        });
        const userPool = new cognito_generated_1.CfnUserPool(this, 'Resource', {
            userPoolName: props.userPoolName,
            usernameAttributes: signIn.usernameAttrs,
            aliasAttributes: signIn.aliasAttrs,
            autoVerifiedAttributes: signIn.autoVerifyAttrs,
            lambdaConfig: core_1.Lazy.any({ produce: () => undefinedIfNoKeys(this.triggers) }),
            smsAuthenticationMessage: this.mfaMessage(props),
            smsConfiguration: this.smsConfiguration(props),
            adminCreateUserConfig,
            emailVerificationMessage,
            emailVerificationSubject,
            smsVerificationMessage,
            verificationMessageTemplate,
            schema: this.schemaConfiguration(props),
            mfaConfiguration: props.mfa,
            enabledMfas: this.mfaConfiguration(props),
            policies: passwordPolicy !== undefined ? { passwordPolicy } : undefined,
            emailConfiguration,
            usernameConfiguration: undefinedIfNoKeys({
                caseSensitive: props.signInCaseSensitive,
            }),
            accountRecoverySetting: this.accountRecovery(props),
            deviceConfiguration: props.deviceTracking,
        });
        userPool.applyRemovalPolicy(props.removalPolicy);
        this.userPoolId = userPool.ref;
        this.userPoolArn = userPool.attrArn;
        this.userPoolProviderName = userPool.attrProviderName;
        this.userPoolProviderUrl = userPool.attrProviderUrl;
    }
    /**
     * Import an existing user pool based on its id.
     */
    static fromUserPoolId(scope, id, userPoolId) {
        let userPoolArn = core_1.Stack.of(scope).formatArn({
            service: 'cognito-idp',
            resource: 'userpool',
            resourceName: userPoolId,
        });
        return UserPool.fromUserPoolArn(scope, id, userPoolArn);
    }
    /**
     * Import an existing user pool based on its ARN.
     */
    static fromUserPoolArn(scope, id, userPoolArn) {
        const arnParts = core_1.Stack.of(scope).splitArn(userPoolArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        if (!arnParts.resourceName) {
            throw new Error('invalid user pool ARN');
        }
        const userPoolId = arnParts.resourceName;
        class ImportedUserPool extends UserPoolBase {
            constructor() {
                super(scope, id, {
                    account: arnParts.account,
                    region: arnParts.region,
                });
                this.userPoolArn = userPoolArn;
                this.userPoolId = userPoolId;
            }
        }
        return new ImportedUserPool();
    }
    /**
     * Add a lambda trigger to a user pool operation
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html
     */
    addTrigger(operation, fn) {
        jsiiDeprecationWarnings._aws_cdk_aws_cognito_UserPoolOperation(operation);
        if (operation.operationName in this.triggers) {
            throw new Error(`A trigger for the operation ${operation.operationName} already exists.`);
        }
        this.addLambdaPermission(fn, operation.operationName);
        switch (operation.operationName) {
            case 'customEmailSender':
            case 'customSmsSender':
                if (!this.triggers.kmsKeyId) {
                    throw new Error('you must specify a KMS key if you are using customSmsSender or customEmailSender.');
                }
                this.triggers[operation.operationName] = {
                    lambdaArn: fn.functionArn,
                    lambdaVersion: 'V1_0',
                };
                break;
            default:
                this.triggers[operation.operationName] = fn.functionArn;
        }
    }
    addLambdaPermission(fn, name) {
        const capitalize = name.charAt(0).toUpperCase() + name.slice(1);
        fn.addPermission(`${capitalize}Cognito`, {
            principal: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com'),
            sourceArn: core_1.Lazy.string({ produce: () => this.userPoolArn }),
        });
    }
    mfaMessage(props) {
        const CODE_TEMPLATE = '{####}';
        const MAX_LENGTH = 140;
        const message = props.mfaMessage;
        if (message && !core_1.Token.isUnresolved(message)) {
            if (!message.includes(CODE_TEMPLATE)) {
                throw new Error(`MFA message must contain the template string '${CODE_TEMPLATE}'`);
            }
            if (message.length > MAX_LENGTH) {
                throw new Error(`MFA message must be between ${CODE_TEMPLATE.length} and ${MAX_LENGTH} characters`);
            }
        }
        return message;
    }
    verificationMessageConfiguration(props) {
        var _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
        const CODE_TEMPLATE = '{####}';
        const VERIFY_EMAIL_TEMPLATE = '{##Verify Email##}';
        const emailStyle = (_d = (_c = props.userVerification) === null || _c === void 0 ? void 0 : _c.emailStyle) !== null && _d !== void 0 ? _d : VerificationEmailStyle.CODE;
        const emailSubject = (_f = (_e = props.userVerification) === null || _e === void 0 ? void 0 : _e.emailSubject) !== null && _f !== void 0 ? _f : 'Verify your new account';
        const smsMessage = (_h = (_g = props.userVerification) === null || _g === void 0 ? void 0 : _g.smsMessage) !== null && _h !== void 0 ? _h : `The verification code to your new account is ${CODE_TEMPLATE}`;
        if (emailStyle === VerificationEmailStyle.CODE) {
            const emailMessage = (_k = (_j = props.userVerification) === null || _j === void 0 ? void 0 : _j.emailBody) !== null && _k !== void 0 ? _k : `The verification code to your new account is ${CODE_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${CODE_TEMPLATE}'`);
            }
            if (!core_1.Token.isUnresolved(smsMessage) && smsMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`SMS message must contain the template string '${CODE_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.CODE,
                emailMessage,
                emailSubject,
                smsMessage,
            };
        }
        else {
            const emailMessage = (_m = (_l = props.userVerification) === null || _l === void 0 ? void 0 : _l.emailBody) !== null && _m !== void 0 ? _m : `Verify your account by clicking on ${VERIFY_EMAIL_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(VERIFY_EMAIL_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${VERIFY_EMAIL_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.LINK,
                emailMessageByLink: emailMessage,
                emailSubjectByLink: emailSubject,
                smsMessage,
            };
        }
    }
    signInConfiguration(props) {
        var _c;
        let aliasAttrs;
        let usernameAttrs;
        let autoVerifyAttrs;
        const signIn = (_c = props.signInAliases) !== null && _c !== void 0 ? _c : { username: true };
        if (signIn.preferredUsername && !signIn.username) {
            throw new Error('username signIn must be enabled if preferredUsername is enabled');
        }
        if (signIn.username) {
            aliasAttrs = [];
            if (signIn.email) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
            if (signIn.preferredUsername) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.preferredUsername);
            }
            if (aliasAttrs.length === 0) {
                aliasAttrs = undefined;
            }
        }
        else {
            usernameAttrs = [];
            if (signIn.email) {
                usernameAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                usernameAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        if (props.autoVerify) {
            autoVerifyAttrs = [];
            if (props.autoVerify.email) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (props.autoVerify.phone) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        else if (signIn.email || signIn.phone) {
            autoVerifyAttrs = [];
            if (signIn.email) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        return { usernameAttrs, aliasAttrs, autoVerifyAttrs };
    }
    smsConfiguration(props) {
        var _c, _d, _e;
        if (props.enableSmsRole === false && props.smsRole) {
            throw new Error('enableSmsRole cannot be disabled when smsRole is specified');
        }
        if (props.smsRole) {
            return {
                snsCallerArn: props.smsRole.roleArn,
                externalId: props.smsRoleExternalId,
                snsRegion: props.snsRegion,
            };
        }
        if (props.enableSmsRole === false) {
            return undefined;
        }
        const mfaConfiguration = this.mfaConfiguration(props);
        const phoneVerification = ((_c = props.signInAliases) === null || _c === void 0 ? void 0 : _c.phone) === true || ((_d = props.autoVerify) === null || _d === void 0 ? void 0 : _d.phone) === true;
        const roleRequired = (mfaConfiguration === null || mfaConfiguration === void 0 ? void 0 : mfaConfiguration.includes('SMS_MFA')) || phoneVerification;
        if (!roleRequired && props.enableSmsRole === undefined) {
            return undefined;
        }
        const smsRoleExternalId = core_1.Names.uniqueId(this).slice(0, 1223); // sts:ExternalId max length of 1224
        const smsRole = (_e = props.smsRole) !== null && _e !== void 0 ? _e : new aws_iam_1.Role(this, 'smsRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com', {
                conditions: {
                    StringEquals: { 'sts:ExternalId': smsRoleExternalId },
                },
            }),
            inlinePolicies: {
                /*
                  * The UserPool is very particular that it must contain an 'sns:Publish' action as an inline policy.
                  * Ideally, a conditional that restricts this action to 'sms' protocol needs to be attached, but the UserPool deployment fails validation.
                  * Seems like a case of being excessively strict.
                  */
                'sns-publish': new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: ['sns:Publish'],
                            resources: ['*'],
                        }),
                    ],
                }),
            },
        });
        return {
            externalId: smsRoleExternalId,
            snsCallerArn: smsRole.roleArn,
            snsRegion: props.snsRegion,
        };
    }
    mfaConfiguration(props) {
        if (props.mfa === undefined || props.mfa === Mfa.OFF) {
            // since default is OFF, treat undefined and OFF the same way
            return undefined;
        }
        else if (props.mfaSecondFactor === undefined &&
            (props.mfa === Mfa.OPTIONAL || props.mfa === Mfa.REQUIRED)) {
            return ['SMS_MFA'];
        }
        else {
            const enabledMfas = [];
            if (props.mfaSecondFactor.sms) {
                enabledMfas.push('SMS_MFA');
            }
            if (props.mfaSecondFactor.otp) {
                enabledMfas.push('SOFTWARE_TOKEN_MFA');
            }
            return enabledMfas;
        }
    }
    configurePasswordPolicy(props) {
        var _c, _d, _e, _f, _g, _h;
        const tempPasswordValidity = (_c = props.passwordPolicy) === null || _c === void 0 ? void 0 : _c.tempPasswordValidity;
        if (tempPasswordValidity !== undefined && tempPasswordValidity.toDays() > core_1.Duration.days(365).toDays()) {
            throw new Error(`tempPasswordValidity cannot be greater than 365 days (received: ${tempPasswordValidity.toDays()})`);
        }
        const minLength = props.passwordPolicy ? (_d = props.passwordPolicy.minLength) !== null && _d !== void 0 ? _d : 8 : undefined;
        if (minLength !== undefined && (minLength < 6 || minLength > 99)) {
            throw new Error(`minLength for password must be between 6 and 99 (received: ${minLength})`);
        }
        return undefinedIfNoKeys({
            temporaryPasswordValidityDays: tempPasswordValidity === null || tempPasswordValidity === void 0 ? void 0 : tempPasswordValidity.toDays({ integral: true }),
            minimumLength: minLength,
            requireLowercase: (_e = props.passwordPolicy) === null || _e === void 0 ? void 0 : _e.requireLowercase,
            requireUppercase: (_f = props.passwordPolicy) === null || _f === void 0 ? void 0 : _f.requireUppercase,
            requireNumbers: (_g = props.passwordPolicy) === null || _g === void 0 ? void 0 : _g.requireDigits,
            requireSymbols: (_h = props.passwordPolicy) === null || _h === void 0 ? void 0 : _h.requireSymbols,
        });
    }
    schemaConfiguration(props) {
        const schema = [];
        if (props.standardAttributes) {
            const stdAttributes = Object.entries(props.standardAttributes)
                .filter(([, attr]) => !!attr)
                .map(([attrName, attr]) => {
                var _c, _d;
                return ({
                    name: attr_names_1.StandardAttributeNames[attrName],
                    mutable: (_c = attr.mutable) !== null && _c !== void 0 ? _c : true,
                    required: (_d = attr.required) !== null && _d !== void 0 ? _d : false,
                });
            });
            schema.push(...stdAttributes);
        }
        if (props.customAttributes) {
            const customAttrs = Object.keys(props.customAttributes).map((attrName) => {
                var _c, _d, _e, _f, _g, _h, _j, _k;
                const attrConfig = props.customAttributes[attrName].bind();
                const numberConstraints = {
                    minValue: (_d = (_c = attrConfig.numberConstraints) === null || _c === void 0 ? void 0 : _c.min) === null || _d === void 0 ? void 0 : _d.toString(),
                    maxValue: (_f = (_e = attrConfig.numberConstraints) === null || _e === void 0 ? void 0 : _e.max) === null || _f === void 0 ? void 0 : _f.toString(),
                };
                const stringConstraints = {
                    minLength: (_h = (_g = attrConfig.stringConstraints) === null || _g === void 0 ? void 0 : _g.minLen) === null || _h === void 0 ? void 0 : _h.toString(),
                    maxLength: (_k = (_j = attrConfig.stringConstraints) === null || _j === void 0 ? void 0 : _j.maxLen) === null || _k === void 0 ? void 0 : _k.toString(),
                };
                return {
                    name: attrName,
                    attributeDataType: attrConfig.dataType,
                    numberAttributeConstraints: attrConfig.numberConstraints
                        ? numberConstraints
                        : undefined,
                    stringAttributeConstraints: attrConfig.stringConstraints
                        ? stringConstraints
                        : undefined,
                    mutable: attrConfig.mutable,
                };
            });
            schema.push(...customAttrs);
        }
        if (schema.length === 0) {
            return undefined;
        }
        return schema;
    }
    accountRecovery(props) {
        var _c;
        const accountRecovery = (_c = props.accountRecovery) !== null && _c !== void 0 ? _c : AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL;
        switch (accountRecovery) {
            case AccountRecovery.EMAIL_AND_PHONE_WITHOUT_MFA:
                return {
                    recoveryMechanisms: [
                        { name: 'verified_email', priority: 1 },
                        { name: 'verified_phone_number', priority: 2 },
                    ],
                };
            case AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL:
                return {
                    recoveryMechanisms: [
                        { name: 'verified_phone_number', priority: 1 },
                        { name: 'verified_email', priority: 2 },
                    ],
                };
            case AccountRecovery.EMAIL_ONLY:
                return {
                    recoveryMechanisms: [{ name: 'verified_email', priority: 1 }],
                };
            case AccountRecovery.PHONE_ONLY_WITHOUT_MFA:
                return {
                    recoveryMechanisms: [{ name: 'verified_phone_number', priority: 1 }],
                };
            case AccountRecovery.NONE:
                return {
                    recoveryMechanisms: [{ name: 'admin_only', priority: 1 }],
                };
            case AccountRecovery.PHONE_AND_EMAIL:
                return undefined;
            default:
                throw new Error(`Unsupported AccountRecovery type - ${accountRecovery}`);
        }
    }
}
exports.UserPool = UserPool;
_b = JSII_RTTI_SYMBOL_1;
UserPool[_b] = { fqn: "@aws-cdk/aws-cognito.UserPool", version: "1.152.0" };
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).every(val => val === undefined);
    return allUndefined ? undefined : struct;
}
function encodePuny(input) {
    return input !== undefined ? punycode_1.toASCII(input) : input;
}
//# sourceMappingURL=data:application/json;base64,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