"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolClient = exports.UserPoolClientIdentityProvider = exports.OAuthScope = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
/**
 * OAuth scopes that are allowed with this client.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 */
class OAuthScope {
    constructor(scopeName) {
        this.scopeName = scopeName;
    }
    /**
     * Custom scope is one that you define for your own resource server in the Resource Servers.
     * The format is 'resource-server-identifier/scope'.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
     */
    static custom(name) {
        return new OAuthScope(name);
    }
    /**
     * Adds a custom scope that's tied to a resource server in your stack
     */
    static resourceServer(server, scope) {
        jsiiDeprecationWarnings._aws_cdk_aws_cognito_IUserPoolResourceServer(server);
        jsiiDeprecationWarnings._aws_cdk_aws_cognito_ResourceServerScope(scope);
        return new OAuthScope(`${server.userPoolResourceServerId}/${scope.scopeName}`);
    }
}
exports.OAuthScope = OAuthScope;
_a = JSII_RTTI_SYMBOL_1;
OAuthScope[_a] = { fqn: "@aws-cdk/aws-cognito.OAuthScope", version: "1.152.0" };
/**
 * Grants access to the 'phone_number' and 'phone_number_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PHONE = new OAuthScope('phone');
/**
 * Grants access to the 'email' and 'email_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.EMAIL = new OAuthScope('email');
/**
 * Returns all user attributes in the ID token that are readable by the client
 */
OAuthScope.OPENID = new OAuthScope('openid');
/**
 * Grants access to all user attributes that are readable by the client
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PROFILE = new OAuthScope('profile');
/**
 * Grants access to Amazon Cognito User Pool API operations that require access tokens,
 * such as UpdateUserAttributes and VerifyUserAttribute.
 */
OAuthScope.COGNITO_ADMIN = new OAuthScope('aws.cognito.signin.user.admin');
/**
 * Identity providers supported by the UserPoolClient
 */
class UserPoolClientIdentityProvider {
    constructor(name) {
        this.name = name;
    }
    /**
     * Specify a provider not yet supported by the CDK.
     * @param name name of the identity provider as recognized by CloudFormation property `SupportedIdentityProviders`
     */
    static custom(name) {
        return new UserPoolClientIdentityProvider(name);
    }
}
exports.UserPoolClientIdentityProvider = UserPoolClientIdentityProvider;
_b = JSII_RTTI_SYMBOL_1;
UserPoolClientIdentityProvider[_b] = { fqn: "@aws-cdk/aws-cognito.UserPoolClientIdentityProvider", version: "1.152.0" };
/**
 * Allow users to sign in using 'Sign In With Apple'.
 * A `UserPoolIdentityProviderApple` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.APPLE = new UserPoolClientIdentityProvider('SignInWithApple');
/**
 * Allow users to sign in using 'Facebook Login'.
 * A `UserPoolIdentityProviderFacebook` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.FACEBOOK = new UserPoolClientIdentityProvider('Facebook');
/**
 * Allow users to sign in using 'Google Login'.
 * A `UserPoolIdentityProviderGoogle` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.GOOGLE = new UserPoolClientIdentityProvider('Google');
/**
 * Allow users to sign in using 'Login With Amazon'.
 * A `UserPoolIdentityProviderAmazon` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.AMAZON = new UserPoolClientIdentityProvider('LoginWithAmazon');
/**
 * Allow users to sign in directly as a user of the User Pool
 */
UserPoolClientIdentityProvider.COGNITO = new UserPoolClientIdentityProvider('COGNITO');
/**
 * Define a UserPool App Client
 */
class UserPoolClient extends core_1.Resource {
    /*
     * Note to implementers: Two CloudFormation return values Name and ClientSecret are part of the spec.
     * However, they have been explicity not implemented here. They are not documented in CloudFormation, and
     * CloudFormation returns the following the string when these two attributes are 'GetAtt' - "attribute not supported
     * at this time, please use the CLI or Console to retrieve this value".
     * Awaiting updates from CloudFormation.
     */
    constructor(scope, id, props) {
        var _d, _e, _f, _g, _h, _j;
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_cognito_UserPoolClientProps(props);
        if (props.disableOAuth && props.oAuth) {
            throw new Error('OAuth settings cannot be specified when disableOAuth is set.');
        }
        this.oAuthFlows = (_e = (_d = props.oAuth) === null || _d === void 0 ? void 0 : _d.flows) !== null && _e !== void 0 ? _e : {
            implicitCodeGrant: true,
            authorizationCodeGrant: true,
        };
        let callbackUrls = (_f = props.oAuth) === null || _f === void 0 ? void 0 : _f.callbackUrls;
        if (this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) {
            if (callbackUrls === undefined) {
                callbackUrls = ['https://example.com'];
            }
            else if (callbackUrls.length === 0) {
                throw new Error('callbackUrl must not be empty when codeGrant or implicitGrant OAuth flows are enabled.');
            }
        }
        const resource = new cognito_generated_1.CfnUserPoolClient(this, 'Resource', {
            clientName: props.userPoolClientName,
            generateSecret: props.generateSecret,
            userPoolId: props.userPool.userPoolId,
            explicitAuthFlows: this.configureAuthFlows(props),
            allowedOAuthFlows: props.disableOAuth ? undefined : this.configureOAuthFlows(),
            allowedOAuthScopes: props.disableOAuth ? undefined : this.configureOAuthScopes(props.oAuth),
            callbackUrLs: callbackUrls && callbackUrls.length > 0 && !props.disableOAuth ? callbackUrls : undefined,
            logoutUrLs: (_g = props.oAuth) === null || _g === void 0 ? void 0 : _g.logoutUrls,
            allowedOAuthFlowsUserPoolClient: !props.disableOAuth,
            preventUserExistenceErrors: this.configurePreventUserExistenceErrors(props.preventUserExistenceErrors),
            supportedIdentityProviders: this.configureIdentityProviders(props),
            readAttributes: (_h = props.readAttributes) === null || _h === void 0 ? void 0 : _h.attributes(),
            writeAttributes: (_j = props.writeAttributes) === null || _j === void 0 ? void 0 : _j.attributes(),
            enableTokenRevocation: props.enableTokenRevocation,
        });
        this.configureTokenValidity(resource, props);
        this.userPoolClientId = resource.ref;
        this._userPoolClientName = props.userPoolClientName;
    }
    /**
     * Import a user pool client given its id.
     */
    static fromUserPoolClientId(scope, id, userPoolClientId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolClientId = userPoolClientId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The client name that was specified via the `userPoolClientName` property during initialization,
     * throws an error otherwise.
     */
    get userPoolClientName() {
        if (this._userPoolClientName === undefined) {
            throw new Error('userPoolClientName is available only if specified on the UserPoolClient during initialization');
        }
        return this._userPoolClientName;
    }
    configureAuthFlows(props) {
        if (!props.authFlows)
            return undefined;
        const authFlows = [];
        if (props.authFlows.userPassword) {
            authFlows.push('ALLOW_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.adminUserPassword) {
            authFlows.push('ALLOW_ADMIN_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.custom) {
            authFlows.push('ALLOW_CUSTOM_AUTH');
        }
        if (props.authFlows.userSrp) {
            authFlows.push('ALLOW_USER_SRP_AUTH');
        }
        // refreshToken should always be allowed if authFlows are present
        if (authFlows.length > 0) {
            authFlows.push('ALLOW_REFRESH_TOKEN_AUTH');
        }
        if (authFlows.length === 0) {
            return undefined;
        }
        return authFlows;
    }
    configureOAuthFlows() {
        if ((this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) && this.oAuthFlows.clientCredentials) {
            throw new Error('clientCredentials OAuth flow cannot be selected along with codeGrant or implicitGrant.');
        }
        const oAuthFlows = [];
        if (this.oAuthFlows.clientCredentials) {
            oAuthFlows.push('client_credentials');
        }
        if (this.oAuthFlows.implicitCodeGrant) {
            oAuthFlows.push('implicit');
        }
        if (this.oAuthFlows.authorizationCodeGrant) {
            oAuthFlows.push('code');
        }
        if (oAuthFlows.length === 0) {
            return undefined;
        }
        return oAuthFlows;
    }
    configureOAuthScopes(oAuth) {
        var _d;
        const scopes = (_d = oAuth === null || oAuth === void 0 ? void 0 : oAuth.scopes) !== null && _d !== void 0 ? _d : [OAuthScope.PROFILE, OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.OPENID,
            OAuthScope.COGNITO_ADMIN];
        const scopeNames = new Set(scopes.map((x) => x.scopeName));
        const autoOpenIdScopes = [OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.PROFILE];
        if (autoOpenIdScopes.reduce((agg, s) => agg || scopeNames.has(s.scopeName), false)) {
            scopeNames.add(OAuthScope.OPENID.scopeName);
        }
        return Array.from(scopeNames);
    }
    configurePreventUserExistenceErrors(prevent) {
        if (prevent === undefined) {
            return undefined;
        }
        return prevent ? 'ENABLED' : 'LEGACY';
    }
    configureIdentityProviders(props) {
        let providers;
        if (!props.supportedIdentityProviders) {
            const providerSet = new Set(props.userPool.identityProviders.map((p) => p.providerName));
            providerSet.add('COGNITO');
            providers = Array.from(providerSet);
        }
        else {
            providers = props.supportedIdentityProviders.map((p) => p.name);
        }
        if (providers.length === 0) {
            return undefined;
        }
        return Array.from(providers);
    }
    configureTokenValidity(resource, props) {
        this.validateDuration('idTokenValidity', core_1.Duration.minutes(5), core_1.Duration.days(1), props.idTokenValidity);
        this.validateDuration('accessTokenValidity', core_1.Duration.minutes(5), core_1.Duration.days(1), props.accessTokenValidity);
        this.validateDuration('refreshTokenValidity', core_1.Duration.minutes(60), core_1.Duration.days(10 * 365), props.refreshTokenValidity);
        if (props.refreshTokenValidity) {
            this.validateDuration('idTokenValidity', core_1.Duration.minutes(5), props.refreshTokenValidity, props.idTokenValidity);
            this.validateDuration('accessTokenValidity', core_1.Duration.minutes(5), props.refreshTokenValidity, props.accessTokenValidity);
        }
        if (props.accessTokenValidity || props.idTokenValidity || props.refreshTokenValidity) {
            resource.tokenValidityUnits = {
                idToken: props.idTokenValidity ? 'minutes' : undefined,
                accessToken: props.accessTokenValidity ? 'minutes' : undefined,
                refreshToken: props.refreshTokenValidity ? 'minutes' : undefined,
            };
        }
        ;
        resource.idTokenValidity = props.idTokenValidity ? props.idTokenValidity.toMinutes() : undefined;
        resource.refreshTokenValidity = props.refreshTokenValidity ? props.refreshTokenValidity.toMinutes() : undefined;
        resource.accessTokenValidity = props.accessTokenValidity ? props.accessTokenValidity.toMinutes() : undefined;
    }
    validateDuration(name, min, max, value) {
        if (value === undefined) {
            return;
        }
        if (value.toMilliseconds() < min.toMilliseconds() || value.toMilliseconds() > max.toMilliseconds()) {
            throw new Error(`${name}: Must be a duration between ${min.toHumanString()} and ${max.toHumanString()} (inclusive); received ${value.toHumanString()}.`);
        }
    }
}
exports.UserPoolClient = UserPoolClient;
_c = JSII_RTTI_SYMBOL_1;
UserPoolClient[_c] = { fqn: "@aws-cdk/aws-cognito.UserPoolClient", version: "1.152.0" };
//# sourceMappingURL=data:application/json;base64,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