#  This file is part of Pynguin.
#
#  SPDX-FileCopyrightText: 2019–2022 Pynguin Contributors
#
#  SPDX-License-Identifier: LGPL-3.0-or-later
#
"""Provides an assertion generator"""
from __future__ import annotations

import ast
import dataclasses
import logging
import threading
import types
from typing import TYPE_CHECKING

import mutpy
from ordered_set import OrderedSet

import pynguin.assertion.assertion as ass
import pynguin.assertion.assertion_trace as at
import pynguin.assertion.assertiontraceobserver as ato
import pynguin.assertion.mutation_analysis.mutationadapter as ma
import pynguin.configuration as config
import pynguin.ga.chromosomevisitor as cv
import pynguin.testcase.execution as ex
import pynguin.utils.statistics.statistics as stat
from pynguin.analyses.constants import (
    ConstantPool,
    DynamicConstantProvider,
    EmptyConstantProvider,
)
from pynguin.instrumentation.machinery import build_transformer
from pynguin.utils import randomness
from pynguin.utils.statistics.runtimevariable import RuntimeVariable

if TYPE_CHECKING:
    import pynguin.ga.testcasechromosome as tcc
    import pynguin.ga.testsuitechromosome as tsc
    import pynguin.testcase.testcase as tc

_LOGGER = logging.getLogger(__name__)


class AssertionGenerator(cv.ChromosomeVisitor):
    """A simple assertion generator.
    Creates all regression assertions."""

    _logger = logging.getLogger(__name__)

    def __init__(
        self, plain_executor: ex.TestCaseExecutor, filtering_executions: int = 1
    ):
        """
        Create new assertion generator.

        Args:
            plain_executor: The executor that is used to execute on the non mutated
                module.
            filtering_executions: How often should the tests be executed to filter
                out trivially flaky assertions, e.g., str representations based on
                memory locations.
        """
        self._filtering_executions = filtering_executions
        self._plain_executor = plain_executor

    def visit_test_suite_chromosome(self, chromosome: tsc.TestSuiteChromosome) -> None:
        self._add_assertions(
            [chrom.test_case for chrom in chromosome.test_case_chromosomes]
        )

    def visit_test_case_chromosome(self, chromosome: tcc.TestCaseChromosome) -> None:
        self._add_assertions([chromosome.test_case])

    def _add_assertions(self, test_cases: list[tc.TestCase]):
        # First run of executions to add assertions
        with self._plain_executor.temporarily_add_observer(
            ato.AssertionTraceObserver()
        ):
            for test in test_cases:
                self._add_assertions_for(test, self._plain_executor.execute(test))

        # Perform filtering executions to remove trivially flaky assertions.
        with self._plain_executor.temporarily_add_observer(
            ato.AssertionVerificationObserver()
        ):
            for _ in range(self._filtering_executions):
                # Create a copy of the list that is shuffled.
                shuffled_copy = list(test_cases)
                randomness.RNG.shuffle(shuffled_copy)
                for test in shuffled_copy:
                    self.__remove_non_holding_assertions(
                        test, self._plain_executor.execute(test)
                    )

    @staticmethod
    def __remove_non_holding_assertions(test: tc.TestCase, result: ex.ExecutionResult):
        for idx, statement in enumerate(test.statements):
            # Remove failed assertions
            if idx in result.assertion_verification_trace.failed:
                for pos in sorted(
                    result.assertion_verification_trace.failed[idx], reverse=True
                ):
                    statement.assertions.pop(pos)
            # Remove assertions that caused an error.
            if idx in result.assertion_verification_trace.error:
                for pos in sorted(
                    result.assertion_verification_trace.error[idx], reverse=True
                ):
                    statement.assertions.pop(pos)

    def _add_assertions_for(self, test_case: tc.TestCase, result: ex.ExecutionResult):
        # In order to avoid repeating the same assertions after each statement,
        # we keep track of the last assertions and only assert things, if they
        # have changed.
        previous_statement_assertions: OrderedSet[ass.Assertion] = OrderedSet()
        for statement in test_case.statements:
            current_statement_assertions = result.assertion_trace.get_assertions(
                statement
            )
            for assertion in current_statement_assertions:
                if (
                    not config.configuration.test_case_output.allow_stale_assertions
                    and assertion in previous_statement_assertions
                ):
                    # We already saw the same assertion in the previous statement
                    # So the value did not change.
                    continue
                if (
                    test_case.size_with_assertions()
                    >= config.configuration.test_case_output.max_length_test_case
                ):
                    self._logger.debug(
                        "No more assertions are added, because the maximum length "
                        "of a test case with its assertions was reached"
                    )
                    return
                statement.add_assertion(assertion)

            # Only update the previously seen assertions when we encounter a
            # statement that actually affects assertions.
            if statement.affects_assertions:
                previous_statement_assertions = current_statement_assertions


@dataclasses.dataclass
class _MutantInfo:
    """Collect data about a single mutant"""

    # Number of the mutant.
    mut_num: int

    # Did the mutant cause a timeout?
    timed_out_by: list[int] = dataclasses.field(default_factory=list)

    # Was the mutant killed by any test?
    killed_by: list[int] = dataclasses.field(default_factory=list)


@dataclasses.dataclass
class _MutationSummary:
    """Summary about mutation."""

    mutant_information: list[_MutantInfo] = dataclasses.field(default_factory=list)

    def get_survived(self) -> list[_MutantInfo]:
        """Get survived Mutants

        Returns:
            The survived mutants
        """
        return [
            info
            for info in self.mutant_information
            if not info.killed_by and not info.timed_out_by
        ]

    def get_killed(self) -> list[_MutantInfo]:
        """Get killed Mutants

        Returns:
            The killed mutants
        """
        return [
            info
            for info in self.mutant_information
            if info.killed_by and not info.timed_out_by
        ]

    def get_timeout(self) -> list[_MutantInfo]:
        """Get timed out Mutants

        Returns:
            The timed out mutants
        """
        return [info for info in self.mutant_information if info.timed_out_by]

    def get_metrics(self) -> _MutationMetrics:
        """Provide mutation metrics

        Returns:
            The mutation metrics.
        """
        return _MutationMetrics(
            num_created_mutants=len(self.mutant_information),
            num_killed_mutants=len(self.get_killed()),
            num_timeout_mutants=len(self.get_timeout()),
        )


@dataclasses.dataclass
class _MutationMetrics:
    num_created_mutants: int
    num_killed_mutants: int
    num_timeout_mutants: int

    def get_score(self) -> float:
        """Computes the mutation score

        Returns:
            The mutation score
        """
        divisor = self.num_created_mutants - self.num_timeout_mutants
        assert divisor >= 0
        if divisor == 0:
            # No mutants -> all mutants covered.
            return 1.0
        return self.num_killed_mutants / divisor


class MutationAnalysisAssertionGenerator(AssertionGenerator):

    """Uses mutation analysis to filter out less relevant assertions."""

    def _create_module_with_instrumentation(
        self, ast_node, module_name="mutant", module_dict=None
    ):
        # Mimics mutpy.utils.create_module but adds instrumentation to the resulting
        # module
        code = compile(ast_node, module_name, "exec")
        if self._testing:
            self._testing_created_mutants.append(ast.unparse(ast_node))
        code = self._transformer.instrument_module(code)
        module = types.ModuleType(module_name)
        module.__dict__.update(module_dict or {})
        # pylint: disable=exec-used
        exec(code, module.__dict__)  # nosec
        return module

    def __init__(self, plain_executor: ex.TestCaseExecutor, testing: bool = False):
        """

        Args:
            plain_executor: Executor used for plain execution
            testing: Enable test mode, currently required for integration testing.
        """
        super().__init__(plain_executor)

        # We use a separate tracer and executor to execute tests on the mutants.
        self._mutation_tracer = ex.ExecutionTracer()
        self._mutation_tracer.current_thread_identifier = (
            threading.current_thread().ident
        )
        self._mutation_executor = ex.TestCaseExecutor(self._mutation_tracer)
        self._mutation_executor.add_observer(ato.AssertionVerificationObserver())

        self._transformer = build_transformer(
            self._mutation_tracer,
            {config.CoverageMetric.BRANCH},
            DynamicConstantProvider(ConstantPool(), EmptyConstantProvider(), 0, 1),
        )
        # Some debug information
        self._testing = testing
        self._testing_created_mutants: list[str] = []
        self._testing_mutation_summary: _MutationSummary = _MutationSummary()
        adapter = ma.MutationAdapter()

        # Evil hack to change the way mutpy creates mutated modules.
        mutpy.utils.create_module = self._create_module_with_instrumentation
        self._mutated_modules = [x for x, _ in adapter.mutate_module()]

    def _add_assertions(self, test_cases: list[tc.TestCase]):
        super()._add_assertions(test_cases)
        tests_and_results: list[tuple[tc.TestCase, list[ex.ExecutionResult]]] = [
            (test, []) for test in test_cases
        ]

        with self._mutation_executor.temporarily_add_observer(
            ato.AssertionVerificationObserver()
        ):
            for idx, mutated_module in enumerate(self._mutated_modules):
                self._logger.info(
                    "Running tests on mutant %3i/%i",
                    idx + 1,
                    len(self._mutated_modules),
                )
                self._mutation_executor.module_provider.add_mutated_version(
                    module_name=config.configuration.module_name,
                    mutated_module=mutated_module,
                )
                for test, results in tests_and_results:
                    results.append(self._mutation_executor.execute(test))

        summary = self.__compute_mutation_summary(
            len(self._mutated_modules), tests_and_results
        )
        self.__report_mutation_summary(summary)
        self.__remove_non_relevant_assertions(tests_and_results, summary)

    @staticmethod
    def __remove_non_relevant_assertions(
        tests_and_results: list[tuple[tc.TestCase, list[ex.ExecutionResult]]],
        mutation_summary: _MutationSummary,
    ) -> None:
        for test, results in tests_and_results:
            merged = at.AssertionVerificationTrace()
            for result, mut in zip(
                results, mutation_summary.mutant_information, strict=True
            ):
                # Ignore timed out executions
                if len(mut.timed_out_by) == 0:
                    merged.merge(result.assertion_verification_trace)
            for stmt_idx, statement in enumerate(test.statements):
                for assertion_idx in reversed(range(len(statement.assertions))):
                    if not merged.was_violated(stmt_idx, assertion_idx):
                        # TODO(fk) works with expected exceptions?
                        statement.assertions.pop(assertion_idx)

    @staticmethod
    def __compute_mutation_summary(
        number_of_mutants: int,
        tests_and_results: list[tuple[tc.TestCase, list[ex.ExecutionResult]]],
    ) -> _MutationSummary:
        mutation_info = [_MutantInfo(i) for i in range(number_of_mutants)]
        for test_num, (_, results) in enumerate(tests_and_results):
            # For each mutation, check if we had a violated assertion
            for info, result in zip(mutation_info, results, strict=True):
                if info.timed_out_by:
                    continue
                if result.timeout:
                    # Mutant caused timeout
                    info.timed_out_by.append(test_num)
                elif (
                    len(result.assertion_verification_trace.error) > 0
                    or len(result.assertion_verification_trace.failed) > 0
                    or result.has_test_exceptions()
                    # Execution with assertions should not raise exceptions.
                    # If it does, it is probably an incompetent mutant
                ):
                    info.killed_by.append(test_num)
        return _MutationSummary(mutation_info)

    def __report_mutation_summary(self, mutation_summary: _MutationSummary):
        if self._testing:
            self._testing_mutation_summary = mutation_summary
        metrics = mutation_summary.get_metrics()
        stat.track_output_variable(
            RuntimeVariable.NumberOfKilledMutants, metrics.num_killed_mutants
        )
        stat.track_output_variable(
            RuntimeVariable.NumberOfTimedOutMutants, metrics.num_timeout_mutants
        )
        stat.track_output_variable(
            RuntimeVariable.NumberOfCreatedMutants, metrics.num_created_mutants
        )
        stat.track_output_variable(RuntimeVariable.MutationScore, metrics.get_score())

        for info in mutation_summary.mutant_information:
            if info.killed_by:
                _LOGGER.info(
                    "Mutant %i killed by Test(s): %s",
                    info.mut_num,
                    ", ".join(map(str, info.killed_by)),
                )
            elif info.timed_out_by:
                _LOGGER.info(
                    "Mutant %i timed out. First time with test %i.",
                    info.mut_num,
                    info.timed_out_by[0],
                )
        survived = mutation_summary.get_survived()
        _LOGGER.info(
            "Number of Surviving Mutant(s): %i (Mutants: %s)",
            len(survived),
            ", ".join(map(lambda x: str(x.mut_num), survived)),
        )
