from generic_camunda_client import Configuration, ApiClient, VariableValueDto
from typing import Dict, Any
import os
import base64


class CamundaResources:
    """
    Singleton containing resources shared by Camunda sub libraries
    """
    _instance = None

    _client_configuration: Configuration = None

    _api_client: ApiClient = None

    def __new__(cls):
        if cls._instance is None:
            print('Creating the object')
            cls._instance = super(CamundaResources, cls).__new__(cls)
            # Put any initialization here.
        return cls._instance

    # cammunda_url parameter is only required as long not every keyword uses generic-camunda-client
    @property
    def camunda_url(self) -> str:
        return self.client_configuration.host

    @camunda_url.setter
    def camunda_url(self, value: str):
        if not self._client_configuration:
            self.client_configuration = Configuration(host=value)
        else:
            self.client_configuration.host = value

    @property
    def client_configuration(self) -> Configuration:
        return self._client_configuration

    @client_configuration.setter
    def client_configuration(self, value):
        self._client_configuration = value
        if self._api_client:
            self.api_client

    @property
    def api_client(self) -> ApiClient:
        if not self._api_client:
            self._api_client = self._create_task_client()
        return self._api_client

    def _create_task_client(self) -> ApiClient:
        if not self.client_configuration:
            raise ValueError('No URL to camunda set. Please initialize Library with url or use keyword '
                             '"Set Camunda URL" first.')

        return ApiClient(self.client_configuration)

    @staticmethod
    def convert_openapi_variables_to_dict(open_api_variables: Dict[str, VariableValueDto]) -> Dict:
        """
        Converts the variables to a simple dictionary
        :return: dict
            {"var1": {"value": 1}, "var2": {"value": True}}
            ->
            {"var1": 1, "var2": True}
        """
        if not open_api_variables:
            return {}
        return {k: CamundaResources.convert_variable_dto(v) for (k, v) in open_api_variables.items()}

    @staticmethod
    def convert_dict_to_openapi_variables(variabes: dict) -> Dict[str,VariableValueDto]:
        """
        Converts the variables to a simple dictionary
        :return: dict
            {"var1": 1, "var2": True}
            ->
            {'var1': {'type': None, 'value': 1, 'value_info': None}, 'var2': {'type': None, 'value': True, 'value_info': None}}

        Example:
        >>> CamundaResources.convert_dict_to_openapi_variables({"var1": 1, "var2": True})
        {'var1': {'type': None, 'value': 1, 'value_info': None}, 'var2': {'type': None, 'value': True, 'value_info': None}}

        >>> CamundaResources.convert_dict_to_openapi_variables({})
        {}
        """
        if not variabes:
            return {}
        return {k: CamundaResources.convert_to_variable_dto(v) for (k, v) in variabes.items()}

    @staticmethod
    def convert_file_dict_to_openapi_variables(files: Dict[str, str]) -> Dict[str, VariableValueDto]:
        """
        Example:
        >>> CamundaResources.convert_file_dict_to_openapi_variables({'testfile': 'tests/resources/test.txt'})
        {'testfile': {'type': 'File',
         'value': 'VGhpcyBpcyBhIHRlc3QgZmlsZSBmb3IgYSBjYW11bmRhIHByb2Nlc3Mu',
         'value_info': {'filename': 'test.txt', 'mimetype': 'text/plain'}}}

        >>> CamundaResources.convert_file_dict_to_openapi_variables({})
        {}
        """
        if not files:
            return {}
        return {k: CamundaResources.convert_file_to_dto(v) for (k, v) in files.items()}

    @staticmethod
    def convert_file_to_dto(path: str) -> VariableValueDto:
        if not path:
            raise FileNotFoundError('Cannot create DTO from file, because no file provided')

        with open(path, 'r+b') as file:
            file_content = base64.standard_b64encode(file.read()).decode('utf-8')

        base = os.path.basename(path)
        file_name, file_ext = os.path.splitext(base)

        if file_ext.lower() in ['.jpg', '.jpeg', '.jpe']:
            mimetype = 'image/jpeg'
        elif file_ext.lower() in ['.png']:
            mimetype = 'image/png'
        elif file_ext.lower() in ['.pdf']:
            mimetype = 'application/pdf'
        elif file_ext.lower() in ['.txt']:
            mimetype = 'text/plain'
        else:
            mimetype = 'application/octet-stream'
        return VariableValueDto(value=file_content, type='File', value_info={'filename': base, 'mimetype': mimetype})

    @staticmethod
    def convert_to_variable_dto(value: Any) -> VariableValueDto:
        return VariableValueDto(value=value)

    @staticmethod
    def convert_variable_dto(dto: VariableValueDto) -> Any:
        if dto.type == 'File':
            return dto.to_dict()
        return dto.value

if __name__ == '__main__':
    import doctest
    import xmlrunner

    suite = doctest.DocTestSuite()
    xmlrunner.XMLTestRunner(output='logs').run(suite)