"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.RelationalDBTemplateGenerator = void 0;
const ResourceConstants_1 = require("./ResourceConstants");
const dataSource_1 = __importDefault(require("cloudform-types/types/appSync/dataSource"));
const iam_1 = __importDefault(require("cloudform-types/types/iam"));
const cloudform_1 = __importDefault(require("cloudform"));
const cloudform_types_1 = require("cloudform-types");
const RelationalDBResolverGenerator_1 = require("./RelationalDBResolverGenerator");
class RelationalDBTemplateGenerator {
    constructor(context) {
        this.context = context;
    }
    createTemplate(context) {
        const template = {
            AWSTemplateFormatVersion: '2010-09-09',
            Parameters: this.makeParameters(this.context.databaseName),
            Resources: {
                [ResourceConstants_1.ResourceConstants.RESOURCES.RelationalDatabaseDataSource]: this.makeRelationalDataSource(context),
                [ResourceConstants_1.ResourceConstants.RESOURCES.RelationalDatabaseAccessRole]: this.makeIAMDataSourceRole(),
            },
        };
        return template;
    }
    addRelationalResolvers(template, resolverFilePath, improvePluralization) {
        let resolverGenerator = new RelationalDBResolverGenerator_1.RelationalDBResolverGenerator(this.context);
        template.Resources = { ...template.Resources, ...resolverGenerator.createRelationalResolvers(resolverFilePath, improvePluralization) };
        return template;
    }
    printCloudformationTemplate(template) {
        return cloudform_1.default(template);
    }
    makeParameters(databaseName) {
        return {
            [ResourceConstants_1.ResourceConstants.PARAMETERS.AppSyncApiName]: new cloudform_types_1.StringParameter({
                Description: `The name of the AppSync API generated from database ${databaseName}`,
                Default: `AppSyncSimpleTransform`,
            }),
            [ResourceConstants_1.ResourceConstants.PARAMETERS.Env]: new cloudform_types_1.StringParameter({
                Description: 'The environment name. e.g. Dev, Test, or Production',
                Default: 'NONE',
            }),
            [ResourceConstants_1.ResourceConstants.PARAMETERS.S3DeploymentBucket]: new cloudform_types_1.StringParameter({
                Description: 'The S3 bucket containing all deployment assets for the project.',
            }),
            [ResourceConstants_1.ResourceConstants.PARAMETERS.S3DeploymentRootKey]: new cloudform_types_1.StringParameter({
                Description: 'An S3 key relative to the S3DeploymentBucket that points to the root of the deployment directory.',
            }),
            [ResourceConstants_1.ResourceConstants.PARAMETERS.AppSyncApiId]: new cloudform_types_1.StringParameter({
                Description: 'The id of the AppSync API associated with this project.',
            }),
            [ResourceConstants_1.ResourceConstants.PARAMETERS.rdsRegion]: new cloudform_types_1.StringParameter({
                Description: 'The region that the RDS Cluster is located in.',
            }),
            [ResourceConstants_1.ResourceConstants.PARAMETERS.rdsClusterIdentifier]: new cloudform_types_1.StringParameter({
                Description: 'The ARN identifier denoting the RDS cluster.',
            }),
            [ResourceConstants_1.ResourceConstants.PARAMETERS.rdsSecretStoreArn]: new cloudform_types_1.StringParameter({
                Description: 'The ARN for the Secret containing the access for the RDS cluster.',
            }),
            [ResourceConstants_1.ResourceConstants.PARAMETERS.rdsDatabaseName]: new cloudform_types_1.StringParameter({
                Description: 'The name of the database within the RDS cluster to use.',
            }),
        };
    }
    makeIAMDataSourceRole() {
        return new iam_1.default.Role({
            RoleName: cloudform_types_1.Fn.Join('-', ['role', cloudform_types_1.Fn.Ref(ResourceConstants_1.ResourceConstants.PARAMETERS.AppSyncApiId), cloudform_types_1.Fn.Ref(ResourceConstants_1.ResourceConstants.PARAMETERS.Env)]),
            AssumeRolePolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Effect: 'Allow',
                        Principal: {
                            Service: 'appsync.amazonaws.com',
                        },
                        Action: 'sts:AssumeRole',
                    },
                ],
            },
            Policies: [
                new iam_1.default.Role.Policy({
                    PolicyName: 'RelationalDatabaseAccessPolicy',
                    PolicyDocument: {
                        Version: '2012-10-17',
                        Statement: [
                            {
                                Effect: 'Allow',
                                Action: [
                                    'rds-data:ExecuteSql',
                                    'rds-data:ExecuteStatement',
                                    'rds-data:DeleteItems',
                                    'rds-data:GetItems',
                                    'rds-data:InsertItems',
                                    'rds-data:UpdateItems',
                                ],
                                Resource: [cloudform_types_1.Fn.Ref(ResourceConstants_1.ResourceConstants.PARAMETERS.rdsClusterIdentifier)],
                            },
                            {
                                Effect: 'Allow',
                                Action: ['secretsmanager:GetSecretValue'],
                                Resource: [cloudform_types_1.Fn.Ref(ResourceConstants_1.ResourceConstants.PARAMETERS.rdsSecretStoreArn)],
                            },
                        ],
                    },
                }),
            ],
        });
    }
    makeRelationalDataSource(cliContext) {
        return new dataSource_1.default({
            Type: 'RELATIONAL_DATABASE',
            Name: `${this.context.databaseName}_rds_DataSource`,
            Description: `RDS Data Source Provisioned for ${this.context.databaseName}`,
            ApiId: cloudform_types_1.Fn.Ref(ResourceConstants_1.ResourceConstants.PARAMETERS.AppSyncApiId),
            ServiceRoleArn: cloudform_types_1.Fn.GetAtt(ResourceConstants_1.ResourceConstants.RESOURCES.RelationalDatabaseAccessRole, 'Arn'),
            RelationalDatabaseConfig: {
                RelationalDatabaseSourceType: 'RDS_HTTP_ENDPOINT',
                RdsHttpEndpointConfig: {
                    AwsRegion: cloudform_types_1.Fn.Ref(ResourceConstants_1.ResourceConstants.PARAMETERS.rdsRegion),
                    DbClusterIdentifier: cloudform_types_1.Fn.Ref(ResourceConstants_1.ResourceConstants.PARAMETERS.rdsClusterIdentifier),
                    DatabaseName: cloudform_types_1.Fn.Ref(ResourceConstants_1.ResourceConstants.PARAMETERS.rdsDatabaseName),
                    Schema: this.context.databaseSchema,
                    AwsSecretStoreArn: cloudform_types_1.Fn.Ref(ResourceConstants_1.ResourceConstants.PARAMETERS.rdsSecretStoreArn),
                },
            },
        }).dependsOn([ResourceConstants_1.ResourceConstants.RESOURCES.RelationalDatabaseAccessRole]);
    }
}
exports.RelationalDBTemplateGenerator = RelationalDBTemplateGenerator;
//# sourceMappingURL=RelationalDBTemplateGenerator.js.map