"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppSyncTransformer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const aws_appsync_alpha_1 = require("@aws-cdk/aws-appsync-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_dynamodb_1 = require("aws-cdk-lib/aws-dynamodb");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const schema_transformer_1 = require("./transformer/schema-transformer");
const defaultAuthorizationConfig = {
    defaultAuthorization: {
        authorizationType: aws_appsync_alpha_1.AuthorizationType.API_KEY,
        apiKeyConfig: {
            description: 'Auto generated API Key from construct',
            name: 'dev',
        },
    },
};
/**
 * (experimental) AppSyncTransformer Construct.
 *
 * @experimental
 */
class AppSyncTransformer extends constructs_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k;
        super(scope, id);
        this.props = props;
        this.tableMap = {};
        this.isSyncEnabled = props.syncEnabled ? props.syncEnabled : false;
        this.pointInTimeRecovery = (_b = props.enableDynamoPointInTimeRecovery) !== null && _b !== void 0 ? _b : false;
        const transformerConfiguration = {
            schemaPath: props.schemaPath,
            outputPath: props.outputPath,
            syncEnabled: (_c = props.syncEnabled) !== null && _c !== void 0 ? _c : false,
            customVtlTransformerRootDirectory: props.customVtlTransformerRootDirectory,
        };
        // Combine the arrays so we only loop once
        // Test each transformer to see if it implements ITransformer
        const allCustomTransformers = [...(_d = props.preCdkTransformers) !== null && _d !== void 0 ? _d : [], ...(_e = props.postCdkTransformers) !== null && _e !== void 0 ? _e : []];
        if (allCustomTransformers && allCustomTransformers.length > 0) {
            allCustomTransformers.forEach(transformer => {
                if (transformer && !this.implementsITransformer(transformer)) {
                    throw new Error(`Transformer does not implement ITransformer from graphql-transformer-core: ${transformer}`);
                }
            });
        }
        const transformer = new schema_transformer_1.SchemaTransformer(transformerConfiguration);
        this.outputs = transformer.transform(props.preCdkTransformers, props.postCdkTransformers);
        const resolvers = transformer.getResolvers();
        this.functionResolvers = (_f = this.outputs.functionResolvers) !== null && _f !== void 0 ? _f : {};
        // Remove any function resolvers from the total list of resolvers
        // Otherwise it will add them twice
        for (const [_, functionResolvers] of Object.entries(this.functionResolvers)) {
            functionResolvers.forEach((resolver) => {
                switch (resolver.typeName) {
                    case 'Query':
                    case 'Mutation':
                    case 'Subscription':
                        delete resolvers[resolver.fieldName];
                        break;
                }
            });
        }
        this.httpResolvers = (_g = this.outputs.httpResolvers) !== null && _g !== void 0 ? _g : {};
        // Remove any http resolvers from the total list of resolvers
        // Otherwise it will add them twice
        for (const [_, httpResolvers] of Object.entries(this.httpResolvers)) {
            httpResolvers.forEach((resolver) => {
                switch (resolver.typeName) {
                    case 'Query':
                    case 'Mutation':
                    case 'Subscription':
                        delete resolvers[resolver.fieldName];
                        break;
                }
            });
        }
        this.resolvers = resolvers;
        this.nestedAppsyncStack = new aws_cdk_lib_1.NestedStack(this, (_h = props.nestedStackName) !== null && _h !== void 0 ? _h : 'appsync-nested-stack');
        // AppSync
        this.appsyncAPI = new aws_appsync_alpha_1.GraphqlApi(this.nestedAppsyncStack, `${id}-api`, {
            name: props.apiName ? props.apiName : `${id}-api`,
            authorizationConfig: props.authorizationConfig
                ? props.authorizationConfig
                : defaultAuthorizationConfig,
            logConfig: {
                fieldLogLevel: props.fieldLogLevel
                    ? props.fieldLogLevel
                    : aws_appsync_alpha_1.FieldLogLevel.NONE,
            },
            schema: aws_appsync_alpha_1.Schema.fromAsset(path.join(transformer.outputPath, 'schema.graphql')),
            xrayEnabled: (_j = props.xrayEnabled) !== null && _j !== void 0 ? _j : false,
        });
        let tableData = (_k = this.outputs.cdkTables) !== null && _k !== void 0 ? _k : {};
        // Check to see if sync is enabled
        if (tableData.DataStore) {
            this.isSyncEnabled = true;
            this.syncTable = this.createSyncTable(tableData.DataStore);
            delete tableData.DataStore; // We don't want to create this again below so remove it from the tableData map
        }
        this.tableNameMap = this.createTablesAndResolvers(tableData, resolvers, props.tableNames);
        if (this.outputs.noneResolvers) {
            this.createNoneDataSourceAndResolvers(this.outputs.noneResolvers, resolvers);
        }
        this.createHttpResolvers();
        this.publicResourceArns = this.getResourcesFromGeneratedRolePolicy(transformer.unauthRolePolicy);
        this.privateResourceArns = this.getResourcesFromGeneratedRolePolicy(transformer.authRolePolicy);
        // Outputs so we can generate exports
        new aws_cdk_lib_1.CfnOutput(scope, 'appsyncGraphQLEndpointOutput', {
            value: this.appsyncAPI.graphqlUrl,
            description: 'Output for aws_appsync_graphqlEndpoint',
        });
    }
    /**
     * graphql-transformer-core needs to be jsii enabled to pull the ITransformer interface correctly.
     * Since it's not in peer dependencies it doesn't show up in the jsii deps list.
     * Since it's not jsii enabled it has to be bundled.
     * The package can't be in BOTH peer and bundled dependencies
     * So we do a fake test to make sure it implements these and hope for the best
     * @param transformer
     */
    implementsITransformer(transformer) {
        return 'name' in transformer
            && 'directive' in transformer
            && 'typeDefinitions' in transformer;
    }
    /**
     * Creates NONE data source and associated resolvers
     * @param noneResolvers The resolvers that belong to the none data source
     * @param resolvers The resolver map minus function resolvers
     */
    createNoneDataSourceAndResolvers(noneResolvers, resolvers) {
        const noneDataSource = this.appsyncAPI.addNoneDataSource('NONE');
        Object.keys(noneResolvers).forEach((resolverKey) => {
            const resolver = resolvers[resolverKey];
            new aws_appsync_alpha_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                api: this.appsyncAPI,
                typeName: resolver.typeName,
                fieldName: resolver.fieldName,
                dataSource: noneDataSource,
                requestMappingTemplate: aws_appsync_alpha_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                responseMappingTemplate: aws_appsync_alpha_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
            });
        });
    }
    /**
     * Creates each dynamodb table, gsis, dynamodb datasource, and associated resolvers
     * If sync is enabled then TTL configuration is added
     * Returns tableName: table map in case it is needed for lambda functions, etc
     * @param tableData The CdkTransformer table information
     * @param resolvers The resolver map minus function resolvers
     */
    createTablesAndResolvers(tableData, resolvers, tableNames = {}) {
        const tableNameMap = {};
        Object.keys(tableData).forEach((tableKey) => {
            var _b;
            const tableName = (_b = tableNames[tableKey]) !== null && _b !== void 0 ? _b : undefined;
            const table = this.createTable(tableData[tableKey], tableName);
            this.tableMap[tableKey] = table;
            const dataSource = this.appsyncAPI.addDynamoDbDataSource(tableKey, table);
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appsync-datasource-deltasyncconfig.html
            if (this.isSyncEnabled && this.syncTable) {
                //@ts-ignore - ds is the base CfnDataSource and the db config needs to be versioned - see CfnDataSource
                dataSource.ds.dynamoDbConfig.versioned = true;
                //@ts-ignore - ds is the base CfnDataSource - see CfnDataSource
                dataSource.ds.dynamoDbConfig.deltaSyncConfig = {
                    baseTableTtl: '43200',
                    deltaSyncTableName: this.syncTable.tableName,
                    deltaSyncTableTtl: '30',
                };
                // Need to add permission for our datasource service role to access the sync table
                dataSource.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: [
                        'dynamodb:*',
                    ],
                    resources: [this.syncTable.tableArn],
                }));
            }
            const dynamoDbConfig = dataSource.ds
                .dynamoDbConfig;
            tableNameMap[tableKey] = dynamoDbConfig.tableName;
            // Loop the basic resolvers
            tableData[tableKey].resolvers.forEach((resolverKey) => {
                const resolver = resolvers[resolverKey];
                new aws_appsync_alpha_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: dataSource,
                    requestMappingTemplate: aws_appsync_alpha_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                    responseMappingTemplate: aws_appsync_alpha_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
                });
            });
            // Loop the gsi resolvers
            tableData[tableKey].gsiResolvers.forEach((resolverKey) => {
                const resolver = resolvers.gsi[resolverKey];
                new aws_appsync_alpha_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: dataSource,
                    requestMappingTemplate: aws_appsync_alpha_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                    responseMappingTemplate: aws_appsync_alpha_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
                });
            });
        });
        return tableNameMap;
    }
    createTable(tableData, tableName) {
        var _b;
        // I do not want to force people to pass `TypeTable` - this way they are only passing the @model Type name
        const modelTypeName = tableData.tableName.replace('Table', '');
        const streamSpecification = this.props.dynamoDbStreamConfig && this.props.dynamoDbStreamConfig[modelTypeName];
        const tableProps = {
            tableName,
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: {
                name: tableData.partitionKey.name,
                type: this.convertAttributeType(tableData.partitionKey.type),
            },
            pointInTimeRecovery: this.pointInTimeRecovery,
            sortKey: tableData.sortKey && tableData.sortKey.name
                ? {
                    name: tableData.sortKey.name,
                    type: this.convertAttributeType(tableData.sortKey.type),
                } : undefined,
            timeToLiveAttribute: ((_b = tableData === null || tableData === void 0 ? void 0 : tableData.ttl) === null || _b === void 0 ? void 0 : _b.enabled) ? tableData.ttl.attributeName : undefined,
            stream: streamSpecification,
        };
        const table = new aws_dynamodb_1.Table(this.nestedAppsyncStack, tableData.tableName, tableProps);
        tableData.localSecondaryIndexes.forEach((lsi) => {
            table.addLocalSecondaryIndex({
                indexName: lsi.indexName,
                sortKey: {
                    name: lsi.sortKey.name,
                    type: this.convertAttributeType(lsi.sortKey.type),
                },
                projectionType: this.convertProjectionType(lsi.projection.ProjectionType),
            });
        });
        tableData.globalSecondaryIndexes.forEach((gsi) => {
            table.addGlobalSecondaryIndex({
                indexName: gsi.indexName,
                partitionKey: {
                    name: gsi.partitionKey.name,
                    type: this.convertAttributeType(gsi.partitionKey.type),
                },
                sortKey: gsi.sortKey && gsi.sortKey.name
                    ? {
                        name: gsi.sortKey.name,
                        type: this.convertAttributeType(gsi.sortKey.type),
                    } : undefined,
                projectionType: this.convertProjectionType(gsi.projection.ProjectionType),
            });
        });
        return table;
    }
    /**
     * Creates the sync table for Amplify DataStore
     * https://docs.aws.amazon.com/appsync/latest/devguide/conflict-detection-and-sync.html
     * @param tableData The CdkTransformer table information
     */
    createSyncTable(tableData) {
        var _b;
        return new aws_dynamodb_1.Table(this, 'appsync-api-sync-table', {
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: {
                name: tableData.partitionKey.name,
                type: this.convertAttributeType(tableData.partitionKey.type),
            },
            sortKey: {
                name: tableData.sortKey.name,
                type: this.convertAttributeType(tableData.sortKey.type),
            },
            timeToLiveAttribute: ((_b = tableData.ttl) === null || _b === void 0 ? void 0 : _b.attributeName) || '_ttl',
        });
    }
    convertAttributeType(type) {
        switch (type) {
            case 'N':
                return aws_dynamodb_1.AttributeType.NUMBER;
            case 'B':
                return aws_dynamodb_1.AttributeType.BINARY;
            case 'S': // Same as default
            default:
                return aws_dynamodb_1.AttributeType.STRING;
        }
    }
    convertProjectionType(type) {
        switch (type) {
            case 'INCLUDE':
                return aws_dynamodb_1.ProjectionType.INCLUDE;
            case 'KEYS_ONLY':
                return aws_dynamodb_1.ProjectionType.KEYS_ONLY;
            case 'ALL': // Same as default
            default:
                return aws_dynamodb_1.ProjectionType.ALL;
        }
    }
    createHttpResolvers() {
        for (const [endpoint, httpResolvers] of Object.entries(this.httpResolvers)) {
            const strippedEndpoint = endpoint.replace(/[^_0-9A-Za-z]/g, '');
            const httpDataSource = this.appsyncAPI.addHttpDataSource(`${strippedEndpoint}`, endpoint);
            httpResolvers.forEach((resolver) => {
                new aws_appsync_alpha_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: httpDataSource,
                    requestMappingTemplate: aws_appsync_alpha_1.MappingTemplate.fromString(resolver.defaultRequestMappingTemplate),
                    responseMappingTemplate: aws_appsync_alpha_1.MappingTemplate.fromString(resolver.defaultResponseMappingTemplate),
                });
            });
        }
    }
    /**
     * This takes one of the autogenerated policies from AWS and builds the list of ARNs for granting GraphQL access later
     * @param policy The auto generated policy from the AppSync Transformers
     * @returns An array of resource arns for use with grants
     */
    getResourcesFromGeneratedRolePolicy(policy) {
        var _b, _c;
        if (!((_c = (_b = policy === null || policy === void 0 ? void 0 : policy.Properties) === null || _b === void 0 ? void 0 : _b.PolicyDocument) === null || _c === void 0 ? void 0 : _c.Statement))
            return [];
        const { region, account } = this.nestedAppsyncStack;
        const resolvedResources = [];
        for (const statement of policy.Properties.PolicyDocument.Statement) {
            const { Resource: resources = [] } = statement !== null && statement !== void 0 ? statement : {};
            for (const resource of resources) {
                const subs = resource['Fn::Sub'][1];
                const { typeName, fieldName } = subs !== null && subs !== void 0 ? subs : {};
                if (fieldName) {
                    resolvedResources.push(`arn:aws:appsync:${region}:${account}:apis/${this.appsyncAPI.apiId}/types/${typeName}/fields/${fieldName}`);
                }
                else {
                    resolvedResources.push(`arn:aws:appsync:${region}:${account}:apis/${this.appsyncAPI.apiId}/types/${typeName}/*`);
                }
            }
        }
        return resolvedResources;
    }
    /**
     * (experimental) Adds the function as a lambdaDataSource to the AppSync api Adds all of the functions resolvers to the AppSync api.
     *
     * @param functionName The function name specified in the.
     * @param id The id to give.
     * @param lambdaFunction The lambda function to attach.
     * @experimental
     * @function directive of the schema
     */
    addLambdaDataSourceAndResolvers(functionName, id, lambdaFunction, options) {
        const functionDataSource = this.appsyncAPI.addLambdaDataSource(id, lambdaFunction, options);
        for (const resolver of this.functionResolvers[functionName]) {
            new aws_appsync_alpha_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                api: this.appsyncAPI,
                typeName: resolver.typeName,
                fieldName: resolver.fieldName,
                dataSource: functionDataSource,
                requestMappingTemplate: aws_appsync_alpha_1.MappingTemplate.fromString(resolver.defaultRequestMappingTemplate),
                responseMappingTemplate: aws_appsync_alpha_1.MappingTemplate.fromString(resolver.defaultResponseMappingTemplate),
            });
        }
        return functionDataSource;
    }
    /**
     * (experimental) Adds a stream to the dynamodb table associated with the type.
     *
     * @returns string - the stream arn token
     * @experimental
     */
    addDynamoDBStream(props) {
        const tableName = `${props.modelTypeName}Table`;
        const table = this.tableMap[tableName];
        if (!table)
            throw new Error(`Table with name '${tableName}' not found.`);
        const cfnTable = table.node.defaultChild;
        cfnTable.streamSpecification = {
            streamViewType: props.streamViewType,
        };
        return cfnTable.attrStreamArn;
    }
    /**
     * (experimental) Allows for overriding the generated request and response mapping templates.
     *
     * @experimental
     */
    overrideResolver(props) {
        const resolver = this.nestedAppsyncStack.node.tryFindChild(`${props.typeName}-${props.fieldName}-resolver`);
        if (!resolver)
            throw new Error(`Resolver with typeName '${props.typeName}' and fieldName '${props.fieldName}' not found`);
        const cfnResolver = resolver.node.defaultChild;
        if (!cfnResolver)
            throw new Error(`Resolver with typeName '${props.typeName}' and fieldName '${props.fieldName}' not found`);
        if (props.requestMappingTemplateFile) {
            cfnResolver.requestMappingTemplate = fs.readFileSync(props.requestMappingTemplateFile).toString('utf-8');
        }
        if (props.responseMappingTemplateFile) {
            cfnResolver.responseMappingTemplate = fs.readFileSync(props.responseMappingTemplateFile).toString('utf-8');
        }
    }
    /**
     * (experimental) Adds an IAM policy statement granting access to the public fields of the AppSync API.
     *
     * Policy is based off of the @auth transformer
     * https://docs.amplify.aws/cli/graphql-transformer/auth
     *
     * @param grantee The principal to grant access to.
     * @experimental
     */
    grantPublic(grantee) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['appsync:GraphQL'],
            resourceArns: this.publicResourceArns,
            scope: this,
        });
    }
    /**
     * (experimental) Adds an IAM policy statement granting access to the private fields of the AppSync API.
     *
     * Policy is based off of the @auth transformer
     * https://docs.amplify.aws/cli/graphql-transformer/auth
     *
     * @experimental
     */
    grantPrivate(grantee) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['appsync:GraphQL'],
            resourceArns: this.privateResourceArns,
        });
    }
}
exports.AppSyncTransformer = AppSyncTransformer;
_a = JSII_RTTI_SYMBOL_1;
AppSyncTransformer[_a] = { fqn: "cdk-appsync-transformer.AppSyncTransformer", version: "2.0.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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