"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const acm = require("@aws-cdk/aws-certificatemanager");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const cdk = require("@aws-cdk/core");
const index_1 = require("./index");
const app = new cdk.App();
const env = {
    region: process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_DEFAULT_ACCOUNT,
};
const stack = new cdk.Stack(app, 'demo-stack2', { env });
const zoneName = 'svc.local';
const internalAlbRecordName = 'internal';
const externalAlbRecordName = 'external';
const internalALBEndpoint = `http://${internalAlbRecordName}.${zoneName}`;
// order service
const orderTask = new ecs.FargateTaskDefinition(stack, 'orderTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
orderTask.addContainer('order', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/OrderService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        WAREHOUSE_SVC_URL: `${internalALBEndpoint}:9092/api.php`,
        serviceName: 'order',
        versionNum: '1.0',
    },
});
// customer service
const customerTask = new ecs.FargateTaskDefinition(stack, 'customerTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
customerTask.addContainer('customer', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'customer',
        versionNum: '1.0',
    },
});
// product service
const productTask = new ecs.FargateTaskDefinition(stack, 'productTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
productTask.addContainer('product', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'product',
        versionNum: '1.0',
    },
});
// nginx service
const nginxTask = new ecs.FargateTaskDefinition(stack, 'nginxTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nginxTask.addContainer('nginx', {
    image: ecs.ContainerImage.fromRegistry('nginx:latest'),
    portMappings: [
        { containerPort: 80 },
    ],
});
// php service
const phpTask = new ecs.FargateTaskDefinition(stack, 'phpTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
phpTask.addContainer('php', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/nginx-php')),
    portMappings: [
        { containerPort: 80 },
    ],
    environment: {
        serviceName: 'warehouse',
        versionNum: '0.1',
    },
});
// laravel-nginx-php-fpm service
const laravelNginxPhpFpmTask = new ecs.FargateTaskDefinition(stack, 'laravelNginxPhpFpmTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
laravelNginxPhpFpmTask.addContainer('laravel-nginx-php-fpm', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/laravel-nginx-php-fpm')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// laravel-bitnami service
const laravelBitnamiTask = new ecs.FargateTaskDefinition(stack, 'laravelBitnamiTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
laravelBitnamiTask.addContainer('laravel-bitnami', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/laravel-bitnami')),
    portMappings: [
        { containerPort: 3000 },
    ],
});
// NuxtJS service
const nuxtTask = new ecs.FargateTaskDefinition(stack, 'nuxtTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nuxtTask.addContainer('nuxt', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/nuxt')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// Node.js service
const nodeTask = new ecs.FargateTaskDefinition(stack, 'nodeTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nodeTask.addContainer('node', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/node')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// java spring boot service
const javaTask = new ecs.FargateTaskDefinition(stack, 'javaTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
javaTask.addContainer('java', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/java-spring-boot')),
    portMappings: [
        { containerPort: 8080 },
    ],
});
// apache-php service
const apachePHP = new ecs.FargateTaskDefinition(stack, 'apachephpTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
apachePHP.addContainer('apachephp', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/apache-php')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// python-flask service
const flaskTask = new ecs.FargateTaskDefinition(stack, 'flaskTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
flaskTask.addContainer('flask', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/python-flask')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// gorilla-mux
const muxTask = new ecs.FargateTaskDefinition(stack, 'muxTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
muxTask.addContainer('mux', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/gorilla-mux')),
    portMappings: [
        { containerPort: 8080 },
    ],
});
const certArn = stack.node.tryGetContext('ACM_CERT_ARN');
const cert = certArn ? acm.Certificate.fromCertificateArn(stack, 'Cert', certArn) : undefined;
const svc = new index_1.DualAlbFargateService(stack, 'Service', {
    spot: true,
    enableExecuteCommand: true,
    tasks: [
        // The order service with both external/internal access
        {
            task: orderTask,
            desiredCount: 2,
            internal: { port: 80 },
            external: cert ? { port: 443, certificate: [cert] } : { port: 80 },
            // customize the service autoscaling policy
            scalingPolicy: {
                maxCapacity: 20,
                requestPerTarget: 1000,
                targetCpuUtilization: 50,
            },
        },
        {
            // The customer service(internal only)
            task: customerTask,
            desiredCount: 1,
            internal: { port: 8080 },
            capacityProviderStrategy: [
                {
                    capacityProvider: 'FARGATE',
                    base: 1,
                    weight: 1,
                },
                {
                    capacityProvider: 'FARGATE_SPOT',
                    base: 0,
                    weight: 3,
                },
            ],
        },
        // The produce service(internal only)
        {
            task: productTask,
            desiredCount: 1,
            internal: { port: 9090 },
        },
        // The nginx service(external only)
        {
            task: nginxTask,
            desiredCount: 1,
            external: { port: 9091 },
        },
        // The nginx-php-fpm service(external/internal)
        {
            task: phpTask,
            desiredCount: 1,
            internal: { port: 9092 },
            external: { port: 9092 },
        },
        // The NuxtJS service(external/internal)
        {
            task: nuxtTask,
            desiredCount: 1,
            internal: { port: 9093 },
            external: { port: 9093 },
        },
        // The node service(external/internal)
        {
            task: nodeTask,
            desiredCount: 1,
            internal: { port: 9094 },
            external: { port: 9094 },
        },
        // The laravel-nginx-php-fpm service(external/internal)
        {
            task: laravelNginxPhpFpmTask,
            desiredCount: 1,
            internal: { port: 9095 },
            external: { port: 9095 },
        },
        // The laravel-bitnami service(external/internal)
        {
            task: laravelBitnamiTask,
            desiredCount: 1,
            internal: { port: 9096 },
            external: { port: 9096 },
        },
        // java spring boot service(external/internal)
        {
            task: javaTask,
            desiredCount: 1,
            internal: { port: 9097 },
            external: { port: 9097 },
            healthCheck: { path: '/hello-world' },
        },
        // apache-php(external/internal)
        {
            task: apachePHP,
            desiredCount: 1,
            internal: { port: 9098 },
            external: { port: 9098 },
        },
        // python-flask(external/internal)
        {
            task: flaskTask,
            desiredCount: 1,
            internal: { port: 9099 },
            external: { port: 9099 },
        },
        // gorilla-mux(external/internal)
        {
            task: muxTask,
            desiredCount: 1,
            internal: { port: 9100 },
            external: { port: 9100 },
        },
    ],
    route53Ops: {
        zoneName,
        externalElbRecordName: externalAlbRecordName,
        internalElbRecordName: internalAlbRecordName,
    },
});
// create a dummy sg
const dummySg = new ec2.SecurityGroup(stack, 'DummySG', {
    vpc: svc.vpc,
});
// allow all traffic from dummy sg to all the services
for (let i = 0; i < svc.service.length; i++) {
    svc.service[i].connections.allowFrom(dummySg, ec2.Port.allTraffic());
}
//# sourceMappingURL=data:application/json;base64,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