import json
from datetime import datetime, timedelta
from typing import Any, Dict, Optional, Union

from pyspark.sql import DataFrame
from pyspark.sql.functions import lit
from pyspark.sql.types import StructType

from ng_data_pipelines_sdk.aws_interface import AWSInterface
from ng_data_pipelines_sdk.custom_logger import logger
from ng_data_pipelines_sdk.interfaces import (
    DATE_FORMATTER,
    DataFrameDict,
    DatePartition,
    EmptyDataFrameException,
    FileType,
    FnKind,
    InputDataFrameParams,
    InputDataFrameParamsDict,
    OutputDataFrameParams,
    OutputDataFrameParamsDict,
    S3BucketParams,
    S3ReadSchemaParams,
    StepParams,
    StepParamsDict,
    TransformParams,
    TransformParamsDict,
)
from ng_data_pipelines_sdk.spark_manager import SparkManager
from ng_data_pipelines_sdk.utils import (
    generate_date_paths,
    handle_pyspark_timestamp_in_schema,
)

logger.setLevel("INFO")


class DataFrameManager:
    def __init__(
        self,
        file_system: str,
        aws_interface: AWSInterface,
        spark_manager: SparkManager,
    ):
        """
        Initializes a new instance of the DataFrameManager class.

        Args:
            file_system (str): The file system to be used by the DataFrameManager.
            aws_interface (AWSInterface): An instance of the AWSInterface class.
            spark_manager (SparkManager): An instance of the SparkManager class.

        Returns:
            None
        """
        self.file_system = file_system
        self.aws_interface = aws_interface
        self.spark_manager = spark_manager
        self.completed_steps = []
        logger.info("DataFrameManager initialized.")

    def _get_bucket_url(self, bucket_params: S3BucketParams) -> str:
        return f"{self.file_system}://{bucket_params.bucket_name}"

    def convert_schema_object_to_pyspark_schema(
        self, schema_object: Dict[str, Any]
    ) -> StructType:
        logger.debug("Converting schema object to PySpark StructType...")
        pyspark_schema = StructType.fromJson(schema_object)
        pyspark_schema = handle_pyspark_timestamp_in_schema(pyspark_schema)
        logger.debug(f"Schema converted sucessfully: {pyspark_schema}")

        return pyspark_schema

    def retrieve_schema_from_s3(
        self, s3_read_schema_params: S3ReadSchemaParams
    ) -> StructType:
        """
        Retrieve the schema for the data from an AWS S3 bucket.

        Returns:
            pyspark.sql.types.StructType: The retrieved schema.

        Raises:
            ValueError: If there is an issue retrieving or handling the schema.
        """

        if s3_read_schema_params is None:
            raise ValueError("s3_schema_path_params must be provided")

        if s3_read_schema_params.bucket_params.bucket_name is None:
            raise ValueError("bucket_name must be provided")

        bucket_name = s3_read_schema_params.bucket_params.bucket_name
        path_to_file = s3_read_schema_params.path
        full_s3_file_path = f"{self.file_system}://{bucket_name}/{path_to_file}"

        logger.debug(f"Retrieving schema from path '{full_s3_file_path}'...")
        try:
            # Retrieve the schema object from AWS S3
            schema_object = self.aws_interface.get_object_aws(
                env=s3_read_schema_params.bucket_params.env,
                bucket_name=bucket_name,
                object_name=path_to_file,
            )

            # Decode the object to a string
            schema_str = schema_object.decode("utf-8").strip()

            # Convert the string to JSON
            schema_json = json.loads(schema_str)

            # Convert the JSON schema to a PySpark StructType
            pyspark_schema = StructType.fromJson(schema_json)

            logger.debug(f"Schema retrieved sucessfully: {pyspark_schema}")

            # Handle any timestamp issues in PySpark schema, if necessary
            return handle_pyspark_timestamp_in_schema(pyspark_schema)
        except Exception as e:
            error_message = (
                f"Error retrieving or handling schema from {full_s3_file_path}: {e}"
            )
            logger.error(error_message)
            raise ValueError(error_message)

    def read_dataframe(
        self,
        input_df_params: InputDataFrameParams,
        cache: bool = False,
        verbose: bool = True,
    ) -> DataFrame:
        """
        Reads a DataFrame from a specified location.

        Args:
            df_params (InputDataFrameParams): The parameters for reading the DataFrame.

        Returns:
            DataFrame: The read DataFrame.

        """
        bucket_url = self._get_bucket_url(input_df_params.dataframe_bucket_params)
        full_base_path = f"{bucket_url}/{input_df_params.dataframe_base_path}"

        if input_df_params.dataframe_specific_paths:
            if isinstance(input_df_params.dataframe_specific_paths, list):
                paths = [
                    f"{bucket_url}/{path}"
                    for path in input_df_params.dataframe_specific_paths
                ]
            else:
                paths = f"{bucket_url}/{input_df_params.dataframe_specific_paths}"
        elif input_df_params.read_date_params:
            date_partitions = input_df_params.read_date_params.date_partitions
            paths = generate_date_paths(
                base_path=f"{bucket_url}/{input_df_params.dataframe_base_path}",
                dates=input_df_params.read_date_params.read_dates,
                year_partition=date_partitions.get(DatePartition.YEAR, "year"),
                month_partition=date_partitions.get(DatePartition.MONTH, "month"),
                day_partition=date_partitions.get(DatePartition.DAY, "day"),
            )

            # Prevent pyspark from adding date columns to the DataFrame
            if not input_df_params.read_date_params.add_all_date_columns:
                full_base_path = None

        else:
            paths = f"{bucket_url}/{input_df_params.dataframe_base_path}"

        if input_df_params.pyspark_schema_struct:
            schema = self.convert_schema_object_to_pyspark_schema(
                schema_object=input_df_params.pyspark_schema_struct
            )
        elif input_df_params.s3_schema_path_params:
            schema = self.retrieve_schema_from_s3(input_df_params.s3_schema_path_params)
        else:
            schema = None

        if verbose:
            logger.info("Starting read operation.")

        df = self.spark_manager.read(
            env=input_df_params.dataframe_bucket_params.env,
            paths=paths,
            file_type=input_df_params.dataframe_file_type,
            base_path=full_base_path,
            schema=schema,
            cache=cache,
        )

        if verbose:
            logger.info("Read operation completed successfully.\n")

        if df is None:
            raise EmptyDataFrameException

        return df

    def write_schema(
        self, df: DataFrame, output_df_params: OutputDataFrameParams
    ) -> None:
        """
        Write the schema of the DataFrame to the specified location.

        Parameters:
        - df (DataFrame): The DataFrame whose schema is to be written.

        Raises:
        - ValueError: If there is an issue with writing the schema object to AWS S3.
        """
        pyspark_schema_json = df.schema.jsonValue()

        if output_df_params.dataframe_bucket_params.bucket_name is None:
            raise ValueError("bucket_name must be provided to write schema to S3")

        if output_df_params.dataframe_specific_path:
            if isinstance(output_df_params.dataframe_specific_path, list):
                raise ValueError(
                    "Specific path is set as a list. Only a single path is allowed for writing the schema."
                )
            logger.warning(
                "Specific path is set for writing the schema. Schema will be written to the same path."
            )
            path_to_schema = output_df_params.dataframe_specific_path
        else:
            if output_df_params.dataframe_base_path is None:
                raise ValueError(
                    "Neither 'specific_path' nor 'dataframe_base_path' is set for writing the schema."
                )

            parent_folder_of_dataframe_base_path = "/".join(
                output_df_params.dataframe_base_path.split("/")[:-1]
            )
            path_to_schema = f"{parent_folder_of_dataframe_base_path}/schema.json"

        try:
            self.aws_interface.put_object_aws(
                env=output_df_params.dataframe_bucket_params.env,
                bucket_name=output_df_params.dataframe_bucket_params.bucket_name,
                object_name=path_to_schema,
                object_data=pyspark_schema_json,
            )
        except Exception as e:
            raise ValueError(
                f"Error writing schema to {output_df_params.dataframe_bucket_params.bucket_name}/{path_to_schema}: {e}"
            )

    def _write_dataframe_specific_path(
        self, df: DataFrame, output_df_params: OutputDataFrameParams
    ) -> None:
        bucket_url = self._get_bucket_url(output_df_params.dataframe_bucket_params)
        write_path_url = f"{bucket_url}/{output_df_params.dataframe_specific_path}"

        if output_df_params.overwrite:
            logger.info(
                f"Overwrite is set to True. Deleting existing objects under path {write_path_url}"
            )
            try:
                self.aws_interface.delete_objects_aws(
                    env=output_df_params.dataframe_bucket_params.env,
                    bucket_name=output_df_params.dataframe_bucket_params.bucket_name,
                    path=output_df_params.dataframe_specific_path,
                )
            except Exception as e:
                raise ValueError(f"Error deleting dataframe objects: {e}")

            logger.info("Objects deleted successfully.")

        logger.info(f"Writing DataFrame to path: {write_path_url}...")
        self.spark_manager.write(
            env=output_df_params.dataframe_bucket_params.env,
            df=df,
            path=write_path_url,
            file_type=output_df_params.dataframe_file_type,
            partitions=None,
        )

    def _write_dataframe_base_path(
        self,
        df: DataFrame,
        output_df_params: OutputDataFrameParams,
    ) -> None:
        """
        Writes a DataFrame to a specified location with optional date partitions.

        Args:
            df (DataFrame): The DataFrame to be written.
            df_params (OutputDataFrameParams): The parameters for writing the DataFrame.

        Returns:
            None
        """
        bucket_url = self._get_bucket_url(output_df_params.dataframe_bucket_params)
        write_base_path_url = f"{bucket_url}/{output_df_params.dataframe_base_path}"
        logger.debug(f"Write base path: {write_base_path_url}")

        df_to_write = df
        if output_df_params.single_write_date:
            # Add the date partitions columns to the DataFrame
            date_partition_path = ""
            for (
                date_partition,
                date_partition_name,
            ) in (
                output_df_params.single_write_date.single_write_date_partitions.items()
            ):
                date_partition_path += f"{date_partition_name}={date_partition.value}/"

                formatted_date_part_value = DATE_FORMATTER[date_partition.value](
                    output_df_params.single_write_date.single_write_date
                )

                df_to_write = df_to_write.withColumn(
                    date_partition_name, lit(formatted_date_part_value)
                )

            if output_df_params.overwrite:
                full_path = f"{write_base_path_url}/{date_partition_path}"
                logger.info(
                    f"Overwrite is set to True. Deleting existing objects under path {full_path}"
                )
                try:
                    self.aws_interface.delete_objects_aws(
                        env=output_df_params.dataframe_bucket_params.env,
                        bucket_name=output_df_params.dataframe_bucket_params.bucket_name,
                        path=f"{output_df_params.dataframe_base_path}/{date_partition_path}",
                    )
                except Exception as e:
                    raise ValueError(f"Error deleting dataframe objects: {e}")

                logger.info("Objects deleted successfully.")

        logger.info(
            f"Writing DataFrame to base path {write_base_path_url}{'' if not output_df_params.partition_by else f' with partitions {output_df_params.partition_by}'}..."
        )
        self.spark_manager.write(
            env=output_df_params.dataframe_bucket_params.env,
            df=df_to_write,
            path=write_base_path_url,
            file_type=output_df_params.dataframe_file_type,
            partitions=output_df_params.partition_by,
        )

        if (
            output_df_params.write_schema_on_s3
            or output_df_params.dataframe_file_type != FileType.PARQUET
        ):
            self.write_schema(df, output_df_params)

        logger.info("Dataframe written successfully.\n")

    def write_dataframe(
        self,
        df: DataFrame,
        output_df_params: OutputDataFrameParams,
    ) -> None:
        """
        Writes a DataFrame to a specified location with optional date partitions.

        Args:
            df (DataFrame): The DataFrame to be written.
            df_params (OutputDataFrameParams): The parameters for writing the DataFrame.

        Returns:
            None
        """
        if output_df_params.dataframe_base_path:
            self._write_dataframe_base_path(df, output_df_params)
        else:
            self._write_dataframe_specific_path(df, output_df_params)

        if (
            output_df_params.write_schema_on_s3
            or output_df_params.dataframe_file_type != FileType.PARQUET
        ):
            self.write_schema(df, output_df_params)

        logger.info("Dataframe written successfully.\n")

    @staticmethod
    def read_and_validate_step_json(step_params_json_file_path):
        """
        Reads and validates a JSON file containing step parameters.

        Args:
            step_params_json_file_path (str): The file path to the JSON file.

        Returns:
            StepParams: An instance of the StepParams class representing the step configuration.

        Raises:
            ValueError: If there is an error loading or decoding the JSON file.
        """
        try:
            with open(step_params_json_file_path, "r") as f:
                step_config = json.load(f)
        except (IOError, json.JSONDecodeError) as e:
            raise ValueError(f"Error loading config file: {e}") from e

        return StepParams(**step_config)

    def read_input_dataframes_params(
        self, input_df_params_dict: InputDataFrameParamsDict
    ) -> DataFrameDict:
        """Reads multiple input dataframes based on the provided configuration."""
        dfs: DataFrameDict = {}

        for df_name, input_df_params in input_df_params_dict.items():
            logger.info(f"Starting read operation for DataFrame '{df_name}'...")

            try:
                dfs[df_name] = self.read_dataframe(
                    input_df_params=input_df_params, verbose=False
                )
                logger.info(f"Dataframe '{df_name}' read successfully.\n")
            except EmptyDataFrameException:
                logger.warning(
                    f"DataFrame '{df_name}' is empty. No data could be found in the specified path(s). Skipping."
                )

        return dfs

    def _select_transformation_function(self, transform_params: TransformParams):
        """Select the transformation function based on direct or indirect reference."""
        if transform_params.transform_function:
            return transform_params.transform_function
        elif transform_params.fn_indirect:
            raise NotImplementedError("Indirect function references are not supported.")
        else:
            raise ValueError(
                "TransformParams must have either 'transform_function' or 'fn_indirect' set."
            )

    def apply_transform_to_df(
        self, df: DataFrame, transform_params: TransformParams
    ) -> DataFrame:
        """Apply transformation to a single DataFrame."""
        transform_fn = self._select_transformation_function(transform_params)
        fn_kwargs = transform_params.fn_kwargs or {}

        return transform_fn(df, **fn_kwargs)

    def apply_transform(
        self, input_dfs_dict: DataFrameDict, transform_params: TransformParams
    ) -> DataFrameDict:
        """Apply transformations to single or multiple DataFrames."""
        transform_fn = self._select_transformation_function(transform_params)

        fn_kwargs = transform_params.fn_kwargs or {}

        dfs_to_transform = transform_params.apply_only_on or list(input_dfs_dict.keys())

        if transform_params.fn_kind == FnKind.SINGLE:
            for df_name in dfs_to_transform:
                if df_name in input_dfs_dict:
                    input_dfs_dict[df_name] = transform_fn(
                        input_dfs_dict[df_name], **fn_kwargs
                    )
                else:
                    logger.warning(f"DataFrame '{df_name}' not found.")
        elif transform_params.fn_kind == FnKind.BATCH:
            input_dfs_dict = transform_fn(input_dfs_dict, **fn_kwargs)

        return input_dfs_dict

    def apply_all_transforms(
        self,
        input_dfs_dict: DataFrameDict,
        transform_params_dict: TransformParamsDict,
    ) -> DataFrameDict:
        """Apply a list of transformations to the given DataFrames."""
        for transform_id, transform_params in transform_params_dict.items():
            logger.info(
                f"Applying transformation '{transform_id}': {transform_params.transform_label}..."
            )
            input_dfs_dict = self.apply_transform(input_dfs_dict, transform_params)
        return input_dfs_dict

    def write_output_dataframes(
        self,
        dfs_to_output: DataFrameDict,
        output_df_params_dict: OutputDataFrameParamsDict,
    ) -> None:
        for df_name, df in dfs_to_output.items():
            logger.info(f"Starting write operation for DataFrame '{df_name}'...")
            output_df_params = output_df_params_dict.get(df_name)

            if output_df_params is None:
                raise ValueError(
                    f"No OutputDataFrameParams found for dataframe '{df_name}'"
                )

            self.write_dataframe(df, output_df_params)

    def inject_processing_date_into_step_params(
        self, step_params: StepParams, processing_date: datetime
    ) -> None:
        for _, df_params in step_params.input_dataframes_params.items():
            if df_params.read_date_params:
                df_params.read_date_params.was_offset_applied = False

                if df_params.read_date_params.read_dates == "{{processing_date}}":
                    df_params.read_date_params.read_dates = processing_date

                    offset = df_params.read_date_params.processing_date_offset_days
                    if offset and offset != 0:
                        df_params.read_date_params.read_dates
                        +timedelta(days=offset)
                        df_params.read_date_params.was_offset_applied = True

        if step_params.output_dataframes_params:
            for _, df_params in step_params.output_dataframes_params.items():
                if (
                    df_params.single_write_date
                    and df_params.single_write_date.single_write_date
                    == "{{processing_date}}"
                ):
                    df_params.single_write_date.single_write_date = processing_date

    def process_step(
        self,
        step_params_json_file_path: Optional[str] = None,
        step_params: Optional[StepParams] = None,
        processing_date: Optional[datetime] = None,
        write_output_dfs: bool = True,
        return_output_dfs: bool = False,
    ) -> Optional[DataFrameDict]:
        """
        Process a step in the data pipeline.

        Args:
            step_params_json_file_path (str, optional): The file path to the JSON file containing the step parameters. Either this or `step_params` must be provided. Defaults to None.
            step_params (StepParams, optional): The step parameters object. Either this or `step_params_json_file_path` must be provided. Defaults to None.
            processing_date (datetime, optional): The processing date to be injected into the step parameters. Defaults to None.
            write_output_dfs (bool, optional): Flag indicating whether to write the output DataFrames. Defaults to True.
            return_output_dfs (bool, optional): Flag indicating whether to return the output DataFrames. This is ignored if `write_output_dfs` is False, in which case it is set to True. Defaults to False.

        Returns:
            Optional[DataFrameDict]: A dictionary containing the output DataFrames, if `return_output_dfs` is True.

        Raises:
            ValueError: If neither `step_params_json_file_path` nor `step_params` is provided, or if both are provided.
            Exception: If an error occurs during the processing step.

        """
        if step_params_json_file_path is None and step_params is None:
            raise ValueError(
                "Either 'step_params_json_file_path' or 'step_params' must be provided"
            )

        if step_params_json_file_path is not None and step_params is not None:
            raise ValueError(
                "Only one of 'step_params_json_file_path' or 'step_params' should be provided"
            )

        if step_params_json_file_path:
            logger.info("Reading and validating step configuration...")
            step_params = self.read_and_validate_step_json(step_params_json_file_path)
            logger.info("Step configuration read and validated.")

        assert step_params is not None, "'step_params' was unexpectedly None"

        if processing_date is not None:
            logger.debug(
                f"Injecting processing date '{processing_date}' into step parameters..."
            )
            self.inject_processing_date_into_step_params(
                step_params=step_params,
                processing_date=processing_date,
            )
            logger.debug("Processing date injected successfully.")

        try:
            print("\nPhase 1: Read Input DataFrames\n" + "-" * 80)
            input_dfs_dict = self.read_input_dataframes_params(
                step_params.input_dataframes_params
            )
            logger.info("Read all Input DataFrames successfully!")

            print("\nPhase 2: Apply Transformations\n" + "-" * 80)
            if step_params.transform_params:
                dfs_to_output_dict = self.apply_all_transforms(
                    input_dfs_dict, step_params.transform_params
                )
                print("")
                logger.info("Applied all Transformations successfully!")
            else:
                print("No transformations to apply. Moving to Phase 3.")
                dfs_to_output_dict = input_dfs_dict

            if step_params.output_dataframes_params and write_output_dfs:
                print("\nPhase 3: Write Output DataFrames\n" + "-" * 80)
                self.write_output_dataframes(
                    dfs_to_output_dict, step_params.output_dataframes_params
                )
                logger.info("Wrote all Output DataFrames successfully!")

                if return_output_dfs:
                    return dfs_to_output_dict
            else:
                return dfs_to_output_dict
        except Exception as e:
            logger.exception(f"Error during processing step: {e}")
            raise e from None  # Re-raise to propagate the exception

    def process_step_dict(
        self,
        step_params_dict: Union[StepParamsDict, Dict[str, str]],
        processing_date: Optional[datetime] = None,
    ) -> Optional[DataFrameDict]:
        """
        Process a dictionary of step parameters.

        Args:
            step_params_dict (Union[StepParamsDict, Dict[str, str]]): A dictionary containing the step IDs as keys and the corresponding step parameters as values.
            processing_date (Optional[datetime], optional): The processing date to be used. Defaults to None.

        Returns:
            Optional[DataFrameDict]: A dictionary containing the processed data frames, if any.

        Raises:
            Exception: If an error occurs during the processing of any step.

        """
        for step_num, (step_id, step_params) in enumerate(
            step_params_dict.items(), start=1
        ):
            if step_id in self.completed_steps:
                logger.info(f"Step '{step_id}' has already been processed. Skipping...")
                continue
            try:
                if isinstance(step_params, str):
                    logger.debug(
                        f"Step params for '{step_id}' is a file path. Reading and validating..."
                    )
                    step_params_json_file_path = step_params
                    step_params = self.read_and_validate_step_json(
                        step_params_json_file_path
                    )
                    logger.debug(
                        f"Step params for '{step_id}' read and validated successfully."
                    )

                print("=" * 80)
                print(
                    f"Processing step {step_num}/{len(step_params_dict)}: '{step_id}'"
                )
                print("=" * 80)
                self.process_step(
                    step_params=step_params.model_copy(deep=True),
                    processing_date=processing_date,
                )

                print("\n" + "-" * 80)
                print(f"Step '{step_id}' processed successfully!\n")
                print("\n" + "-" * 80)

                self.completed_steps.append(step_id)
            except Exception as e:
                logger.exception(f"Error during processing step '{step_id}': {e}")
                raise e from None

        self.completed_steps = []
        print("=" * 80)
        print("All steps processed successfully!")
        print("=" * 80)
