# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['perceiver',
 'perceiver.data',
 'perceiver.data.image',
 'perceiver.data.text',
 'perceiver.model',
 'perceiver.model.core',
 'perceiver.model.image',
 'perceiver.model.text',
 'perceiver.scripts',
 'perceiver.scripts.image',
 'perceiver.scripts.text',
 'perceiver.scripts.utils']

package_data = \
{'': ['*']}

install_requires = \
['einops>=0.4.0,<0.5.0',
 'fairscale>=0.4.0,<0.5.0',
 'jsonargparse[signatures]>=4.7.0,<4.8.0',
 'pytorch-lightning>=1.6.0,<1.7.0',
 'torch-optimizer>=0.3.0,<0.4.0',
 'torch==1.11.0',
 'torchmetrics>=0.8.0,<0.9.0']

extras_require = \
{'image': ['datasets>=2.3.0,<2.4.0', 'torchvision==0.12.0'],
 'text': ['datasets>=2.3.0,<2.4.0',
          'tokenizers>=0.12.0,<0.13.0',
          'transformers>=4.19.0,<4.20.0']}

setup_kwargs = {
    'name': 'perceiver-io',
    'version': '0.4.0',
    'description': 'Perceiver IO',
    'long_description': '# Perceiver IO\n\nThis repository is a PyTorch and [PyTorch Lightning](https://www.pytorchlightning.ai/) implementation of\n\n- [Perceiver IO: A General Architecture for Structured Inputs & Outputs](https://arxiv.org/abs/2107.14795) and\n- [Perceiver: General Perception with Iterative Attention](https://arxiv.org/abs/2103.03206)\n\n\nThe codebase is designed for easy extension to new tasks and datasets. If you are a researcher or practitioner working\non new Perceiver IO models and use cases, you might find this repository useful. The integration with PyTorch Lightning\nsupports model training at any scale. On the other hand, if you are mainly interested in using or fine-tuning models\nfrom the Perceiver IO paper you may want to take a look at the 🤗 [Perceiver IO](https://huggingface.co/docs/transformers/model_doc/perceiver)\nimplementation.\n\n## Overview\n\nThe following figure maps Perceiver IO and Perceiver concepts to the [core modules](perceiver/model/core/modules.py) of\nthe implementation (see [Architecture](docs/architecture.md) for details).\n\n![architecture](docs/images/architecture.png)\n\nInterfaces are defined on three levels:\n\n- *PyTorch model API*: defines generic `PerceiverEncoder` and `PerceiverDecoder` classes and task-specific `InputAdapter`\n  and `OutputAdapter` subclasses from which PyTorch models can be constructed.\n- *PyTorch Lightning model API*: defines wrappers for PyTorch models to support training with the\n  [PyTorch Lightning Trainer](https://pytorch-lightning.readthedocs.io/en/stable/common/trainer.html).\n- *PyTorch Lightning model CLI*: binds the PyTorch Lightning model API to the command line via the\n  [Lightning CLI](https://pytorch-lightning.readthedocs.io/en/stable/common/lightning_cli.html).\n\nInterface usage examples are available for two models from the [Perceiver IO paper](https://arxiv.org/abs/2107.14795):\n\n| Model                                                           | Parameters |                                                              |                                                      |\n|-----------------------------------------------------------------|-----------:|--------------------------------------------------------------|------------------------------------------------------|\n| Language model (Perceiver IO Base, SentencePiece tokenization)  |       223M | [construction](docs/models/language-model/construction.md)   | [training](docs/models/language-model/training.md)   |\n| Image classifier (Perceiver IO config A, 2D Fourier Features)   |      48.4M | [construction](docs/models/image-classifier/construction.md) | [training](docs/models/image-classifier/training.md) |\n\nTraining of smaller models is shown in section [Training examples](#training-examples), their usage in section\n[Inference examples](#training-examples).\n\n## Installation\n\n### Via pip\n\n```shell\npip install perceiver-io[image,text]\n```\n\n### From sources\n\n#### Conda + Poetry\n\n```shell\nconda env create -f environment.yml\nconda activate perceiver-io\npoetry install --all-extras\n```\n\nThis requires a [Poetry installation](https://python-poetry.org/docs/master/#installation), version 1.2.0b2 or higher.\nIf running `poetry` fails with a `KeyringError`, refer to the [keyring](https://keyring.readthedocs.io/) documentation\nhow to [disable keyring](https://keyring.readthedocs.io/en/latest/#disabling-keyring) usage.\n\n#### Docker image\n\nA `perceiver-io` Docker image can be built with:\n\n```shell\ndocker build -t perceiver-io .\n```\n\nTraining of Perceiver IO models with this image is described [here](docs/docker.md).\n\n## Training examples\n\nThis section uses rather small Perceiver IO models so that the following training examples can be run on limited hardware\nresources. Training automatically scales to more than one GPU and was tested on 4 RTX 3080 GPUs. For GPUs with less memory\nyou may need to reduce the `--data.batch_size` or turn on [activation checkpointing](#activation-checkpointing) for some\nof the examples.\n\nDatasets used for model training are 🤗 [Datasets](https://huggingface.co/docs/datasets) wrapped into PyTorch Lightning\ndata modules (see [data](perceiver/data) package). Datasets are automatically downloaded, preprocessed and cached when\ntheir corresponding Lightning data module is loaded during training. Manual dataset preprocessing is described [here](docs/datasets.md).\n\nAn archive with training checkpoints can be downloaded [here](https://martin-krasser.com/perceiver/logs-update-5.zip)\nand should be extracted in project\'s root directory to be compatible with the example command lines below. It also\ncontains Tensorboard logs and config files.\n\nI didn\'t really tune hyperparameters, so you\'ll likely get better results with a bit of experimentation\n(see also [training tips](docs/training-tips.md)).\n\n### Masked language modeling\n\nThis section trains a very small language model (2.9M parameters) on masked language modeling with whole word masking.\nIt is first pretrained on [WikiText-103](https://huggingface.co/datasets/wikitext) and then adapted to the [IMDb](https://huggingface.co/datasets/imdb)\ndataset. The encoder of the trained language model is then used for [sentiment classification](#sentiment-classification).\n\nThe tokenizer is a customized BERT tokenizer (`tokenizers/bert-base-uncased-10k-bookcorpus-ext`), trained on [BookCorpus](https://huggingface.co/datasets/bookcorpus)\nwith a vocabulary size of 10,000. You can also use any other 🤗 [fast tokenizer](https://huggingface.co/docs/transformers/fast_tokenizers)\nfrom the 🤗 Hub with the `--data.tokenizer` option (see [Tokenizers](docs/tokenizer.md) for details).\n\nThe training script is [mlm.py](perceiver/scripts/text/mlm.py). It implements the command line interface and defines\ntraining defaults (see also [trainer.yaml](perceiver/scripts/trainer.yaml) for further defaults). Pretraining on\nWikiText-103 can be started with:\n\n```shell\npython -m perceiver.scripts.text.mlm fit \\\n  --model.num_latents=128 \\\n  --model.num_latent_channels=128 \\\n  --model.encoder.num_input_channels=128 \\\n  --model.encoder.num_cross_attention_layers=3 \\\n  --model.encoder.num_self_attention_layers_per_block=6 \\\n  --model.encoder.num_self_attention_blocks=3 \\\n  --model.encoder.first_self_attention_block_shared=false \\\n  --model.encoder.dropout=0.1 \\\n  --model.decoder.dropout=0.1 \\\n  --data=WikiTextDataModule \\\n  --data.tokenizer=tokenizers/bert-base-uncased-10k-bookcorpus-ext \\\n  --data.max_seq_len=512 \\\n  --data.batch_size=64 \\\n  --optimizer=AdamW \\\n  --optimizer.lr=1e-3 \\\n  --optimizer.weight_decay=0.01 \\\n  --lr_scheduler.warmup_steps=5000 \\\n  --trainer.accelerator=gpu \\\n  --trainer.devices=-1 \\\n  --trainer.max_steps=50000 \\\n  --trainer.check_val_every_n_epoch=5 \\\n  --trainer.logger=TensorBoardLogger \\\n  --trainer.logger.save_dir=logs \\\n  --trainer.logger.name=mlm\n```\n\n| Model parameters                                                                           | Validation loss                           | Mask prediction samples                     |\n|--------------------------------------------------------------------------------------------|-------------------------------------------|---------------------------------------------|\n| <pre>Total params:      2.9M<br/>Frozen params:       0M<br/>Trainable params:  2.9M</pre> | ![val-loss-1](docs/images/val-loss-1.png) | ![mask-pred-1](docs/images/mask-pred-1.png) |\n\nStarting from the best pretraining checkpoint, the language model is then adapted to the IMDb dataset for further 15,000\nsteps. If you ran pretraining yourself, you\'ll need to modify the `--model.ckpt` value accordingly, otherwise the checkpoint\nfrom the downloaded archive is used.\n\n```shell\npython -m perceiver.scripts.text.mlm fit \\\n  --model.ckpt="logs/mlm/version_0/checkpoints/epoch=044-val_loss=3.917.ckpt" \\\n  --model.num_latents=128 \\\n  --model.num_latent_channels=128 \\\n  --model.encoder.num_input_channels=128 \\\n  --model.encoder.num_cross_attention_layers=3 \\\n  --model.encoder.num_self_attention_layers_per_block=6 \\\n  --model.encoder.num_self_attention_blocks=3 \\\n  --model.encoder.first_self_attention_block_shared=false \\\n  --model.encoder.dropout=0.1 \\\n  --model.decoder.dropout=0.1 \\\n  --data=ImdbDataModule \\\n  --data.tokenizer=tokenizers/bert-base-uncased-10k-bookcorpus-ext \\\n  --data.max_seq_len=512 \\\n  --data.batch_size=64 \\\n  --optimizer=AdamW \\\n  --optimizer.lr=1e-3 \\\n  --optimizer.weight_decay=0.01 \\\n  --lr_scheduler.warmup_steps=1000 \\\n  --trainer.accelerator=gpu \\\n  --trainer.devices=-1 \\\n  --trainer.max_steps=15000 \\\n  --trainer.check_val_every_n_epoch=3 \\\n  --trainer.logger=TensorBoardLogger \\\n  --trainer.logger.save_dir=logs \\\n  --trainer.logger.name=mlm\n```\n\n| Model parameters                                                                           | Validation loss                           | Mask prediction samples                     |\n|--------------------------------------------------------------------------------------------|-------------------------------------------|---------------------------------------------|\n| <pre>Total params:      2.9M<br/>Frozen params:       0M<br/>Trainable params:  2.9M</pre> | ![val-loss-2](docs/images/val-loss-2.png) | ![mask-pred-2](docs/images/mask-pred-2.png) |\n\nAfter adaption to IMDb, mask prediction samples are obviously more related to movie reviews compared to pretraining on\nWikiText-103 only. Prediction samples are screenshots from Tensorboard logs.\n\n### Sentiment classification\n\nThis section trains a Perceiver IO text classifier on IMDb reviews. The encoder is initialized with weights from\n[masked language modeling](#masked-language-modeling) (`--model.mlm_ckpt` option), the decoder is a randomly initialized\nclassification decoder. In a first step, only the decoder is trained and the encoder is frozen. The training script is\n[classifier.py](perceiver/scripts/text/classifier.py).\n\n```shell\npython -m perceiver.scripts.text.classifier fit \\\n  --model.mlm_ckpt="logs/mlm/version_1/checkpoints/epoch=113-val_loss=3.904.ckpt" \\\n  --model.num_latents=128 \\\n  --model.num_latent_channels=128 \\\n  --model.encoder.num_input_channels=128 \\\n  --model.encoder.num_cross_attention_layers=3 \\\n  --model.encoder.num_self_attention_layers_per_block=6 \\\n  --model.encoder.num_self_attention_blocks=3 \\\n  --model.encoder.first_self_attention_block_shared=false \\\n  --model.encoder.dropout=0.1 \\\n  --model.encoder.freeze=true \\\n  --model.decoder.num_output_query_channels=128 \\\n  --model.decoder.dropout=0.1 \\\n  --data=ImdbDataModule \\\n  --data.tokenizer=tokenizers/bert-base-uncased-10k-bookcorpus-ext \\\n  --data.target_task=clf \\\n  --data.max_seq_len=512 \\\n  --data.batch_size=256 \\\n  --optimizer=AdamW \\\n  --optimizer.lr=1e-4 \\\n  --optimizer.weight_decay=0.01 \\\n  --trainer.accelerator=gpu \\\n  --trainer.devices=-1 \\\n  --trainer.max_epochs=30 \\\n  --trainer.log_every_n_steps=10 \\\n  --trainer.logger=TensorBoardLogger \\\n  --trainer.logger.save_dir=logs \\\n  --trainer.logger.name=clf\n```\n\n| Model parameters                                                                           | Validation accuracy                     |\n|--------------------------------------------------------------------------------------------|-----------------------------------------|\n| <pre>Total params:      2.9M<br/>Frozen params:     2.8M<br/>Trainable params:  100K</pre> | ![val-acc-1](docs/images/val-acc-1.png) |\n\nThe small classification decoder (100K parameters) can be trained to a validation accuracy of 88% when using an\nencoder that has been adapted to the IMDb dataset (red line). When using an encoder that has been pretrained on\nWikiText-103 only, the validation accuracy saturates at 78% (pink line). Unfreezing the encoder and fine-tuning it\njointly with the classification decoder further improves validation accuracy:\n\n```shell\npython -m perceiver.scripts.text.classifier fit \\\n  --model.clf_ckpt="logs/clf/version_0/checkpoints/epoch=028-val_loss=0.301.ckpt" \\\n  --model.num_latents=128 \\\n  --model.num_latent_channels=128 \\\n  --model.encoder.num_input_channels=128 \\\n  --model.encoder.num_cross_attention_layers=3 \\\n  --model.encoder.num_self_attention_layers_per_block=6 \\\n  --model.encoder.num_self_attention_blocks=3 \\\n  --model.encoder.first_self_attention_block_shared=false \\\n  --model.encoder.dropout=0.1 \\\n  --model.decoder.num_output_query_channels=128 \\\n  --model.decoder.dropout=0.1 \\\n  --data=ImdbDataModule \\\n  --data.tokenizer=tokenizers/bert-base-uncased-10k-bookcorpus-ext \\\n  --data.target_task=clf \\\n  --data.max_seq_len=512 \\\n  --data.batch_size=256 \\\n  --optimizer=AdamW \\\n  --optimizer.lr=1e-5 \\\n  --optimizer.weight_decay=0.01 \\\n  --trainer.accelerator=gpu \\\n  --trainer.devices=-1 \\\n  --trainer.max_epochs=30 \\\n  --trainer.log_every_n_steps=10 \\\n  --trainer.logger=TensorBoardLogger \\\n  --trainer.logger.save_dir=logs \\\n  --trainer.logger.name=clf\n```\n\n| Model parameters                                                                           | Validation accuracy                     |\n|--------------------------------------------------------------------------------------------|-----------------------------------------|\n| <pre>Total params:      2.9M<br/>Frozen params:       0M<br/>Trainable params:  2.9M</pre> | ![val-acc-2](docs/images/val-acc-2.png) |\n\n### Image classification\n\nThis section trains a tiny Perceiver IO image classifier (805K parameters) on MNIST digits. The model attends to each\npixel in input images and does not use convolutional layers. In contrast to other examples only a single cross-attention\nlayer is used. The training script is [classifier.py](perceiver/scripts/image/classifier.py).\n\n```shell\npython -m perceiver.scripts.image.classifier fit \\\n  --model.num_latents=32 \\\n  --model.num_latent_channels=128 \\\n  --model.encoder.num_frequency_bands=32 \\\n  --model.encoder.num_cross_attention_layers=1 \\\n  --model.encoder.num_self_attention_layers_per_block=3 \\\n  --model.encoder.num_self_attention_blocks=3 \\\n  --model.encoder.first_self_attention_block_shared=false \\\n  --model.encoder.dropout=0.0 \\\n  --model.encoder.init_scale=0.1 \\\n  --model.decoder.num_output_query_channels=128 \\\n  --model.decoder.dropout=0.0 \\\n  --model.decoder.init_scale=0.1 \\\n  --data=MNISTDataModule \\\n  --data.batch_size=128 \\\n  --optimizer=AdamW \\\n  --optimizer.lr=1e-3 \\\n  --optimizer.weight_decay=0.01 \\\n  --trainer.accelerator=gpu \\\n  --trainer.devices=-1 \\\n  --trainer.max_epochs=20 \\\n  --trainer.logger=TensorBoardLogger \\\n  --trainer.logger.save_dir=logs \\\n  --trainer.logger.name=exp\n```\n\n| Model parameters                                                                           | Validation accuracy                     |\n|--------------------------------------------------------------------------------------------|-----------------------------------------|\n| <pre>Total params:      805K<br/>Frozen params:       0K<br/>Trainable params:  805K</pre> | ![val-acc-3](docs/images/val-acc-3.png) |\n\n## Inference examples\n\n- Sentiment classification  \n  [![Open In Colab](https://colab.research.google.com/assets/colab-badge.svg)](https://colab.research.google.com/github/krasserm/perceiver-io/blob/main/notebooks/txt-clf.ipynb)\n- Image classification  \n  [![Open In Colab](https://colab.research.google.com/assets/colab-badge.svg)](https://colab.research.google.com/github/krasserm/perceiver-io/blob/main/notebooks/img-clf.ipynb)\n',
    'author': 'Martin Krasser',
    'author_email': 'krasserm@googlemail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/krasserm/perceiver-io',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
